#include <stdio.h>
#include <sys/iomap.h>
#include <fcntl.h>

#include "lib.c.h1"
#include "lib.c.h2"

/***************************** global variables ****************************/

/* control and frame buffer pointers and descriptors */
char cntrl_name[80] = "/dev/catseye";
char fb_name[80] = "/dev/cat_fb";
int fdes_cntrl, fdes_frm_bfr;
unsigned char *cntrl = (unsigned char *) 0x900000;
unsigned char *frm_bfr = (unsigned char *) 0xa00000; 


/* lib state variables */
int cats_await_retrace;
int cats_max_x, cats_max_y;
int cats_nplanes;
int cats_board_init = FALSE;


/* overlay state */
int overlay_select;
int overlay_enable;


/* hardware control */
reg cats_rug_enable;
reg pri_rug_enable;
reg ovl_rug_enable;

reg cats_linetype_enable;
reg pri_linetype_enable;
reg ovl_linetype_enable;

reg pri_control;
reg ovl_control;

reg pri_ltype;
reg ovl_ltype;

reg pri_ltp;
reg ovl_ltp;

int ovl_opaque_mask;


/* drawing attributes */
int cats_cx, cats_cy;
int pri_cx, pri_cy;
int ovl_cx, ovl_cy;

int cats_clipping_indicator;
int pri_clipping_indicator;
int ovl_clipping_indicator;

int cats_clip_xmin, cats_clip_ymin, cats_clip_xmax, cats_clip_ymax;
int pri_clip_xmin, pri_clip_ymin, pri_clip_xmax, pri_clip_ymax;
int ovl_clip_xmin, ovl_clip_ymin, ovl_clip_xmax, ovl_clip_ymax;

int cats_line_color;
int pri_line_color;
int ovl_line_color;

int cats_fill_color;
int pri_fill_color;
int ovl_fill_color;

int cats_write_mask;
int pri_write_mask;
int ovl_write_mask;

int cats_disp_mask;
int pri_disp_mask;
int ovl_disp_mask;

int cats_draw_mode;
int pri_draw_mode;
int ovl_draw_mode;

int fill_three_op;
int blit_three_op;
int color0_three_op;
int color1_three_op;

int cats_poly_filled;
int pri_poly_filled;
int ovl_poly_filled;

int cats_poly_edged;
int pri_poly_edged;
int ovl_poly_edged;

int cats_enable_linetype;
int pri_enable_linetype;
int ovl_enable_linetype;

int blit_type;
reg blit_source_mask;


/* double buffer variables */
int db_mode;
int db_disp_index;
int db_flag;
reg db_disp_mask[5][2] = {
    0x07, 0x38,            /* 6 plane primary */
    0x03, 0x0c,            /* 4 plane primary */
    0x01, 0x02,            /* overlay for LCC */
    0xf0, 0x0f,            /* 8 plane primary */
    0x01, 0x02 };          /* overlay for HRC */
reg db_write_mask[5][2] = {
    0x38, 0x07,            /* 6 plane primary */
    0x0c, 0x03,            /* 4 plane primary */
    0x20, 0x10,            /* overlay for LCC */
    0x0f, 0xf0,            /* 8 plane primary */
    0x02, 0x01 };          /* overlay for HRC */
int oldrgb[256][3];




DefineDeviceFiles(cname, fname)
char *cname, *fname;
{
    int cnt;
    char *cptr;

    /* copy new control space device file name */
    cnt = 0;
    cptr = cname;
    while ((cnt < 80) && ((cntrl_name[cnt++] = *cptr++) != '\0'));

    /* copy new frame buffer device file name */
    cnt = 0;
    cptr = fname;
    while ((cnt < 80) && ((fb_name[cnt++] = *cptr++) != '\0'));
}



int Init(clr,init_cmap,overlay)
int clr, init_cmap, overlay;
{

    register int i;
    register int mask;


    /* if the library has already been initialized, then return */
    if (cats_board_init)
	return(-1);

    /* attempt to establish connection with board */
    /* use IOMAP because present kernels cannot GCMAP DIO II space */
    if ((fdes_cntrl = open(cntrl_name,O_RDWR)) < 0) {
        fprintf(stderr,"Cannot open catseye control space\n");
        return(-1);
    }
    ioctl(fdes_cntrl, IOMAPMAP,&cntrl);
    if ((fdes_frm_bfr = open(fb_name,O_RDWR)) < 0) {
        fprintf(stderr,"Cannot open catseye frame buffer\n");
        return(-1);
    }
    ioctl(fdes_frm_bfr,IOMAPMAP,&frm_bfr);


    /* determine how many planes this board has */
    cats_nplanes = *((unsigned char *)(cntrl + 0x005b));
    if (cats_nplanes == 6) {
         cats_max_x = 1023;
         cats_max_y = 767;
    }
    else {
         cats_max_x = 1279;
         cats_max_y = 1023;
    }

    /* set up for use of overlay planes if selected */
    /* except in Monochrome display case where there are no overlay planes */
    overlay_select = ((overlay) && (cats_nplanes != 1));
    overlay_enable = FALSE;

    /* initialize global variables */
    if (overlay_select) {
	ovl_cx = ovl_cy = 0;
	ovl_clipping_indicator = TRUE;
	ovl_clip_xmin = ovl_clip_ymin = 0;
	ovl_clip_xmax = cats_max_x;
	ovl_clip_ymax = cats_max_y;
	ovl_line_color = 1;
	ovl_fill_color = 1;
	ovl_write_mask = (cats_nplanes == 8) ? 0x03 : 0x30;
	ovl_disp_mask = 0x03;
	ovl_draw_mode = 3;
	ovl_poly_filled = INT_HOLLOW;
	ovl_poly_edged = TRUE;
	ovl_linetype_enable = 0x0000;
	ovl_enable_linetype = FALSE;
	ovl_rug_enable = 0x0010;
	ovl_ltype = 0xffff;
	ovl_ltp = 0x0ff0;
    }

    cats_await_retrace = FALSE;

    cats_cx = cats_cy = pri_cx = pri_cy = 0;
    cats_clipping_indicator = pri_clipping_indicator = TRUE;
    cats_clip_xmin = cats_clip_ymin = pri_clip_xmin = pri_clip_ymin = 0;
    cats_clip_xmax = pri_clip_xmax = cats_max_x;
    cats_clip_ymax = pri_clip_ymax = cats_max_y;
    cats_line_color = pri_line_color = 1;
    cats_fill_color = pri_fill_color = 1;
    if (cats_nplanes == 8) {
        cats_write_mask  = pri_write_mask = 0xff;
        cats_disp_mask = pri_disp_mask = 0xff;
    }
    else if (cats_nplanes == 6) {
	if (overlay_select) {
            cats_write_mask  = pri_write_mask = 0x0f;
            cats_disp_mask = pri_disp_mask = 0x0f;
        }
	else {
            cats_write_mask  = pri_write_mask = 0x3f;
            cats_disp_mask = pri_disp_mask = 0x3f;
        }
    }
    else {
        cats_write_mask  = pri_write_mask = 0x01;
        cats_disp_mask = pri_disp_mask = 0x01;
    }
    cats_draw_mode = pri_draw_mode = 3;
    cats_poly_filled = pri_poly_filled = INT_HOLLOW;
    cats_poly_edged = pri_poly_edged = TRUE;
    cats_enable_linetype = pri_enable_linetype = FALSE;
    cats_linetype_enable = pri_linetype_enable = 0x0000;
    cats_rug_enable  = pri_rug_enable = 0x0010;
    pri_ltype = 0xffff;
    pri_ltp = 0x0ff0;

    generate_three_ops(3);    /* Source */

    blit_type = BLIT_WITHIN;
    blit_source_mask = 0x00;

    db_mode = DB_OFF;
    db_disp_index = 0;
    db_flag = 0;

    ovl_opaque_mask = 0x00;



    /* initialize the color map if init_cmap */
    if (init_cmap) {

	/* define primary color map */
	generatecmap(cats_nplanes,oldrgb);
	if (cats_nplanes == 8) 
            DefineColorTable(0,256,oldrgb);
        else 
            DefineColorTable(0,64,oldrgb);

	/* define overlay color map */
        if (overlay_select) {
	    overlay_enable = TRUE;
	    DefineColorTable(0,1,oldrgb);   /* 0 entry is transparent */
	    DefineColorTable(1,3,oldrgb);   /* 1 - black, 2 - white, 3 - red */
	    overlay_enable = FALSE;
        }
    }


    /* Initialize board */
    if (cats_nplanes == 6)
	*((reg *)(cntrl + STATUS)) = 0x0000;  /* Disable scratch plane */


    /* Initialize IRIS */
    *((reg *) (cntrl + BLINK)) = 0x00;      /* No blinking planes */
    if (overlay_select)                      /* Display overlays if selected */
         *((reg *) (cntrl + OVLCNTRL)) = 0x03 | ovl_opaque_mask;
    else
         *((reg *) (cntrl + OVLCNTRL)) = 0x00 | ovl_opaque_mask;
    *((reg *) (cntrl + RSEL)) = 0x00; 
    *((reg *) (cntrl + PMASK)) = cats_disp_mask;


    /* Initialize first bank of  Barc(s) */
    *(cntrl + VB) = 0x01;   /* enable vector writing */
    *(cntrl + TCNTRL) = 0x00;   /* no three operand rrs */
    *(cntrl + ACNTRL) = 0x00;   /* spu accesses byte per pixel */
    *(cntrl + PNCNTRL) = 0x00;   /* all blts internal, plane 0 talks */
    *(cntrl + FBEN0) = cats_write_mask;
    *(cntrl + TCREN0) = 0x01;   /* read rr's from plane 0 */
    if (cats_nplanes == 1)
        *(cntrl + TCWEN0) = 0x01;  /* no changes to scratch */
    else
        *(cntrl + TCWEN0) = 0xff;
    *(cntrl + PRR0) = cats_draw_mode;  
    *((reg *)(cntrl + WRR0)) = cats_draw_mode;
    *(cntrl + TRR0) = fill_three_op;
    *(cntrl + COLOR0) = cats_line_color;

    /* Initialize second bank of BARCs if they exist */
    if (cats_nplanes == 8) {
        *(cntrl + COLOR1) = 0x00;
        *(cntrl + FBEN1) = 0x00;
        *(cntrl + TCREN1) = 0x01;
        *(cntrl + TCWEN1) = 0x07;
	*(cntrl + PRR1) = 0x00;
	*((reg *)(cntrl + WRR1)) = 0x0000;
	*(cntrl + TRR1) = 0x00;
        *(cntrl + TCWEN1) = 0x00;
    }


    /* initialize RUG */
    *((reg *)(cntrl + COMMAND)) = cats_rug_enable;
					    /* RUG BUG FIX */
    *((reg *)(cntrl + CONTROL)) |= 0x0083;  /* enable clipping */
					    /* set 2 lsb with clip bit */
    pri_control = ovl_control = *((reg *)(cntrl + CONTROL));
    *((reg *)(cntrl + CXMIN)) = cats_clip_xmin;
    *((reg *)(cntrl + CXMAX)) = cats_clip_xmax;
    *((reg *)(cntrl + CYMIN)) = cats_clip_ymin;
    *((reg *)(cntrl + CYMAX)) = cats_clip_ymax;
    *((reg *)(cntrl + LTYPE)) = pri_ltype;
    *((reg *)(cntrl + LTP)) = pri_ltp;

    /* clear screan if clr - clear overlay planes if selected too */
    /* NOTE: scratch plane is not cleared */
    if (clr) {
        *(cntrl + VB) = 0x00;  /* blits for barc */
        *((reg *)(cntrl + WRR0)) = 0x0000;  /* WRR = clear */
	if (overlay_select) {
	    if (cats_nplanes == 8) 
		*(cntrl + FBEN1) = 0x03;
	    else
		*(cntrl + FBEN0) = 0x3f;
	}
        *((reg *)(cntrl + COMMAND)) = 0x0090;  /* blits for rug */
        *((reg *)(cntrl + XSRC)) = 0x0000;  /* Source x = 0 */
        *((reg *)(cntrl + YSRC)) = 0x0000;  /* Source y = 0 */
        *((reg *)(cntrl + XDST)) = 0x0000;  /* Destination x = 0 */
        *((reg *)(cntrl + YDST)) = 0x0000;  /* Destination x = 0 */
        *((reg *)(cntrl + FX)) = cats_max_x + 1;  /* Width */
        *((reg *)(cntrl + TFY)) = cats_max_y + 1;  /* Height */
        WAIT_FOR_RUG;
        *((reg *)(cntrl + COMMAND)) = cats_rug_enable;  /* reset rug */
        *(cntrl + VB) = 0x01;  /* reset barc */
        *((reg *)(cntrl + WRR0)) = cats_draw_mode;
	if (overlay_select) {
	    if (cats_nplanes == 8) 
		*(cntrl + FBEN1) = 0x00;
	    else
		*(cntrl + FBEN0) = cats_write_mask;
	}
    }

    cats_board_init = TRUE;
    return(0);

}



Close()
{
    if (!cats_board_init)
	return;

    else {
        ioctl(fdes_cntrl,IOMAPUNMAP,&cntrl);
        close(fdes_cntrl);

        ioctl(fdes_frm_bfr,IOMAPUNMAP,&frm_bfr);
        close(fdes_frm_bfr);
    }
}




generatecmap(n,cmap)
int n, cmap[];
{
    static int primary[8][3] = {
	0, 0, 0, 255, 255, 255, 255, 0, 0, 255, 255, 0,
	0, 255, 0, 0, 255, 255, 0, 0, 255, 255, 0, 255 };
    int i;
    double r, g, b;

    /* set up first 8 entries - same for all display cards */
    for (i=0; i<8; i++) {
	cmap[3*i] = primary[i][0];
	cmap[3*i+1] = primary[i][1];
	cmap[3*i+2] = primary[i][2];
    }

    /* now do separate things for 6 and 8 plane systems */
    if (n == 6) {
        /* generate gray scale for entires 8-15 */
	i = 8;
	for (g=0.00; g<1.0; g += 1.00/8.00) {
	    cmap[3*i] = cmap[3*i+1] = cmap[3*i+2] = (int) (g * 255.0);
	    i++;
        }

	/* generate remaining 48 entries */
	i = 16;
	for (b=1.00; b>=0.0; b -= 1.00/2.00)
	    for (g=1.00; g>=0.0; g -= 1.00/3.00)
		for (r=1.00; r>=0.0; r -= 1.00/3.00) {
		    cmap[3*i] = (int) (r * 255.0);
		    cmap[3*i+1] = (int) (g * 255.0);
		    cmap[3*i+2] = (int) (b * 255.0);
		    i++;
                 }
    }
    else if (n == 8) {
	/* first do 2 extra colors between initial 8 and next 245 */
        cmap[24] = cmap[25] = cmap[26] = 85;
        cmap[27] = cmap[28] = cmap[29] = 171;

	/* then generate next 245 colors */
	i = 10;
	for (b=1.00; b>=0.0; b -= 1.00/4.00)
	    for (g=1.00; g>=0.0; g -= 1.00/6.00)
		for (r=1.00; r>=0.0; r -= 1.00/6.00) {
		    cmap[3*i] = (int) (r * 255);
		    cmap[3*i+1] = (int) (g * 255);
		    cmap[3*i+2] = (int) (b * 255);
		    i++;
                }
        /* finally write the last entry */
	cmap[765] = cmap[766] = cmap[767] = 255;

    }
}



EnableOverlay(flag)
int flag;
{
    
    if ((!overlay_select) || (overlay_enable == flag))
	return;

    else {

	/* Save necessary variables */
	if (overlay_enable) {
	    ovl_cx = cats_cx;
	    ovl_cy = cats_cy;
	    ovl_control = *((reg *)(cntrl + CONTROL));
	}
	else {
	    pri_cx = cats_cx;
	    pri_cy = cats_cy;
	    pri_control = *((reg *)(cntrl + CONTROL));
	}

	/* switch overlay_enable */
	overlay_enable = flag;

	/* Assign proper variables to current catseye variables */
	cats_rug_enable = (flag) ? ovl_rug_enable : pri_rug_enable;
	cats_linetype_enable = (flag) ? ovl_linetype_enable : pri_linetype_enable;
	cats_cx = (flag) ? ovl_cx : pri_cx;
	cats_cy = (flag) ? ovl_cy : pri_cy;
	cats_clipping_indicator = (flag) ? ovl_clipping_indicator : pri_clipping_indicator;
	cats_clip_xmin = (flag) ? ovl_clip_xmin : pri_clip_xmin;
	cats_clip_ymin = (flag) ? ovl_clip_ymin : pri_clip_ymin;
	cats_clip_xmax = (flag) ? ovl_clip_xmax : pri_clip_xmax;
	cats_clip_ymax = (flag) ? ovl_clip_ymax : pri_clip_ymax;
	cats_line_color = (flag) ? ovl_line_color : pri_line_color;
	cats_fill_color = (flag) ? ovl_fill_color : pri_fill_color;
	cats_write_mask = (flag) ? ovl_write_mask : pri_write_mask;
	cats_disp_mask = (flag) ? ovl_disp_mask : pri_disp_mask;
	cats_draw_mode = (flag) ? ovl_draw_mode : pri_draw_mode;
	cats_poly_filled = (flag) ? ovl_poly_filled : pri_poly_filled;
	cats_poly_edged = (flag) ? ovl_poly_edged : pri_poly_edged;
	cats_enable_linetype = (flag) ? ovl_enable_linetype : pri_enable_linetype;

	/* Set up proper drawing mode and frame buffer enables */
	generate_three_ops(cats_draw_mode);
	WAIT_FOR_RUG;
	if (cats_nplanes == 8) {
	    if (overlay_enable) {
		*(cntrl + FBEN0) = 0x00;
		if (db_mode == DB_OVL)
		    *(cntrl + FBEN1) = db_write_mask[db_disp_index][db_flag];
		else
		    *(cntrl + FBEN1) = cats_write_mask;

		*(cntrl + COLOR1) = cats_line_color;

		*(cntrl + TCWEN0) = 0xff;
		*(cntrl + PRR0) = 0x00;
		*((reg *)(cntrl + WRR0)) = 0x0000;
		*(cntrl + TRR0) = 0x00;
		*(cntrl + TCWEN0) = 0x00;

		*(cntrl + TCWEN1) = 0x03;
		*(cntrl + PRR1) = cats_draw_mode;
		*((reg *)(cntrl + WRR1)) = cats_draw_mode;
		*(cntrl + TRR1) = fill_three_op;
            }
	    else {
		if (db_mode == DB_PRI)
		    *(cntrl + FBEN0) = db_write_mask[db_disp_index][db_flag];
		else
		    *(cntrl + FBEN0) = cats_write_mask;
		*(cntrl + FBEN1) = 0x00;

		*(cntrl + COLOR0) = cats_line_color;

		*(cntrl + TCWEN0) = 0xff;
		*(cntrl + PRR0) = cats_draw_mode;
		*((reg *)(cntrl + WRR0)) = cats_draw_mode;
		*(cntrl + TRR0) = fill_three_op;

		*(cntrl + TCWEN1) = 0x07;
		*(cntrl + PRR1) = 0x00;
		*((reg *)(cntrl + WRR1)) = 0x0000;
		*(cntrl + TRR1) = 0x00;
		*(cntrl + TCWEN1) = 0x00;
	    }
        }
	else {
	    if (db_mode != DB_OFF)
	        *(cntrl + FBEN0) = db_write_mask[db_disp_index][db_flag];
	    else
	        *(cntrl + FBEN0) = cats_write_mask;
	    *(cntrl + PRR0) = cats_draw_mode;
	    *((reg *)(cntrl + WRR0)) = cats_draw_mode;
	    *(cntrl + TRR0) = fill_three_op;
	    if (overlay_enable)
		*(cntrl + COLOR0) = cats_line_color << 4;
            else
		*(cntrl + COLOR0) = cats_line_color;
        }

	/* Reset state to RUG */
	*((reg *)(cntrl + CONTROL)) = (flag) ? ovl_control : pri_control;
	*((reg *)(cntrl + LTYPE)) = (flag) ? ovl_ltype : pri_ltype;
	*((reg *)(cntrl + LTP)) = (flag) ? ovl_ltp : pri_ltp;

	/* Restore clipping limits and enable */
	*((reg *)(cntrl + CXMIN)) = cats_clip_xmin;
	*((reg *)(cntrl + CXMAX)) = cats_clip_xmax;
	*((reg *)(cntrl + CYMIN)) = cats_clip_ymin;
	*((reg *)(cntrl + CYMAX)) = cats_clip_ymax;
	set_clipping(cats_clipping_indicator);

    }
}




InquireSize(x,y)
int *x, *y;
{
    *x = cats_max_x;
    *y = cats_max_y;
}


InquirePlanes(n)
int *n;
{
    if (cats_nplanes == 6) {
	if (overlay_select)
            *n = 4;
	else
            *n = 6;
    }
    else
        *n = cats_nplanes;
}


InquirePointers(control, frame_buffer)
unsigned char **control, **frame_buffer;
{
    *control = cntrl;
    *frame_buffer = frm_bfr;
}


InteriorStyle(edge_mode, fill_type)
int edge_mode, fill_type;
{
    if (overlay_enable) {
	ovl_poly_edged = edge_mode;
	ovl_poly_filled = fill_type;
    }
    else {
	pri_poly_edged = edge_mode;
	pri_poly_filled = fill_type;
    }

    cats_poly_edged = edge_mode;
    cats_poly_filled = fill_type;
}


WriteEnable(mask)
int mask;
{
    reg new_mask;

    WAIT_FOR_RUG;

    /* overlay planes not selected */
    if (!overlay_select) {
	switch (cats_nplanes) {
	    case 8: new_mask = mask & 0xff;
		    break;
	    case 6: new_mask = mask & 0x3f;
		    break;
	    case 1:
	    default: new_mask = mask & 0x01;
		    break;
	}
        *(cntrl + FBEN0) = new_mask;
	pri_write_mask = new_mask;
    }

    /* overlay planes selected but not enabled */
    else if (!overlay_enable) {
	new_mask = mask & ((cats_nplanes == 8) ? 0xff : 0x0f);
        *(cntrl + FBEN0) = new_mask;
	pri_write_mask = new_mask;
    }

    /* overlay planes selected and enabled */
    else {
	if (cats_nplanes == 8) {
	    new_mask = mask & 0x03;
            *(cntrl + FBEN1) = new_mask;
	}
        else {
	    new_mask = (mask & 0x03) << 4;
            *(cntrl + FBEN0) = new_mask;
	}
	ovl_write_mask = new_mask;
    }

    cats_write_mask = new_mask;

}
    



/************************** IRIS ROUTINES ************************************/

BlinkPlanes(mask)
reg mask;
{
    register unsigned char rval;

    if (overlay_enable) {
	rval = *((reg *) (cntrl + OVLCNTRL));
	*((reg *) (cntrl + OVLCNTRL)) = (rval & 0x13) | ((mask << 2) & 0xc);
    }
    else {
        switch (cats_nplanes) {
	    case 8: *(cntrl + BLINK) = mask;
		    break;
            case 6: if (overlay_select)
			*(cntrl + BLINK) = mask & 0x0f;
                    else
			*(cntrl + BLINK) = mask & 0x3f;
                    break;
	    case 1: *(cntrl + BLINK) = mask & 0x01;
	    default: break;
       }
    }
}


EnableTransparency(flag)
int flag;
{
    register unsigned char rval;

    if (overlay_select) {
        if (flag)
	    ovl_opaque_mask = 0x00;
        else
	    ovl_opaque_mask = 0x10;

        rval = *((reg *) (cntrl + OVLCNTRL));
        *((reg *) (cntrl + OVLCNTRL)) = (rval & 0xf) | ovl_opaque_mask;
    }
}


int DefineColorTable(start,num,cmap)
int start, num;
int cmap[];
{
    register int i;

    if (overlay_enable) 
	i = 4;
    else {
        i = 256;
        *((reg *) (cntrl + RSEL)) = 0x0000;
    }

    if ((start < 0) || (start + num > i)) return(-1);

    CHECK_AND_WAIT_FOR_RETRACE;
    for (i=0; i<num; i++) {
        WAIT_FOR_IRIS;
	if (!overlay_enable)
            *((reg *) (cntrl + RAMADDR)) = start + i;
	else
            *((reg *) (cntrl + RSEL)) = (((start + i) << 1) & 0x6) | 0x8;
        *((reg *) (cntrl + RED)) = cmap[3*i];
        *((reg *) (cntrl + GREEN)) = cmap[3*i + 1];
        *((reg *) (cntrl + BLUE)) = cmap[3*i + 2];
        *((reg *) (cntrl + WTRIG)) = 0xff;
    }

    WAIT_FOR_IRIS;
    if (overlay_select)
        *((reg *) (cntrl + RSEL)) = 0x0000;

    return(0);
}



int ReadColorTable(start,num,cmap)
int start, num;
int cmap[];
{
    register int i;

    if (overlay_enable) 
	i = 4;
    else {
        i = 256;
        *((reg *) (cntrl + RSEL)) = 0x0000;
    }

    if ((start < 0) || (start + num > i)) return(-1);

    for (i=0; i<num; i++) {
        WAIT_FOR_IRIS;
	if (!overlay_enable)
            *((reg *) (cntrl + RAMADDR)) = start + i;
	else
            *((reg *) (cntrl + RSEL)) = (((start + i) << 1) & 0x6) | 0x8;
        *((reg *) (cntrl + RTRIG)) = 0x00ff;
        WAIT_FOR_IRIS;
        cmap[3*i] = *((reg *) (cntrl + RED)) & 0xff;
        cmap[3*i+1] = *((reg *) (cntrl + GREEN)) & 0xff;
        cmap[3*i+2] = *((reg *) (cntrl + BLUE)) & 0xff;
    }

    WAIT_FOR_IRIS;
    if (overlay_select)
	*((reg *) (cntrl + RSEL)) = 0x0000;

    return(0);
}


AwaitVerticalRetrace()
{
    cats_await_retrace = TRUE;
}


DisplayEnable(mask)
int mask;
{
    register unsigned char rval;

    if (overlay_enable) {
	cats_disp_mask = ovl_disp_mask = mask & 0x3;
	rval = *((reg *) (cntrl + OVLCNTRL));
	*((reg *) (cntrl + OVLCNTRL)) = (rval & 0x1c) | cats_disp_mask;
    }
    else {
	switch (cats_nplanes) {
	    case 8: pri_disp_mask = mask & 0xff;
		    break;
	    case 6: if (overlay_select)
	                pri_disp_mask = mask & 0x0f;
		    else
	                pri_disp_mask = mask & 0x3f;
		    break;
            case 1:
	    default: pri_disp_mask = mask & 0x01;
        }
        cats_disp_mask = pri_disp_mask;
	*((reg *) (cntrl + PMASK)) = cats_disp_mask;
    }
}


/************************* BARC ROUTINES *************************************/

generate_three_ops(mode)
int mode;
{
    /* blit_rule is used for pattern filled polygons */
    static unsigned char blit_rule[16] = {
        0x22, 0xa2, 0x62, 0xe2, 0x2a, 0xaa, 0x6a, 0xea,
        0x26, 0xa6, 0x66, 0xe6, 0x2e, 0xae, 0x6e, 0xee };

    /* fill_rule is used for pattern filled circles and rectangles */
    static unsigned char fill_rule[16] = {
        0x00, 0xa0, 0x50, 0xf0, 0x0a, 0xaa, 0x5a,0xfa,
        0x05, 0xa5, 0x55, 0xf5, 0x0f, 0xaa, 0x5f, 0xff };

    /* color0_rule is used for solid color filled polygons */
    /* color0_rule is used for destination planes that should be 0 */
    static unsigned char color0_rule[16] = {
        0x22, 0x22, 0x22, 0x22, 0xaa, 0xaa, 0xaa, 0xaa,
        0x66, 0x66, 0x66, 0x66, 0xee, 0xee, 0xee, 0xee };

    /* color1_rule is used for solid color filled polygons */
    /* color1_rule is used for destination planes that should be 1 */
    static unsigned char color1_rule[16] = {
        0x22, 0xaa, 0x66, 0xee, 0x22, 0xaa, 0x66, 0xee,
        0x22, 0xaa, 0x66, 0xee, 0x22, 0xaa, 0x66, 0xee };


    blit_three_op = blit_rule[mode];
    fill_three_op = fill_rule[mode]; 
    color0_three_op = color0_rule[mode];
    color1_three_op = color1_rule[mode];

}


DrawingMode(mode)
int mode;
{

    if ((mode >= 0) && (mode <= 15)) {

	generate_three_ops(mode);

        WAIT_FOR_RUG;
	/* overlay planes not currently enabled */
	if (!overlay_enable) {
	    pri_draw_mode = mode;
            *(cntrl + PRR0) = mode;
            *((reg *)(cntrl + WRR0)) = mode;
            *(cntrl + TRR0) = fill_three_op; 
	}

	/* overlay planes selected and currently enabled */
	else {
	    ovl_draw_mode = mode;
	    if (cats_nplanes == 8) {
                *(cntrl + PRR1) = mode;
                *((reg *)(cntrl + WRR1)) = mode;
                *(cntrl + TRR1) = fill_three_op; 
	    }
	    else {
                *(cntrl + PRR0) = mode;
                *((reg *)(cntrl + WRR0)) = mode;
                *(cntrl + TRR0) = fill_three_op; 
	    }
	}

        cats_draw_mode = mode;
    }

}


FillColorIndex(color)
int color;
{
    /* overlay planes not selected */
    if (!overlay_select) {
	switch (cats_nplanes) {
	    case 8: pri_fill_color = color & 0xff;
		    break;
	    case 6: pri_fill_color = color & 0x3f;
		    break;
            case 1:
	    default: pri_fill_color = color & 0x01;
		     break;
        }
	cats_fill_color = pri_fill_color;
    }

    /* overlay planes selected and enabled */
    else if (overlay_enable) {
	cats_fill_color = ovl_fill_color = color & 0x03;
    }

    /* overlay planes selected but not enabled */
    else {
	pri_fill_color = (cats_nplanes == 8) ? color & 0xff : color & 0x0f;
	cats_fill_color = pri_fill_color;
    }
}


LineColorIndex(color)
int color;
{
    /* NOTE: the line color index is the default value in the COLOR registers */

    WAIT_FOR_RUG;

    /* overlay planes not selected */
    if (!overlay_select) {
	switch (cats_nplanes) {
	    case 8: pri_line_color = color & 0xff;
		    break;
	    case 6: pri_line_color = color & 0x3f;
		    break;
            case 1:
	    default: pri_line_color = color & 0x01;
		     break;
        }
	cats_line_color = pri_line_color;
	*(cntrl + COLOR0) = cats_line_color;
    }

    /* overlay planes selected and enabled */
    else if (overlay_enable) {
	cats_line_color = ovl_line_color = color & 0x03;
        if (cats_nplanes == 8)
	    *(cntrl + COLOR1) = cats_line_color;
        else
	    *(cntrl + COLOR0) = cats_line_color << 4;
    }

    /* overlay planes selected but not enabled */
    else {
	pri_line_color = (cats_nplanes == 8) ? color & 0xff : color & 0x0f;
	cats_line_color = pri_line_color;
	*(cntrl + COLOR0) = pri_line_color;
    }
}


EnableDoubleBuffer(flag)
int flag;
{
    register unsigned char rval;

    /* Kindergarten colors and gray scale */
    static int rgb[16][3] = {
	0, 0, 0, 255, 255, 255, 255, 0, 0, 255, 255, 0,
	0, 255, 0, 0, 255, 255, 0, 0, 255, 255, 0, 255,
	26, 26, 26, 51, 51, 51, 76, 76, 76, 102, 102, 102,
	128, 128, 128, 153, 153, 153, 178, 178, 178, 204, 204, 204 };

    if (!flag) {
	/* disable double buffering */

	if ((db_mode == DB_OFF) || 
	    ((db_mode == DB_PRI) && (overlay_enable)) ||
	    ((db_mode == DB_OVL) && (!overlay_enable)))
	    return;

        else {
	    WAIT_FOR_RUG;
	    switch (cats_nplanes) {
		case 8: if (!overlay_enable) {
			    DefineColorTable(0,256,oldrgb);
			    *(cntrl + FBEN0) = 0xff;
			    ClearScreen();
			    WAIT_FOR_RUG;
			    *(cntrl + FBEN0) = cats_write_mask;
			    *((reg *) (cntrl + PMASK)) = cats_disp_mask;
                        }
			else {
			    DefineColorTable(0,3,oldrgb);
			    *(cntrl + FBEN1) = 0x03;
			    ClearScreen();
			    WAIT_FOR_RUG;
			    *(cntrl + FBEN1) = cats_write_mask;
			    rval = *((reg *) (cntrl + OVLCNTRL));
			    *((reg *) (cntrl + OVLCNTRL)) = (rval & 0x1c) | (cats_disp_mask & 0x03);
                        }
			break;
                case 6: if (!overlay_select) {
			    DefineColorTable(0,64,oldrgb);
			    *(cntrl + FBEN0) = 0x3f;
			    ClearScreen();
			    WAIT_FOR_RUG;
			    *(cntrl + FBEN0) = cats_write_mask;
			    *((reg *) (cntrl + PMASK)) = cats_disp_mask;
                        }
			else if (!overlay_enable) {
			    DefineColorTable(0,16,oldrgb);
			    *(cntrl + FBEN0) = 0x0f;
			    ClearScreen();
			    WAIT_FOR_RUG;
			    *(cntrl + FBEN0) = cats_write_mask;
			    *((reg *) (cntrl + PMASK)) = cats_disp_mask;
                        }
			else {
			    DefineColorTable(0,3,oldrgb);
			    *(cntrl + FBEN0) = 0x30;
			    ClearScreen();
			    WAIT_FOR_RUG;
			    *(cntrl + FBEN0) = cats_write_mask;
			    rval = *((reg *) (cntrl + OVLCNTRL));
			    *((reg *) (cntrl + OVLCNTRL)) = (rval & 0x1c) | (cats_disp_mask & 0x3);
                        }
                default: break;
	    }
        }
	db_mode = DB_OFF;
    }

    else {
	/* enable double buffering */

        if ((cats_nplanes == 1) || (db_mode != DB_OFF)) 
	    return;

        else if (cats_nplanes == 6) {
	    WAIT_FOR_RUG;
	    if (!overlay_select) {
	        /* 6 plane primary */
	        ReadColorTable(0,64,oldrgb);
	        DefineColorTable(0,8,rgb);
                DefineColorTable(8,1,&rgb[1][0]);
                DefineColorTable(16,1,&rgb[2][0]);
                DefineColorTable(24,1,&rgb[3][0]);
                DefineColorTable(32,1,&rgb[4][0]);
                DefineColorTable(40,1,&rgb[5][0]);
                DefineColorTable(48,1,&rgb[6][0]);
                DefineColorTable(56,1,&rgb[7][0]);
	        db_mode = DB_PRI;
	        db_disp_index = 0;
            }
	    else if (!overlay_enable) {
	        /* 4 plane primary index */
	        ReadColorTable(0,16,oldrgb);
	        DefineColorTable(0,4,rgb);
                DefineColorTable(4,1,&rgb[1][0]);
                DefineColorTable(8,1,&rgb[2][0]);
                DefineColorTable(12,1,&rgb[3][0]);
	        db_mode = DB_PRI;
	        db_disp_index = 1;
            }
	    else {
	        /* overlay planes */
	        ReadColorTable(0,3,oldrgb);
	        DefineColorTable(0,2,rgb);
	        DefineColorTable(2,1,&rgb[1][0]);
	        db_mode = DB_OVL;
	        db_disp_index = 2;
            }
        }
        else if (cats_nplanes == 8) {
	    if (!overlay_enable) {
	        /* 8 plane primary */
                ReadColorTable(0,256,oldrgb);
                DefineColorTable(0,16,rgb);
                DefineColorTable(16,1,&rgb[1][0]);
                DefineColorTable(32,1,&rgb[2][0]);
                DefineColorTable(48,1,&rgb[3][0]);
                DefineColorTable(64,1,&rgb[4][0]);
                DefineColorTable(80,1,&rgb[5][0]);
                DefineColorTable(96,1,&rgb[6][0]);
                DefineColorTable(112,1,&rgb[7][0]);
                DefineColorTable(128,1,&rgb[8][0]);
                DefineColorTable(144,1,&rgb[9][0]);
                DefineColorTable(160,1,&rgb[10][0]);
                DefineColorTable(176,1,&rgb[11][0]);
                DefineColorTable(192,1,&rgb[12][0]);
                DefineColorTable(208,1,&rgb[13][0]);
                DefineColorTable(224,1,&rgb[14][0]);
                DefineColorTable(240,1,&rgb[15][0]);
	        db_mode = DB_PRI;
	        db_disp_index = 3;
            }
	    else {
	        /* overlay planes */
	        ReadColorTable(0,3,oldrgb);
	        DefineColorTable(0,2,rgb);
	        DefineColorTable(2,1,&rgb[1][0]);
	        db_mode = DB_OVL;
	        db_disp_index = 4;
            }
        }

        /* Set up initial config lower set displayed, upper set writable */
        WAIT_FOR_RETRACE;
        if (!overlay_enable) {
            *((reg *) (cntrl + PMASK)) = db_disp_mask[db_disp_index][0];
            *(cntrl + FBEN0) = db_write_mask[db_disp_index][0];
        }
        else {
	    rval = *((reg *) (cntrl + OVLCNTRL));
	    *((reg *) (cntrl + OVLCNTRL)) = (rval & 0x1c) | (db_disp_mask[db_disp_index][0]);
	    if (cats_nplanes == 8)
                *(cntrl + FBEN1) = db_write_mask[db_disp_index][0];
            else
                *(cntrl + FBEN0) = db_write_mask[db_disp_index][0];
        }
    }
}


DoubleBuffer(flag)
int flag;
{
    register int nflg;
    register unsigned char rval;

    if ((cats_nplanes == 1) || (db_mode == DB_OFF) ||
	((db_mode == DB_PRI) && (overlay_enable)) ||
	((db_mode == DB_OVL) && (!overlay_enable)))
        return;

    nflg = flag & 0x1;
    db_flag = nflg;

    WAIT_FOR_RETRACE;
    if (!overlay_enable) {
        *((reg *) (cntrl + PMASK)) = db_disp_mask[db_disp_index][nflg];
        *(cntrl + FBEN0) = db_write_mask[db_disp_index][nflg];
    }
    else {
	rval = *((reg *) (cntrl + OVLCNTRL));
	*((reg *) (cntrl + OVLCNTRL)) = (rval & 0x1c) | (db_disp_mask[db_disp_index][nflg]);
	if (cats_nplanes == 8)
            *(cntrl + FBEN1) = db_write_mask[db_disp_index][nflg];
        else
            *(cntrl + FBEN0) = db_write_mask[db_disp_index][nflg];
    }

    ClearScreen();
}

/************************** RUG ROUTINES *************************************/


Line(x1,y1,x2,y2)
int x1, y1, x2, y2;
{
    WAIT_FOR_RUG;
    *((reg *)(cntrl + COMMAND)) = cats_rug_enable | cats_linetype_enable;
    *((reg *)(cntrl + XSRC)) = x1;
    *((reg *)(cntrl + YSRC)) = y1;
    *((reg *)(cntrl + XDST)) = x2;
    CHECK_AND_WAIT_FOR_RETRACE;
    *((reg *)(cntrl + TYDST)) = y2;
    cats_cx = x2;
    cats_cy = y2;
}


Move(x,y)
int x,y;
{
    cats_cx = x;
    cats_cy = y;
}


Draw(x,y)
int x,y;
{
    WAIT_FOR_RUG;
    *((reg *)(cntrl + COMMAND)) = cats_rug_enable | cats_linetype_enable;
    *((reg *)(cntrl + XSRC)) = cats_cx;
    *((reg *)(cntrl + YSRC)) = cats_cy;
    *((reg *)(cntrl + XDST)) = x;
    CHECK_AND_WAIT_FOR_RETRACE;
    *((reg *)(cntrl + TYDST)) = y;
    cats_cx = x;
    cats_cy = y;
   
}

Polyline(num,plist)
int num;
int plist[];
{
    register int i, flip;
    register int *ptr, *endptr;

    WAIT_FOR_RUG;
    if (cats_enable_linetype) {
        *((reg *)(cntrl + COMMAND)) = cats_rug_enable | 0x0040;
        ptr = plist;
        endptr = plist + 2;
        for (i=1; i<num; i++) {
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = *(ptr++);
            *((reg *)(cntrl + YSRC)) = *(ptr++);
            *((reg *)(cntrl + XDST)) = *(endptr++);
            *((reg *)(cntrl + TYDST)) = *(endptr++);
        }
        endptr -= 2;
        cats_cx = *(endptr++);
        cats_cy = *endptr;
    }

    /* optimization for non-linetype case */
    else {
        *((reg *)(cntrl + COMMAND)) = cats_rug_enable;
        flip = 0;
        ptr = plist;
        endptr = plist + (num << 1);
        *((reg *)(cntrl + XSRC)) = *(ptr++);
        *((reg *)(cntrl + YSRC)) = *(ptr++);
        while (ptr < endptr) {
            WAIT_FOR_RUG;
            if (flip) {
                *((reg *)(cntrl + XSRC)) = *(ptr++);
                *((reg *)(cntrl + TYSRC)) = *(ptr++);
            }
            else {
                *((reg *)(cntrl + XDST)) = *(ptr++);
                *((reg *)(cntrl + TYDST)) = *(ptr++);
            }
            flip = !flip;
        }
        ptr -= 2;
        cats_cx = *(ptr++);
        cats_cy = *ptr;
    }
}


Polygon(num,plist)
int num;
int plist[];
{
    if (cats_poly_filled == INT_SOLID) {
        raw_polygon(num,plist,FALSE);
    }
    else if (cats_poly_filled == INT_PATTERN) {
        raw_polygon(num,plist,TRUE);
    }

    if (cats_poly_edged) {
        Polyline(num,plist);
        Draw(plist[0],plist[1]);
    }
}


raw_polygon(num,plist,pattern)
int num;
int plist[];
int pattern;
{
    register int i, cnt, xmin, xmax, ymin, ymax;
    register int *pptr, *ymin_ptr, *endptr;
    reg old_control_reg;
    int old_fben0, old_fben1;                        /* for double buffering */

    /* determine minimum y point for anchor and bounding box */
    pptr = plist;
    endptr = pptr + 2*(num-1) + 1;
    ymin_ptr = pptr + 1;
    xmin = xmax = *(pptr++);
    ymin_ptr = pptr;
    ymin = ymax = *(pptr++);

    while (pptr < endptr) {
        if (*pptr < xmin) xmin = *pptr;
        if (*pptr > xmax) xmax = *pptr;
        pptr++;
        if (*pptr < ymin) {
            ymin = *pptr;
            ymin_ptr = pptr;
        }
        if (*pptr > ymax) ymax = *pptr;
        pptr++;
    }

    /* if this polygon is totally clipped then exit */
    if (cats_clipping_indicator)
	if ((xmax < cats_clip_xmin) | (ymax < cats_clip_ymin) |
	    (xmin > cats_clip_xmax) | (ymin > cats_clip_ymax))
	    return;

    /* set up for drawing polygon */
    WAIT_FOR_RUG;
    CHECK_AND_WAIT_FOR_RETRACE;

    /* set up to draw only in scratch plane */
    *(cntrl + VB) = 0x00;                     /* Blit mode */
    *(cntrl + PNCNTRL) = 0x00;                /* Within plane */
    switch (cats_nplanes) {
        case  8 : /* HRC */
		  old_fben1 = *(cntrl + FBEN1);
                  *(cntrl + FBEN1) = 0x04;     /* phantom only */
		  if (overlay_enable) {      /* make sure other RRs are fast */
		      *(cntrl + PRR1) = 0x00;
		      *((reg *)(cntrl + WRR1)) = 0x0000;
		  }
		  else {
		      old_fben0 = *(cntrl + FBEN0);
		      *(cntrl + FBEN0) = 0x00;
		      *(cntrl + PRR0) = 0x00;
		      *((reg *)(cntrl + WRR0)) = 0x0000;
		  }
		  *(cntrl + TCWEN1) = 0x04;
                  *((reg *)(cntrl + WRR1)) = 0x0000;
                  break;
        case  6 : /* LCC */
                  *((reg *)(cntrl + STATUS)) = 0x0080;    /* write to phantom */
		  old_fben0 = *(cntrl + FBEN0);
                  *(cntrl + FBEN0) = 0x20;     /* phantom only */
		  *(cntrl + PRR0) = 0x00;    /* make sure other RRs are fast */
		  *((reg *)(cntrl + WRR0)) = 0x0000;
		  *(cntrl + TCWEN0) = 0x20;
                  break;
        case  1 : /* Monochrome */
		  old_fben0 = *(cntrl + FBEN0);
                  *(cntrl + FBEN0) = 0x02;     /* phantom only */
		  *(cntrl + PRR0) = 0x00;   /* make sure other RRs are fast */
                  *((reg *)(cntrl + WRR0)) = 0x0000;
		  *(cntrl + TCWEN0) = 0x02;
                  *((reg *)(cntrl + WRR0)) = 0x0000;
                  break;
        default : break;
    }

    /* clear drawing area in phantom plane */
    *((reg *)(cntrl + COMMAND)) = 0x0090;   /* set RUG for BLITS */
    *((reg *)(cntrl + XSRC)) = xmin;
    *((reg *)(cntrl + YSRC)) = ymin;
    *((reg *)(cntrl + XDST)) = xmin;
    *((reg *)(cntrl + YDST)) = ymin;
    *((reg *)(cntrl + FX)) = xmax - xmin + 1;
    *((reg *)(cntrl + TFY)) = ymax - ymin + 1;

    /* draw polygon, one triangle at a time */
    WAIT_FOR_RUG;
    *(cntrl + VB) = 0x01;   /* set BARC for vectors */
    switch (cats_nplanes) {
        case  8 : /* HRC */
                  *(cntrl + COLOR1) = 0x04;  /* Draw to phantom */
                  *(cntrl + PRR1) = 0x06;  /* PRR = XOR */
                  break;
        case  6 : /* LCC */
                  *(cntrl + COLOR0) = 0x20;  /* Draw to phantom */
                  *(cntrl + PRR0) = 0x06;  /* PRR = XOR */
                  break;
        case  1 : /* Monochrome */
                  *(cntrl + COLOR0) = 0x02;  /* Draw to phantom */
                  *(cntrl + PRR0) = 0x06;  /* PRR = XOR */
                  break;
        default : break;
    }
    *((reg *)(cntrl + COMMAND)) = 0x00d0;   /* set RUG for fill */
    *((reg *)(cntrl + FX)) = *(ymin_ptr - 1);  /* anchor x */
    *((reg *)(cntrl + FY)) = *ymin_ptr;        /* anchor y */
    pptr = ymin_ptr + 1;                           /* to x pos following ymin */
    endptr -= 1;                                   /* to last x pos */
    cnt = num - 2;                                 /* num triangles to draw */
    if ((endptr - pptr) < 0) {                     /* anchor is last point */
        pptr = plist;
        for (i=0; i<cnt; i++) {
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = *(pptr++);
            *((reg *)(cntrl + YSRC)) = *(pptr++);
            *((reg *)(cntrl + XDST)) = *(pptr);
            *((reg *)(cntrl + TYDST)) = *(pptr + 1);
        }
    }
    else {
        for (i=0; i<cnt; i++) {
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = *(pptr++);
            *((reg *)(cntrl + YSRC)) = *(pptr++);
            if (pptr > endptr) pptr = plist;
            *((reg *)(cntrl + XDST)) = *(pptr);
            *((reg *)(cntrl + TYDST)) = *(pptr + 1);
        }
    }

    /* set up to edge polygon */
    WAIT_FOR_RUG;
    if (cats_nplanes == 8)
        *(cntrl + PRR1) = 0x0f;   /* PRR = 1 */
    else
        *(cntrl + PRR0) = 0x0f;   /* PRR = 1 */
    *((reg *)(cntrl + COMMAND)) = 0x0010;   /* solid line type vectors */
    cnt = 0;
    pptr = plist;
    endptr = plist + 2*num;
    *((reg *)(cntrl + XSRC)) = *(pptr++);
    *((reg *)(cntrl + YSRC)) = *(pptr++);
    while (pptr < endptr) {
        WAIT_FOR_RUG;
        if (cnt) {
            *((reg *)(cntrl + XSRC)) = *(pptr++);
            *((reg *)(cntrl + TYSRC)) = *(pptr++);
        }
        else {
            *((reg *)(cntrl + XDST)) = *(pptr++);
            *((reg *)(cntrl + TYDST)) = *(pptr++);
        }
        cnt = !cnt;
    }
    WAIT_FOR_RUG;                            /* draw last segment */
    if (cnt) {
        *((reg *)(cntrl + XSRC)) = *(plist);
        *((reg *)(cntrl + TYSRC)) = *(plist + 1);
    }
    else {
        *((reg *)(cntrl + XDST)) = *(plist);
        *((reg *)(cntrl + TYDST)) = *(plist + 1);
    }

    /* set up to blit from phantom plane to main planes */
    WAIT_FOR_RUG;
    old_control_reg = *((reg *)(cntrl + COMMAND));   /* save for picking */
    switch (cats_nplanes) {
        case  8 : /* HRC */
                  *(cntrl + PNCNTRL) = 0x1a;   /* blit source */
		  *(cntrl + PRR1) = 0x00;      /* reset phantom */
		  *((reg *)(cntrl + WRR1)) = 0x0000;
                  break;
        case  6 : /* LCC */
                  *((reg *)(cntrl + STATUS)) = 0x0040;     /* read phantom */
                  *(cntrl + PNCNTRL) = 0x15;   /* blit source */
                  break;
        case  1 : /* Monochrome */
                  *(cntrl + PNCNTRL) = 0x11;   /* blit source */
		  *(cntrl + PRR0) = 0x00;      /* reset phantom */
		  *((reg *)(cntrl + WRR0)) = 0x0000;
                  break;
        default : break;
    }
    if (pattern) {                                     /* pattern fill */
	if ((cats_nplanes == 8) && (overlay_enable)) {
	    *(cntrl + TCWEN1) = 0x03;
            *(cntrl + TRR1) = blit_three_op;
	    *(cntrl + FBEN1) = old_fben1;
	}
	else {
	    *(cntrl + TCWEN0) = (cats_nplanes == 1) ? 0x01 : 0xff;
            *(cntrl + TRR0) = blit_three_op;
	    *(cntrl + FBEN0) = old_fben0;
	}
    }
    else {                                             /* solid fill */
	/* overlay planes not enabled */
	if (!overlay_enable) {
            *(cntrl + TCWEN0) = cats_fill_color;
            *(cntrl + TRR0) = color1_three_op;
            *(cntrl + TCWEN0) = ~cats_fill_color;
            *(cntrl + TRR0) = color0_three_op;
            *(cntrl + TCWEN0) = (cats_nplanes == 1) ? 0x01 : 0xff;
            *(cntrl + FBEN0) = old_fben0;
	}

	/* overlay planes selected and enabled */
	else {
	    if (cats_nplanes == 8) {
                *(cntrl + TCWEN1) = cats_fill_color;
                *(cntrl + TRR1) = color1_three_op;
                *(cntrl + TCWEN1) = ~cats_fill_color;
                *(cntrl + TRR1) = color0_three_op;
                *(cntrl + TCWEN1) = 0x03;
                *(cntrl + FBEN1) = old_fben1;
	    }
	    else {
                *(cntrl + TCWEN0) = cats_fill_color << 4;
                *(cntrl + TRR0) = color1_three_op;
                *(cntrl + TCWEN0) = ~(cats_fill_color << 4);
                *(cntrl + TRR0) = color0_three_op;
                *(cntrl + TCWEN0) = (cats_nplanes == 1) ? 0x01 : 0xff;
                *(cntrl + FBEN0) = old_fben0;
	    }
	}
    }
    *(cntrl + TCNTRL) = 0x01;
    *(cntrl + VB) = 0x00;   /* set BARC for blits */
    *((reg *)(cntrl + COMMAND)) = 0x0090;
    *((reg *)(cntrl + XSRC)) = xmin;
    *((reg *)(cntrl + YSRC)) = ymin;
    *((reg *)(cntrl + XDST)) = xmin;
    *((reg *)(cntrl + YDST)) = ymin;
    *((reg *)(cntrl + FX)) = xmax - xmin + 1;
    CHECK_AND_WAIT_FOR_RETRACE;
    *((reg *)(cntrl + TFY)) = ymax - ymin + 1;

    /* reset state */
    WAIT_FOR_RUG;
    *((reg *)(cntrl + COMMAND)) = old_control_reg; /* restore picking info */
    *(cntrl + TCNTRL) = 0x00;
    if (!overlay_enable) {
        *(cntrl + FBEN0) = old_fben0;
        *(cntrl + TRR0) = fill_three_op;
	*(cntrl + COLOR0) = cats_line_color;
        *(cntrl + PRR0) = cats_draw_mode;
        *((reg *)(cntrl + WRR0)) = cats_draw_mode;
        if (cats_nplanes == 8) {
            *(cntrl + FBEN1) = 0x00;   /* disable phantom */
	    *(cntrl + COLOR1) = 0x00;
	}
    }
    else {
	if (cats_nplanes == 8) {
            *(cntrl + FBEN1) = old_fben1;
            *(cntrl + TRR1) = fill_three_op;
	    *(cntrl + COLOR1) = cats_line_color;
            *(cntrl + PRR1) = cats_draw_mode;
            *((reg *)(cntrl + WRR1)) = cats_draw_mode;
	}
	else {
            *(cntrl + FBEN0) = old_fben0;
            *(cntrl + TRR0) = fill_three_op;
	    *(cntrl + COLOR0) = cats_line_color << 4;
            *(cntrl + PRR0) = cats_draw_mode;
            *((reg *)(cntrl + WRR0)) = cats_draw_mode;
	}
    }
    *(cntrl + VB) = 0x01;   /* set BARC for vectors */
    if (cats_nplanes == 6)
        *((reg *)(cntrl + STATUS)) = 0x0000;   /* disable phantom */

}

Circle(x,y,r)
int r, x, y;
{
    CHECK_AND_WAIT_FOR_RETRACE;
    if (cats_poly_filled == INT_SOLID) {
        WAIT_FOR_RUG;
	if (!overlay_enable)
            *(cntrl + COLOR0) = cats_fill_color;
	else {
	    if (cats_nplanes == 8)
                *(cntrl + COLOR1) = cats_fill_color;
	    else
                *(cntrl + COLOR0) = cats_fill_color << 4;
	}
        *((reg *)(cntrl + COMMAND)) = cats_rug_enable | 0x00e0;
        *((reg *)(cntrl + XSRCXDST)) = x;
        *((reg *)(cntrl + YSRC)) = y;
        *((reg *)(cntrl + TFY)) = r;

        WAIT_FOR_RUG;
	/* reset COLOR registers to default */
	if (!overlay_enable)
            *(cntrl + COLOR0) = cats_line_color;
        else {
	    if (cats_nplanes == 8)
                *(cntrl + COLOR1) = cats_line_color;
	    else
                *(cntrl + COLOR0) = cats_line_color << 4;
	}
    }
    else if (cats_poly_filled == INT_PATTERN) {
        WAIT_FOR_RUG;
        *(cntrl + TCNTRL) = 0x01;         /* Pattern filled */
        *((reg *)(cntrl + COMMAND)) = cats_rug_enable | 0x00e0;
        *((reg *)(cntrl + XSRCXDST)) = x;
        *((reg *)(cntrl + YSRC)) = y;
        *((reg *)(cntrl + TFY)) = r;

        WAIT_FOR_RUG;
        *(cntrl + TCNTRL) = 0x00;         /* Reset */
    }

    if (cats_poly_edged) {
        WAIT_FOR_RUG;
        *((reg *)(cntrl + COMMAND)) = 0x0020 | cats_rug_enable | cats_linetype_enable;
        *((reg *)(cntrl + XSRCXDST)) = x;
        *((reg *)(cntrl + YSRC)) = y;
        *((reg *)(cntrl + TFY)) = r;
    }
}

Rectangle(x1,y1,x2,y2)
int x1, y1, x2, y2;
{
    register int sx, sy, dx, dy;

    /* find the minimum x and y endpoints to start drawing from */
    if (x1 < x2) {
        sx = x1; dx = x2;
    } else {
        sx = x2; dx = x1;
    }
    if (y1 < y2) {
        sy = y1; dy = y2;
    } else {
        sy = y2; dy = y1;
    }
        

    CHECK_AND_WAIT_FOR_RETRACE;
    if (cats_poly_filled == INT_SOLID) {
        WAIT_FOR_RUG;
	if (!overlay_enable)
            *(cntrl + COLOR0) = cats_fill_color;
	else {
	    if (cats_nplanes == 8)
                *(cntrl + COLOR1) = cats_fill_color;
	    else
                *(cntrl + COLOR0) = cats_fill_color << 4;
	}
        *(cntrl + VB) = 0x01;             /* Blit using color */
        *((reg *)(cntrl + COMMAND)) = 0x0080 | cats_rug_enable;
        *((reg *)(cntrl + XSRC)) = sx;
        *((reg *)(cntrl + YSRC)) = sy;
        *((reg *)(cntrl + XDST)) = sx;
        *((reg *)(cntrl + YDST)) = sy;
        *((reg *)(cntrl + FX)) = dx - sx + 1;
        *((reg *)(cntrl + TFY)) = dy - sy + 1;

        WAIT_FOR_RUG;
	if (!overlay_enable)
            *(cntrl + COLOR0) = cats_line_color;
	else {
	    if (cats_nplanes == 8)
                *(cntrl + COLOR1) = cats_line_color;
	    else
                *(cntrl + COLOR0) = cats_line_color << 4;
	}
    }
    else if (cats_poly_filled == INT_PATTERN) {
        WAIT_FOR_RUG;
        *(cntrl + TCNTRL) = 0x01;         /* Pattern filled */
        *(cntrl + VB) = 0x00;
        *((reg *)(cntrl + COMMAND)) = 0x0080 | cats_rug_enable;
        *((reg *)(cntrl + XSRC)) = sx;
        *((reg *)(cntrl + YSRC)) = sy;
        *((reg *)(cntrl + XDST)) = sx;
        *((reg *)(cntrl + YDST)) = sy;
        *((reg *)(cntrl + FX)) = dx - sx + 1;
        *((reg *)(cntrl + TFY)) = dy - sy + 1;

        WAIT_FOR_RUG;
        *(cntrl + TCNTRL) = 0x00;         /* Reset TCNTRL */
        *(cntrl + VB) = 0x01;             /* Reset VB */
    }

    if (cats_poly_edged) {
        WAIT_FOR_RUG;
        if (cats_enable_linetype) {
            *((reg *)(cntrl + COMMAND)) = 0x0040 | cats_rug_enable;
            *((reg *)(cntrl + XSRC)) = sx;
            *((reg *)(cntrl + YSRC)) = sy;
            *((reg *)(cntrl + XDST)) = dx;
            *((reg *)(cntrl + TYDST)) = sy;
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = dx;
            *((reg *)(cntrl + YSRC)) = sy;
            *((reg *)(cntrl + XDST)) = dx;
            *((reg *)(cntrl + TYDST)) = dy;
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = dx;
            *((reg *)(cntrl + YSRC)) = dy;
            *((reg *)(cntrl + XDST)) = sx;
            *((reg *)(cntrl + TYDST)) = dy;
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = sx;
            *((reg *)(cntrl + YSRC)) = dy;
            *((reg *)(cntrl + XDST)) = sx;
            *((reg *)(cntrl + TYDST)) = sy;
        }
        else {
            *((reg *)(cntrl + COMMAND)) = cats_rug_enable;
            *((reg *)(cntrl + XSRC)) = sx;
            *((reg *)(cntrl + YSRC)) = sy;
            *((reg *)(cntrl + XDST)) = dx;
            *((reg *)(cntrl + TYDST)) = sy;
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = dx;
            *((reg *)(cntrl + TYSRC)) = dy;
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XDST)) = sx;
            *((reg *)(cntrl + TYDST)) = dy;
            WAIT_FOR_RUG;
            *((reg *)(cntrl + XSRC)) = sx;
            *((reg *)(cntrl + TYSRC)) = sy;
        }
    }
}

ClipLimits(x1,y1,x2,y2)
int x1, y1, x2, y2;
{
    if (overlay_enable) {
	ovl_clip_xmin = x1;
	ovl_clip_ymin = y1;
	ovl_clip_xmax = x2;
	ovl_clip_ymax = y2;
    }
    else {
	pri_clip_xmin = x1;
	pri_clip_ymin = y1;
	pri_clip_xmax = x2;
	pri_clip_ymax = y2;
    }

    WAIT_FOR_RUG;
    *((reg *)(cntrl + CXMIN)) = x1;
    cats_clip_xmin = x1;
    *((reg *)(cntrl + CXMAX)) = x2;
    cats_clip_xmax = x2;
    *((reg *)(cntrl + CYMIN)) = y1;
    cats_clip_ymin = y1;
    *((reg *)(cntrl + CYMAX)) = y2;
    cats_clip_ymax = y2;
}

ClipEnable(flag)
int flag;
{
    cats_clipping_indicator = flag & 0x1;
    if (overlay_enable)
	ovl_clipping_indicator = cats_clipping_indicator;
    else
	pri_clipping_indicator = cats_clipping_indicator;

    set_clipping(cats_clipping_indicator);
}

set_clipping(flag)
int flag;
{

    /* NOTE: Due to a bug in RUG, page mode cycles for vertical vectors      */
    /*       must be shut off when clipping is enabled.  To do this without  */
    /*       shutting page mode off for horizontal scan lines, the number    */
    /*       of scan lines/page parameter in RUG CONTROL register is changed */
    /*       to 1 by setting the two least significant bits whenever         */
    /*       clipping is enabled.  It is changed back to 4 scan lines/page   */
    /*       when clipping is disabled so vertical vectors can run 31 %      */
    /*       faster.                                                         */
    WAIT_FOR_RUG;
    if (flag)
	*((reg *)(cntrl + CONTROL)) |= 0x0083;
    else
	*((reg *)(cntrl + CONTROL)) &= 0x007c;
}

SetLinetype(mask,mag)
reg mask;
int mag;
{
    WAIT_FOR_RUG;
    if (overlay_enable) {
	ovl_ltype = mask;
	ovl_ltp = (0xf - (mag & 0xf)) * 0x0110;
        *((reg *)(cntrl + LTYPE)) = ovl_ltype;
        *((reg *)(cntrl + LTP)) = ovl_ltp;
    }
    else {
	pri_ltype = mask;
	pri_ltp = (0xf - (mag & 0xf)) * 0x0110;
        *((reg *)(cntrl + LTYPE)) = pri_ltype;
        *((reg *)(cntrl + LTP)) = pri_ltp;
    }
}


LinetypeEnable(flag)
int flag;
{
    cats_enable_linetype = flag;

    if (flag == FALSE)
	cats_linetype_enable = 0x0000;
    else
        cats_linetype_enable = 0x0040;

    if (overlay_enable) {
	ovl_linetype_enable = cats_linetype_enable;
	ovl_enable_linetype = flag;
    }
    else {
	pri_linetype_enable = cats_linetype_enable;
	pri_enable_linetype = flag;
    }
}


SetPickMode(flag)
int flag;
{
    if (flag == FALSE)
	cats_rug_enable = 0x0000;
    else
        cats_rug_enable = 0x0010;

    if (overlay_enable)
	ovl_rug_enable = cats_rug_enable;
    else
	pri_rug_enable = cats_rug_enable;
}


int ReadPickBit()
{
    WAIT_FOR_RUG;
    if (*((reg *)(cntrl + COMMAND)) & 0x8)
	return(TRUE);
    else
	return(FALSE);
}


LoadPattern(plane, pattern)
int plane;
reg pattern[];
{
    register int valid, offset;
    unsigned char *regptr, *endptr;
    reg *patptr;


    if (plane < 0)
	return;

    /* compute base address into pattern registers for selected plane */
    valid = TRUE;
    switch (cats_nplanes) {
        case 8 : /* HRC */
                 if ((!overlay_enable) && (plane < 8))
                     offset = 0x4400 + 0x20 * plane;
                 else if ((overlay_enable) && (plane <= 1))
                     offset = 0x4600 + 0x20 * plane;
		 else
                     valid = FALSE;
                 break;
        case 6 : /* LCC */
                 if ((!overlay_select) && (plane < 6)) 
                     offset = 0x4400 + 0x20 * plane;
                 else if ((!overlay_enable) && (plane < 4))
                     offset = 0x4400 + 0x20 * plane;
                 else if ((overlay_enable) && (plane <= 1))
                     offset = 0x4480 + 0x20 * plane;
		 else
                     valid = FALSE;
                 break;
        case 1 : /* Monochrome */
                 if (plane <= 1)
                     offset = 0x4400 + 0x20 * plane;
		 else
                     valid = FALSE;
                 break;
        default: valid = FALSE;
                 break;
    }

    /* if valid, then write data into pattern registers */
    if (valid) {
	WAIT_FOR_RUG;
	patptr = pattern;
	regptr = cntrl + offset;
	endptr = regptr + 0x20;
        while (regptr < endptr)
            *((reg *) regptr)++ = *patptr++;
    }
}



LoadPatterns(pattern)
unsigned char pattern[];
{
    register int i, j, k;
    int num_planes;
    reg packed_pattern[16];

    /* compute number of planes to modify */
    if (!overlay_select)
	num_planes = cats_nplanes;

    else if (!overlay_enable) 
	num_planes = (cats_nplanes == 6) ? 4 : 8;

    else
	num_planes = 2;

    /* generate pattern for each plane */
    for (i=0; i<num_planes; i++) {
	for (j=0; j<16; j++) {
	    packed_pattern[j] = 0;
	    for (k=0; k<16; k++) 
	        packed_pattern[j] = (packed_pattern[j] << 1) |
				    ((pattern[(j << 4) | k] >> i) & 0x1);
        }

	LoadPattern(i, packed_pattern);
    }
}



ClearScreen()
{
    int old_blit_type;
    reg old_blit_source_mask;
    int old_drawing_mode;

    /* save old blit state */
    old_blit_type = blit_type;
    old_blit_source_mask = blit_source_mask;
    old_drawing_mode = cats_draw_mode;

    /* set up for clear blit */
    DrawingMode(0);
    BlockMoveType(BLIT_WITHIN,0);
    BlockMove(0,0,0,0,cats_max_x+1,cats_max_y+1);

    /* reset old state */
    blit_type = old_blit_type;
    blit_source_mask = old_blit_source_mask;
    DrawingMode(old_drawing_mode);
}




BlockMoveType(type, source_plane)
int type;
reg source_plane;
{
    blit_type = type;
    if (blit_type == BLIT_BETWEEN) 
        blit_source_mask = 0x10 | (source_plane & 0x0f);
    else
        blit_source_mask = 0x00;
}



BlockMove(sx, sy, dx, dy, w, h)
int sx, sy, dx, dy, h, w;
{
    WAIT_FOR_RUG;
    /* set up for block move */
    *(cntrl + VB) = 0x00;         /* set BARC for blits */
    *(cntrl + PNCNTRL) = blit_source_mask;

    /* perform block move */
    *((reg *)(cntrl + COMMAND)) = 0x0090;
    *((reg *)(cntrl + XSRC)) = sx;
    *((reg *)(cntrl + YSRC)) = sy;
    *((reg *)(cntrl + XDST)) = dx;
    *((reg *)(cntrl + YDST)) = dy;
    *((reg *)(cntrl + FX)) = w;
    CHECK_AND_WAIT_FOR_RETRACE;
    *((reg *)(cntrl + TFY)) = h;

    /* reset */
    WAIT_FOR_RUG;
    *(cntrl + VB) = 0x01;         /* set BARC for vectors */
    *(cntrl + PNCNTRL) = 0x00;
}


ReadBlock(sx, sy, dx, dy, data)
int sx, sy;
int dx, dy;
unsigned char *data;
{
    register int i, deltax, screen_x;
    register reg read_mask;
    register unsigned char *ptr, *leptr, *dptr;
    int no_need_to_shift, do_first, do_end;

    /* set up initial pointers and mask */
    no_need_to_shift = TRUE;
    WAIT_FOR_RUG;
    switch (cats_nplanes) {
        case 8: if (overlay_enable) {
		    read_mask = 0x0303;
		    *(cntrl + ACNTRL) = 0x02;
		}
		else {
		    read_mask = 0xffff;
		    *(cntrl + ACNTRL) = 0x00;
		}
                screen_x = 2048;
                break;
        case 6: if (!overlay_select)
		     read_mask = 0x3f3f;
                else if (!overlay_enable)
		     read_mask = 0x0f0f;
                else {
		     read_mask = 0x0303;
		     no_need_to_shift = FALSE;
                }
		*(cntrl + ACNTRL) = 0x00;
                screen_x = 1024;
                break;
        case 1: 
        default: read_mask = 0x0101;
		*(cntrl + ACNTRL) = 0x00;
                screen_x = 2048;
                break;
    }
    do_first = sx & 0x1;
    do_end = (sx + dx) & 0x1;
    dptr = data;
    deltax = dx - 1 - do_end;              /* # bytes to leptr */
    ptr = frm_bfr + (sy * screen_x) + sx;
    leptr = ptr + deltax;                  /* leptr points to last byte */
					   /* in last full word         */

    /* place data into array */
    /* NOTE: this could be optimized further by doing 32 bit transfers */
    /*       (typecast to int) but you would have to one of 4 cases at */
    /*       the end of each scan line. (0, 1, 2, or 3 extra bytes)    */
    i = dy + 1;
    if (no_need_to_shift) {
        while (--i) {
	    if (do_first)             /* do first byte if not aligned */
		*dptr++ = *ptr++ & read_mask;

            while (ptr < leptr)       /* do all middle words */
		*((reg *) dptr)++ = *((reg *) ptr)++ & read_mask;

            if (do_end)               /* do last byte if not aligned */
		*dptr++ = *(leptr + 1) & read_mask;

            leptr += screen_x;
	    ptr = leptr - deltax;
        }
    }
    else {
        while (--i) {
	    if (do_first)             /* do first byte if not aligned */
		*dptr++ = (*ptr++ >> 4) & read_mask;

            while (ptr < leptr)       /* do all middle words */
		*((reg *) dptr)++ = (*((reg *) ptr)++ >> 4) & read_mask;

            if (do_end)               /* do last byte if not aligned */
		*dptr++ = (*(leptr + 1) >> 4) & read_mask;

            leptr += screen_x;
	    ptr = leptr - deltax;
        }
    }

    /* reset ACNTRL to default */
    *(cntrl + ACNTRL) = 0x00;

}



PackedReadBlock(sx, sy, w, h, plane, data)
int sx, sy, w, h, plane;
unsigned short *data;
{

    register int i, screen_x, realw;
    register unsigned char *ptr, *leptr;
    register unsigned short *dptr;
    int old_fben0, old_fben1;                /* for double buffering */

    /* make sure the user specified a plane supported on this card */
    if (plane < 0)
	return;
    else if ((overlay_enable) && (plane > 1))
	return;
    else if ((overlay_select) && (cats_nplanes == 6) && (plane > 3))
	return;
    else if (plane > cats_nplanes - 1)
	return;

    /* STEP 1: Move and align block from chosen plane to scratch plane */
    /* NOTE: This routine could be further optimized by checking for the */
    /*       case where the data is aligned and could be read directly   */
    /*       from the frame buffer instead of being blitted first.       */
    WAIT_FOR_RUG;

    /* set up for the block move */
    *(cntrl + VB) = 0x00;
    switch (cats_nplanes) {
	case 8 : old_fben0 = *(cntrl + FBEN0);
		 old_fben1 = *(cntrl + FBEN1);
	         *(cntrl + FBEN0) = 0x00;
		 *(cntrl + FBEN1) = 0x04;
		 *(cntrl + TCWEN1) = 0x04;
		 *((reg *)(cntrl + WRR1)) = 0x0003;
		 if (overlay_enable)
		     *(cntrl + PNCNTRL) = 0x18 | plane;
		 else
		     *(cntrl + PNCNTRL) = 0x10 | plane;
		 screen_x = 2048;
		 break;
	case 6 : old_fben0 = *(cntrl + FBEN0);
	         *(cntrl + FBEN0) = 0x20;
	         *((reg *)(cntrl + STATUS)) = 0x0080;   /* write to phantom */
		 *(cntrl + TCWEN0) = 0x20;
		 *((reg *)(cntrl + WRR0)) = 0x0003;
		 if (overlay_enable)
		     *(cntrl + PNCNTRL) = 0x14 | plane;
		 else
		     *(cntrl + PNCNTRL) = 0x10 | plane;
		 screen_x = 1024;
		 break;
	case 1 :
	default: old_fben0 = *(cntrl + FBEN0);
	         *(cntrl + FBEN0) = 0x02;
		 *(cntrl + TCWEN0) = 0x02;
		 *((reg *)(cntrl + WRR0)) = 0x0003;
		 *(cntrl + PNCNTRL) = 0x10;
		 screen_x = 2048;
                 break;
    }

    /* perform the block move */
    *((reg *)(cntrl + COMMAND)) = 0x0090;
    *((reg *)(cntrl + XSRC)) = sx;
    *((reg *)(cntrl + YSRC)) = sy;
    *((reg *)(cntrl + XDST)) = 0;
    *((reg *)(cntrl + YDST)) = 0;
    *((reg *)(cntrl + FX)) = w;
    *((reg *)(cntrl + TFY)) = h;


    /* STEP 2: Transfer aligned data to user's array */
    WAIT_FOR_RUG;

    /* set up BARC for packed pixel read from scratch plane */
    switch (cats_nplanes) {
	case 8 : *(cntrl + ACNTRL) = 0x03;
		 *(cntrl + PNCNTRL) = 0x0a;
		 break;
	case 6 : *(cntrl + ACNTRL) = 0x01;
		 *(cntrl + PNCNTRL) = 0x05;
		 *((reg *)(cntrl + STATUS)) = 0x0040;  /* read from phantom */
		 break;
	case 1 :
	default: *(cntrl + ACNTRL) = 0x01;
		 *(cntrl + PNCNTRL) = 0x01;
		 break;
    }

    /* set up pointers */
    dptr = data;              /* pointer to data array */
    ptr = frm_bfr;            /* pointer to current location in frame buffer */
    if (w & 0xf)              /* pointer to end of current scan line in fb */
	realw = (w & 0xfff0) + 0x10;  /* extra after last full word */
    else
	realw = (w & 0xfff0);         /* whole number of words */
    leptr = ptr + realw;

    /* read data into array */
    for (i=0; i<h; i++) {
	for ( ; ptr < leptr; ptr += 0x10) 
	    *dptr++ = *((unsigned short *) ptr);
	leptr += screen_x;
	ptr = leptr - realw;
    }

    /* STEP 3: Reset for normal operation */
    *(cntrl + VB) = 0x01;
    *(cntrl + PNCNTRL) = 0x00;
    *(cntrl + ACNTRL) = 0x00;
    switch (cats_nplanes) {
	case 8 : *((reg *)(cntrl + WRR1)) = 0x0000;   /* reset scratch WRR */
	         if (overlay_enable) {
		     *(cntrl + FBEN1) = old_fben1;
		     *(cntrl + TCWEN1) = 0x03;
		 }
                 else {
		     *(cntrl + FBEN1) = 0x00;
		     *(cntrl + FBEN0) = old_fben0;
		     *(cntrl + TCWEN1) = 0x00;
		 }
		 break;
	case 6 : *(cntrl + FBEN0) = old_fben0;
		 *(cntrl + TCWEN0) = 0x3f;
		 *((reg *)(cntrl + WRR0)) = cats_draw_mode;
		 *((reg *)(cntrl + STATUS)) = 0x0000;
		 break;
	case 1 :
	default: *(cntrl + FBEN0) = old_fben0;
		 *((reg *)(cntrl + WRR0)) = 0x0000;   /* reset scratch WRR */
		 *(cntrl + TCWEN0) = 0x01;
		 break;
    }
}



WriteBlock(sx, sy, dx, dy, data)
int sx, sy;
int dx, dy;
unsigned char *data;
{
    register int i, deltax, screen_x;
    int no_need_to_shift, do_first, do_end;
    register unsigned char *ptr, *leptr, *dptr;

    /* set up initial pointers */
    no_need_to_shift = TRUE;
    WAIT_FOR_RUG;
    if (cats_nplanes == 6) {
        screen_x = 1024;
	*(cntrl + ACNTRL) = 0x00;
	if (overlay_enable)
	    no_need_to_shift = FALSE;
    }
    else {
        screen_x = 2048;
	if ((overlay_enable) && (cats_nplanes == 8))
	    *(cntrl + ACNTRL) = 0x02;
        else
	    *(cntrl + ACNTRL) = 0x00;
    }

    do_first = sx & 0x1;
    do_end = (sx + dx) & 0x1;
    dptr = data;
    deltax = dx - 1 - do_end;              /* # bytes to leptr */
    ptr = frm_bfr + (sy * screen_x) + sx;
    leptr = ptr + deltax;                  /* leptr points to last byte */
					   /* in last full word         */

    /* place data into frame buffer */
    CHECK_AND_WAIT_FOR_RETRACE;

    /* NOTE: this could be optimized further by doing 32 bit transfers */
    /*       (typecast to int) but you would have to one of 4 cases at */
    /*       the end of each scan line. (0, 1, 2, or 3 extra bytes)    */
    i = dy + 1;
    if (no_need_to_shift) {
        while (--i) {
	    if (do_first)             /* do first byte if not aligned */
		*ptr++ = *dptr++;

            while (ptr < leptr)       /* do all middle words */
		*((reg *) ptr)++ = *((reg *) dptr)++;

            if (do_end)               /* do last byte if not aligned */
		*(leptr + 1) = *dptr++;

            leptr += screen_x;
	    ptr = leptr - deltax;
        }
    }
    else {
        while (--i) {
	    if (do_first)             /* do first byte if not aligned */
		*ptr++ = *dptr++ << 4;

            while (ptr < leptr)       /* do all middle words */
		*((reg *) ptr)++ = *((reg *) dptr)++ << 4;

            if (do_end)               /* do last byte if not aligned */
		*(leptr + 1) = *dptr++ << 4;

            leptr += screen_x;
	    ptr = leptr - deltax;
        }
    }

    /* reset ACNTRL to default */
    *(cntrl + ACNTRL) = 0x00;

}



PackedWriteBlock(sx, sy, w, h, plane_mask, data)
int sx, sy, w, h;
unsigned char plane_mask;
unsigned short *data;
{

    register int i, screen_x, realw;
    register unsigned char *ptr, *leptr;
    register unsigned short *dptr;
    int old_fben0, old_fben1;                /* for double buffering */

    WAIT_FOR_RUG;

    /* STEP 1: Write the user's data into the scratch plane unaligned */
    /* NOTE: This routine could be further optimized by checking for the  */
    /*       case where the data is aligned and could be written directly */
    /*       to the frame buffer instead of being blitted.                */

    /* set up BARC(s) for packed pixel writes */
    switch (cats_nplanes) {
	case 8 : screen_x = 2048;
		 *(cntrl + ACNTRL) = 0x03;
		 old_fben0 = *(cntrl + FBEN0);
		 old_fben1 = *(cntrl + FBEN1);
		 *(cntrl + FBEN0) = 0x00;
		 *(cntrl + FBEN1) = 0x04;
		 *(cntrl + TCWEN1) = 0x04;
		 *(cntrl + PRR1) = 0x03;
		 break;
	case 6 : screen_x = 1024;
		 *(cntrl + ACNTRL) = 0x01;
		 old_fben0 = *(cntrl + FBEN0);
		 *(cntrl + FBEN0) = 0x20;
		 *((reg *)(cntrl + STATUS)) = 0x0080;   /* write scratch */
		 *(cntrl + TCWEN0) = 0x20;
		 *(cntrl + PRR0) = 0x03;
		 break;
	case 1 :
	default: screen_x = 2048;
		 *(cntrl + ACNTRL) = 0x01;
		 old_fben0 = *(cntrl + FBEN0);
		 *(cntrl + FBEN0) = 0x02;
		 *(cntrl + TCWEN0) = 0x02;
		 *(cntrl + PRR0) = 0x03;
		 break;
    }
    /* set up pointers */
    dptr = data;              /* pointer to data array */
    ptr = frm_bfr;            /* pointer to current location in frame buffer */
    if (w & 0xf)              /* pointer to end of current scan line in fb */
	realw = (w & 0xfff0) + 0x10;  /* extra after last full word */
    else
	realw = (w & 0xfff0);         /* whole number of words */
    leptr = ptr + realw;

    /* write data from array into scratch plane */
    for (i=0; i<h; i++) {
	for ( ; ptr < leptr; ptr += 0x10) 
	    *((unsigned short *) ptr) = *dptr++;
	leptr += screen_x;
	ptr = leptr - realw;
    }


    /* STEP 2: Blit the data to the correct plane in the correct place */
    /* enable BARC(s) for writing */
    *(cntrl + VB) = 0x00;
    switch (cats_nplanes) {
	case 8 : *(cntrl + PNCNTRL) = 0x1a;  /* scratch is source */
		 *(cntrl + PRR1) = 0x00;     /* reset scratch PRR */
		 if (overlay_enable) {
	             *(cntrl + FBEN1) = plane_mask & 0x03;
		     *(cntrl + TCWEN1) = 0x03;  /* reset TCWEN */
		 }
		 else {
	             *(cntrl + FBEN0) = plane_mask & 0xff;
		     *(cntrl + TCWEN1) = 0x00;  /* reset TCWEN */
		 }
		 break;
	case 6 : *(cntrl + PNCNTRL) = 0x15;
		 *(cntrl + PRR0) = cats_draw_mode;
	         *((reg *)(cntrl + STATUS)) = 0x0400;  /* read scratch */
		 *(cntrl + TCWEN0) = 0x3f;
		 if (!overlay_select) {
		     *(cntrl + FBEN0) = plane_mask & 0x3f;
		 }
		 else if (overlay_enable) {
		     *(cntrl + FBEN0) = (plane_mask << 4) & 0x30;
		 }
		 else {
		     *(cntrl + FBEN0) = plane_mask & 0x0f;
		 }
		 break;
	case 1 :
	default: *(cntrl + PNCNTRL) = 0x11;
		 *(cntrl + PRR0) = 0x00;    /* reset scratch PRR */
	         *(cntrl + FBEN0) = plane_mask & 0x01;
		 *(cntrl + TCWEN0) = 0x01;
		 break;
    }

    /* blit the data */
    *((reg *)(cntrl + COMMAND)) = 0x0090;
    *((reg *)(cntrl + XSRC)) = 0;
    *((reg *)(cntrl + YSRC)) = 0;
    *((reg *)(cntrl + XDST)) = sx;
    *((reg *)(cntrl + YDST)) = sy;
    *((reg *)(cntrl + FX)) = w;
    CHECK_AND_WAIT_FOR_RETRACE;
    *((reg *)(cntrl + TFY)) = h;


   /* STEP 3: Reset for normal operation */
   WAIT_FOR_RUG;
   *(cntrl + VB) = 0x01;
   *(cntrl + PNCNTRL) = 0x00;
   *(cntrl + ACNTRL) = 0x00;
   switch (cats_nplanes) {
       case 8 : *(cntrl + FBEN1) = old_fben1;
		*(cntrl + FBEN0) = old_fben0;
		break;
       case 6 : *(cntrl + FBEN0) = old_fben0;
		*((reg *)(cntrl + STATUS)) = 0x0000;
		break;
       case 1 :
       default: *(cntrl + FBEN0) = old_fben0;
		break;
   }

}




int PixReplicate(sx, sy, dx, dy, delta, mag)
int sx, sy, dx, dy, delta, mag;
{
    int wx, wy;

    /* the final picture cannot be larger than 256x256 pixels */
    /* magnification must be 2, 4, 8 or 16                    */
    switch (mag) {
        case 2 : if (delta > 128) return(-1);
                 break;
        case 4 : if (delta > 64) return(-1);
                 break;
        case 8 : if (delta > 32) return(-1);
                 break;
        case 16: if (delta > 16) return(-1);
                 break;
        default : return(-1);
    }

    if (cats_nplanes == 6) {
        wx = 768; 
	wy = 896;
    } 
    else {
        wx = 1792; 
	wy = 896;
    } 

    raw_pix_rep(sx, sy, dx, dy, delta, mag, wx, wy);
}



raw_pix_rep(sx, sy, dx, dy, delta, mag, workx, worky)
int sx, sy, dx, dy, delta, mag, workx, worky;
{
    register int i, j;
    register int split_width_factor;
    register int num_splits;
    register int shift_index;
    int wx, wy;
    int base_plane, end_plane;
    reg old_ltp;

    static reg rep_patterns[4][16] = {
        0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, /* first shift */
        0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff, 0x00ff,

        0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, /* second shift */
        0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f, 0x0f0f,

        0x3333, 0x3333, 0x3333, 0x3333, 0x3333, 0x3333, 0x3333, 0x3333, /* third shift */
        0x3333, 0x3333, 0x3333, 0x3333, 0x3333, 0x3333, 0x3333, 0x3333,

        0x5555, 0x5555, 0x5555, 0x5555, 0x5555, 0x5555, 0x5555, 0x5555, /* fourth shift */
        0x5555, 0x5555, 0x5555, 0x5555, 0x5555, 0x5555, 0x5555, 0x5555};

    static int shift_offset[16] = {
	4, 2, 1, 1,                        /* for mag = 2 */
	6, 3, 2, 1,                        /* for mag = 4 */
	7, 4, 2, 1,                        /* for mag = 8 */
	8, 4, 2, 1 };                      /* for mag = 16 */


    switch (mag) {
        case 2 : split_width_factor = 3; 
		 num_splits = (delta >> 3) + ((delta & 0x07) != 0); 
		 shift_index = 0;
                 break;
        case 4 : split_width_factor = 2;
		 num_splits = (delta >> 2) + ((delta & 0x03) != 0);
		 shift_index = 4;
                 break;
        case 8 : split_width_factor = 1;
		 num_splits = (delta >> 1) + ((delta & 0x01) != 0);
		 shift_index = 8;
                 break;
        case 16: split_width_factor = 0;
		 num_splits = delta;
		 shift_index = 12;
                 break;
        default: return;
    }


    /* set up workspace */
    wx = workx;
    wy = worky;

    /* set up for blits */
    WAIT_FOR_RUG;
    *(cntrl + VB) = 0x00;         /* set BARC for blits */
    if ((overlay_enable) && (cats_nplanes == 8))
        *((reg *)(cntrl + WRR1)) = 0x0003;         /* copy in source mode */
    else
        *((reg *)(cntrl + WRR0)) = 0x0003;         /* copy in source mode */
    *((reg *)(cntrl + COMMAND)) = 0x0090;

    /* set up clip limits around work space */
    *((reg *)(cntrl + CXMIN)) = wx;
    *((reg *)(cntrl + CYMIN)) = wy;
    *((reg *)(cntrl + CXMAX)) = wx + (delta * mag) - 1;
    *((reg *)(cntrl + CYMAX)) = wy + delta;
    if (!cats_clipping_indicator)
	set_clipping(TRUE);

    /* STEP 1: spread source with initial blits */
    *((reg *)(cntrl + YSRC)) = sy;
    *((reg *)(cntrl + YDST)) = wy;
    *((reg *)(cntrl + FX)) = 1 << split_width_factor;
    *((reg *)(cntrl + FY)) = delta;
    for (i=0; i<num_splits; i++) {
        WAIT_FOR_RUG;
        *((reg *)(cntrl + XSRC)) = sx + (i << split_width_factor);
        *((reg *)(cntrl + TXDST)) = wx + (i << 4);
    }

    /* STEP 2: do binary horizontal pixel replication */
    WAIT_FOR_RUG;

    /* set up BARC for correct masked blits */
    *(cntrl + TCNTRL) = 0x01;       /* Use three op and pattern */
    if ((overlay_enable) && (cats_nplanes == 8))
        *(cntrl + TRR1) = 0xca; 
    else
        *(cntrl + TRR0) = 0xca; 

    /* set up RUG */
    *((reg *)(cntrl + XSRC)) = wx;
    *((reg *)(cntrl + YSRC)) = wy;
    *((reg *)(cntrl + YDST)) = wy;
    *((reg *)(cntrl + FX)) = delta * mag;
    *((reg *)(cntrl + FY)) = delta;

    if (!overlay_select) 
	end_plane = cats_nplanes;
    else if (!overlay_enable)
	end_plane = (cats_nplanes == 6) ? 4 : cats_nplanes;
    else
	end_plane = 2;

    for (i=0; i<4; i++) {
        WAIT_FOR_RUG;

        /* change pattern ram in planes that are enabled */
        for (j=0; j<end_plane; j++) 
            if (cats_write_mask & (0x01 << j))
                LoadPattern(j,&rep_patterns[i][0]);
        
        /* blit workspace on top of itself shifted over correct shift_offset to replicate */
        *((reg *)(cntrl + TXDST)) = wx + shift_offset[shift_index + i];
    }

    /* STEP 3: do final vertical pixel replication */
    WAIT_FOR_RUG;

    /* restore original clipping */
    set_clipping(cats_clipping_indicator);
    *((reg *)(cntrl + CXMIN)) = cats_clip_xmin;
    *((reg *)(cntrl + CYMIN)) = cats_clip_ymin;
    *((reg *)(cntrl + CXMAX)) = cats_clip_xmax;
    *((reg *)(cntrl + CYMAX)) = cats_clip_ymax;

    *(cntrl + TCNTRL) = 0x00;           /* user rr */
    if ((overlay_enable) && (cats_nplanes == 8))
        *((reg *)(cntrl + WRR1)) = cats_draw_mode;
    else
        *((reg *)(cntrl + WRR0)) = cats_draw_mode;

    *((reg *)(cntrl + COMMAND)) = 0x0084 | cats_rug_enable;  /* vertical rep */
    old_ltp = *((reg *)(cntrl + LTP));
    *((reg *)(cntrl + LTP)) = 0x0110 * (0x10 - mag);
    *((reg *)(cntrl + XDST)) = dx;
    *((reg *)(cntrl + FY)) = delta * mag;
    CHECK_AND_WAIT_FOR_RETRACE;
    *((reg *)(cntrl + TYDST)) = dy;


    /* reset - everything except pattern rams */
    WAIT_FOR_RUG;
    *(cntrl + VB) = 0x01;
    if ((overlay_enable) && (cats_nplanes == 8))
        *(cntrl + TRR1) = fill_three_op;
    else
        *(cntrl + TRR0) = fill_three_op;
    *((reg *)(cntrl + LTP)) = old_ltp;
}



SerialMode(mode, radius)
int mode;
int radius;
{

    /* set up RUG for following operations to be done from serial input */
    /* the next operation that uses RUG will erase this mode */

    WAIT_FOR_RUG;
    if (mode == VECTOR)
	*((reg *)(cntrl + COMMAND)) = cats_rug_enable | cats_linetype_enable | 0x2;

    else if (mode == HCIRCLE) {
	*((reg *)(cntrl + COMMAND)) = cats_rug_enable | cats_linetype_enable | 0x22;
	*((reg *)(cntrl + FY)) = radius;
    }

    else if (mode == FCIRCLE) {
	*((reg *)(cntrl + COMMAND)) = cats_rug_enable | 0xe2;
	*((reg *)(cntrl + FY)) = radius;
    }

}
