/*==============================================================*
 *   Version 1.16       YPHD.C                                  *
 *                      Physical interface for the PC/AT hard   *
 *                      disk driver.                            *
 *--------------------------------------------------------------*
 *    VERSION   DATE    BY      CHANGE/COMMENTS                 *
 *--------------------------------------------------------------*
 *   1.01     8-APR-85  pjp    bug in fgap would'nt allow format*
 *                                                              *
 *   1.02    24-APR-85  pjp    isr to assembler                 *
 *                                                              *
 *   1.03     1-MAY-85  pjp    check DKASR for error returns    *
 *                                                              *
 *   1.04    18-JUN-85  pjp    longer timeout for recal         *
 *                                                              *
 *   1.05    02-JUL-85  pjp    don't mask master pic in un-init *
 *                                                              *
 *   1.06    06-OCT-85  reb    eliminated globi/hdglut          *
 *                                                              *
 *   1.07    851111     reb    added phd_startio module and     *
 *                             synching on the controller       *
 *                             to prevent overlap with data ops.*
 *   1.08    860204     reb    added error check to asr         *
 *                             invocations to avoid lockup on   *
 *                             possible undetected controller   *
 *                             errors and timeouts.             *
 *   1.09    860204     reb    added or in of step rate to seek *
 *   1.10    08/13/86   mei    Metaware port. OR'd disk error   *
 * 			       codes with ED_DISK.		*  	 
 *   1.11    08/22/86   ldt    Cast function * on comparisons   *
 *                             for Lattice.                     *
 *   1.12     2/12/87   KPB    Moved head check out of ads loop *
 *   1.13    870317     reb    Added seek timeouts per WJ       *
 *   1.14     4/10/87   KPB    Added device sync                *
 *   1.15    7/17/87    KPB    Added Compact Support and seek   *
 *                             fix                              *
 *   1.16    7/17/87    KPB    Slowed seeks down for Compact    *
 *			       support.				*
 *==============================================================*
 *  INCLUDES:                                                   */
/* yphd.c tasklist:
   find a way to prevent the occurence of interrupts during the setup
of the controller for the next operation.

   Insert timer timeouts for the detection of a locked up controller.

   Convert the current algorithm to use asrwait and nextasr rather than the 
doasr spin loops that it is currently using.

   possibly nextasr if seek issued and doasr if already on track.
   what happens if evnum is zero when nextasr is called.

*/

#include    "portab.h"
#include    "system.h"
#include    "yio.h"
#include    "evb.h"
#include    "ydksync.h"
#include    "ydk.h"
#include    "yhd.h"
#include    "ymd.h"


/*
 *  local routines
 */


VOID    phd_init() ;
VOID    phd_uninit() ;
ASR     phd_recal() ;
ERROR   phd_select() ;
ASR     phd_format() ;
ASR     phd_io() ;
VOID    phd_io1() ;
ASR     phd_startio() ;
ASR     headasr() ;
ASR     formatasr() ;
ASR     readasr() ;
ASR     writeasr() ;
ASR     verifyasr() ;

VOID    LiersBsnbi( );
VOID    BuildLiersFormat();
VOID    phd_terminate() ;
VOID    phd_iocompl() ;

ISR     dummyisr() ;
ISR     readisr() ;
ISR     writeisr() ;
ISR     verifyisr() ;
ISR     formatisr() ;
ISR     headisr() ;
ISR     seekisr() ;

ERROR   phd_setparm() ;

VOID    phd_bldtf() ;
VOID    phd_bsnbi() ;
VOID    hd_settimer() ;
VOID    hd_cleartimer() ;
ASR     hd_timex() ;

BYTE    bpsx() ;
BYTE    fgap() ;

BOOLEAN outputcmd() ;
BOOLEAN outputbyte() ;
BOOLEAN cmdquery() ;
BYTE    hcmdcode() ;
ERROR   chk_stat() ;

/*
 *  externals
 */
EXTERN  SPB    *DriveSync[2];
EXTERN  ISR     phd_int();                          /* assembler portion of isr */
EXTERN  LONG    OUTPSTR() ;                         /* out word string , bump ptr. */
EXTERN  LONG    INPSTR() ;                          /* in word string , bump ptr. */
EXTERN  VOID    HDPERF() ;                          /* performance hook */
EXTERN  VOID    HDERR() ;                           /* error hook */
EXTERN  VOID	OUTP();
EXTERN	BYTE	INP();
EXTERN  VOID    yGetHscAddr();
EXTERN  VOID    hdunsync();
/*
 *  code macros
 */

#define HDCBUSY         (INP( HDCSTAT ) & HDC_BSY)


/************************************************************************
*
*               Global Data
*
************************************************************************/

GLOBAL   EMASK  timemask[DKMAXUNITS] ;
GLOBAL   WORD   timeouts[HDMAXUNITS] ;		/* 1.13 */
#define tortry  10				/* 1.13 */

/***************************************************
* HTF -
*   array of task files, one for each phys unit.
*/

HDTF    HTF[ DKMAXUNITS ] ;

/*************************************************************
* P2LM
* Pysical to Logical mapping array
*/

P2L    P2LM[ HDMAXUNITS ] ;

/**********************************************************
* HDTIMPB - parm block for watch dog timer call
*
*/

DKTIMPB HDTIMPB =
{
    1,                                              /* async */
    0,                                              /* reserved */
    0,                                              /* relative time */
    0L,                                             /* swi address */
    0L                                              /* time in milliseconds */
} ;


/************************************************************************
 *                                                                      *
 *          Hard Disk Driver Low Level Interace                         *
 *                                                                      *
 *  code which interfaces the high level (logical drivers) to the low   *
 *  level routines and their primitives.  These routines run as         *
 *  synchronous routines, but should probably be called from async code,*
 *  thus making them async routines (get it?).  They return...          *
 *                                                                      *
 *  Note that these are primitive (not smart) routines!  They assume    *
 *  that the caller knows what it is doing -- e.g., the caller won't    *
 *  try to perform an operation on a drive (physical unit) that is busy *
 *                                                                      *
 ************************************************************************/


/************************************************************************
*  phd_init - hardware setup
*
*/

VOID    phd_init( u )
LUTE    *u ;
{
    int n,i ;
    /* setup global isr address */

    for (n = 0; n < HDMAXUNITS; n++) {
        P2LM[n].seqisr = dummyisr ;
        P2LM[n].curcyl = -1 ;
	timeouts[n] = 0 ;		/* 1.13 */
    }

        /* Setup the interrupt vector for the hard disk */

        SETVEC( phd_int, HDINTNO ) ;

        /* Setup the PIC for Hard Disk Interrupts */

        OUTP( PIC1MSK , INP(PIC1MSK) & 0xbf ) ;     /* init slave  */
        OUTP( PIC0MSK , INP(PIC0MSK) & 0xfb ) ;     /* init master */

        /* reset the controller */

        OUTP( HDCREG , HDC_RESET ) ;

        /* wait 4.8 us  */
        for (i=0;i<10;i++) ;

        OUTP( HDCREG , HDC_H8 ) ;                   /* 8 head option as default */

        for (i=0; i < 10 ; i++ )  {
            if (cmdquery())  break ;
            HDTIMPB.tim_sync = 0 ;
            HDTIMPB.tim_time = 100L ;
            SUPIF ( F_TIMER, &HDTIMPB ) ;
            HDTIMPB.tim_sync = 1 ;
        }
}


/**************************************************************************
*  phd_uninit - hardware un-setup
*
*/

VOID    phd_uninit( u )
LUTE    *u ;
{
    int n ;

    /* Setup Global isr address */

    for (n = 0; n < HDMAXUNITS; n++) P2LM[n].seqisr = dummyisr ;

        /* Unset the PIC for Hard Disk Interrupts */

        OUTP( PIC1MSK , INP(PIC1MSK) | ~0xbf ) ;    /* uninit slave */

        /* Unset the interrupt vector for the hard disk */

        SETVEC( 0L, HDINTNO ) ;
}


/**************************************************************************
*  phd_recal -
*   recal (restore) to track zero.
*/
ASR phd_recal( u )
LUTE    *u ;
{
    HDTF    *f ;
    LDD     *l ;
    BYTE    driveno ;
    IORB    *i ;

    driveno = u->lu_driveno ;
    l = u->lu_lddp ;

    P2LM[driveno].seqisr = headisr ;     /* seek or recal isr    */
    i = &u->lu_iorb;
    P2LM[driveno].physlute = u ;                    /* save ptr to global lute */

    f = &HTF[u->lu_unitno] ;                        /*  setup task file ptr */
    f->hd_cmd[6] = HDCRECAL | ( u->lu_pudp->pu_step & 0xf ) ;
                                                    /*  disk op code    */

    f->hd_cmd[5] =
    HDCECC | ( bpsx(l->ld_mdb.md_secsiz) << 5 ) | ( driveno << 4 ) ;

    if( ! outputcmd( f )  ) {                       /*  output command  */
        phd_terminate( u , (ED_DISK | E_DKATTACH) ) ;
    }
    else {
        hd_settimer( 20000L, i ) ;                  /* watchdog timer   */
    }
    return(0);
}


/**************************************************************************
*  phd_select -
*   The upper level routines call select to establish a connection
*   between the upper and lower level routines.
*
*/

ERROR   phd_select( u )
LUTE    *u ;
{
/*    P2LM[u->lu_driveno].seqisr = &dummyisr ;*/ /* 851112 reb */
    return(E_SUCCESS) ;                /*851111 reb ( phd_setparm( u )) ;*/
}

/**************************************************************************
 *  phd_format -
 *  format a track.
 */

ASR phd_format( u )
LUTE    *u ;
{
    HDTF    *f ;                                    /* task file pointer */
    LDD     *l ;                                    /* logical drive discriptor */
    IORB    *i ;

    i = &u->lu_iorb ;                               /* pointer to io request block */
    P2LM[u->lu_driveno].physlute = u ;              /* global pointer to u      */
    P2LM[u->lu_driveno].seqisr = formatisr ;       /* isr to be run @ interrrupt time */

    l = u->lu_lddp ;


#if COMPACT_DISK
/* if there are 16 heads than this is a lie */
/* there are only 8 heads so we must make a new task file */\

 if(l->ld_mdb.md_nheads == 16)
  {
   if(u->lu_iorb.io_dkaddr.hsc_head & 1)
    {
     phd_terminate(u,0L);
     return;
    }
   else
    {
    i->io_buffer = SALLOC( (LONG) l->ld_mdb.md_secsiz );
     LiersBsnbi(u);
     f = &HTF[ u->lu_unitno ] ;
     BuildLiersFormat(f,u);
     
    }
   }
  else
   {
    i->io_buffer = SALLOC( (LONG) l->ld_mdb.md_secsiz );
    phd_bsnbi( u ) ;                                /* sector #/block indicator list */

    f = &HTF[ u->lu_unitno ] ;
    phd_bldtf( f , u ) ;
                      /* task file as if for rd/write */
    }

#else

    i->io_buffer = SALLOC( (LONG) l->ld_mdb.md_secsiz );
    phd_bsnbi( u ) ;                                /* sector #/block indicator list */

    f = &HTF[ u->lu_unitno ] ;
    phd_bldtf( f , u ) ;                            /* task file as if for rd/write */
#endif

    f->hd_cmd[2] = fgap( u ) ;                      /* format specific gap to task file */

        OUTPSTR( HDCDATA , i->io_buffer , l->ld_mdb.md_secsiz ) ;
    if(  ! outputcmd( f ) ) {                       /* output command   */
        SFREE(i->io_buffer) ;
        phd_terminate( u , (ED_DISK | E_DKATTACH) ) ;
        return(0);
    }
   
        OUTPSTR( HDCDATA , i->io_buffer , l->ld_mdb.md_secsiz ) ;
        hd_settimer( 50000L,i ) ;
   
    return(0);
}


/*********************************************************************
*  phd_io -
*   starts the i/o request, is re-entered from phd_io1()
*   if there is any more left to do,
*/

ASR phd_io( u )
LUTE    *u ;
{
    HDTF    *f ;
    LDD     *l ;                                    /* logical drive discriptor  */
    IORB    *i ;
    WORD    driveno,y ;
    ERROR   r ;

    driveno = u->lu_driveno ;
    P2LM[driveno].physlute = u ;              /* LUTE pointer for physical unit */

    i = &u->lu_iorb ;                               /* global ptr to iorb  */

    l = u->lu_lddp ;

    yGetHscAddr( &i->io_dkaddr , i->io_stssn , l ) ;

    /*  nsecs, or number of sectors the controller can handle.  */

    i->io_nsecs = MIN( HDMAXSECS , i->io_totnsecs ) ;

    /* Key change area build task file only if physical unit is free */

    y = u->lu_driveno ? 0 : 1 ;      /* get index for the other unit */

    if (i->io_error) return(0);     /* parachute clause timeouts etc */

    /* syncing on seeks only */
    if ((P2LM[y].seqisr == (ISR (*)() )seekisr) || 
	(P2LM[y].seqisr == (ISR (*)() )dummyisr)) { 
			/* check the other phys drive for in read/write */

        f = &HTF[u->lu_unitno] ;
        phd_bldtf( f , u ) ;                            /*  ;build task file */
        if ( P2LM[driveno].curcyl != i->io_dkaddr.hsc_cyl )  {

            f->hd_cmd[6] = (HDCSEEK|u->lu_pudp->pu_step) ;

            P2LM[driveno].seqisr = seekisr ;

            if(  ! outputcmd( f )  ) {          /*  output command  */
                P2LM[driveno].seqisr = dummyisr ;
                if(( r = DKASR( phd_io , u , 0L )) < 0L)  { /*851111 reb*/
                    phd_terminate( u , r ) ;
                }
                return(0) ;
            }
	    hd_settimer(5000L,i) ;
	    /* P2LM[driveno].curcyl = i->io_dkaddr.hsc_cyl ;*/   /* undo assumption, assume the seek will succeed */
            return(0) ;
        }
        else  {
            /* set discrete isr addr. */
            switch( (BYTE)i->io_op )  {
                case DKREAD:    P2LM[driveno].seqisr = readisr ;
                        break ;
                case DKWRITE:   P2LM[driveno].seqisr = writeisr ;
                        break ;
                case DKVERIFY:  P2LM[driveno].seqisr = verifyisr ;
                        break ;
            }
            if(( r = DKASR( phd_startio , u , 0L )) < 0L)  {
                phd_terminate( u , r ) ;
            }
            return (0);
        }
    }
    else  {
        if(( r = DKASR( phd_io , u , 0L )) < 0L)  { /*851112 reb*/
            phd_terminate( u , r ) ;
        }
        return(0) ;
    }
}

/*********************************************************************
*  phd_io1 -
*   gets execution after the interrupt from the actual i/o.
*   determines if more to be done.
*/

VOID    phd_io1(i)
IORB    *i ;
{
    LUTE    *u ;

    u = (LUTE *)i ;
    i->io_totnsecs -= i->io_nsecs ;
    i->io_stssn += i->io_nsecs ;

    if( i->io_error )  {
        phd_iocompl(u) ;
    }
    else {
        if( i->io_totnsecs )  {
            phd_io(u) ;
        }
        else  {
            if( (BYTE)i->io_op == DKWRITE )  {
                if( u->lu_iflag & 0x0200 )  {
                    i->io_op = DKVERIFY ;
                    i->io_totnsecs = i->io_savnsecs ;
                    i->io_stssn = i->io_savssn ;
                    phd_io( u ) ;
                    return ;
                }
            }
            phd_iocompl(u) ;
        }
    }
}


/*********************************************************************
* phd_startio -
* If the controller is ready to accept read/write/verify then output the
* command else spin off new asr to retry.
*/
ASR phd_startio(u)
LUTE    *u ;
{
    WORD    y ;
    HDTF    *f ;
    IORB    *i ;
    LDD     *l ;
    ERROR   r ;

    i = (IORB *)u ;
    f = &HTF[u->lu_unitno] ;
    l = u->lu_lddp ;

    y = u->lu_driveno ? 0 : 1 ;             /* get index for the other unit */
    if ((P2LM[y].seqisr == (ISR (*)() )dummyisr)||
        (P2LM[y].seqisr == (ISR (*)() )headisr) ||
        (P2LM[y].seqisr == (ISR (*)() )seekisr))  {   /* check the other phys drive for in read/write */

        if (i->io_error) return(0) ;

        if(  ! outputcmd( f )  ) {          /*  output command  */
            if(( r = DKASR( phd_startio , u , 0L )) < 0L)  {
                phd_terminate( u , r ) ;
            }
            return(0) ;
        }
        else {
            if( (BYTE)i->io_op == DKWRITE )  {          /* have to wet nurse 1st one */
                if( u->lu_flags & DKF_UADDR ) MAPU( u->lu_b_pdaddr ) ;
                i->io_buffer = OUTPSTR( HDCDATA , i->io_buffer , l->ld_mdb.md_secsiz ) ;
                if( u->lu_flags & DKF_UADDR ) UNMAPU() ;
            }
            HDPERF( i , f ) ;               /* performance hook */
        }
    }
    else  {
        if(( r = DKASR( phd_startio , u , 0L )) < 0L)  {
            phd_terminate( u , r ) ;
        }
        return (0);
    }
}



/*********************************************************************
*  phd_terminate -
*   terminate the i/o request.
*/

VOID    phd_terminate( u , r )
LUTE    *u;
ERROR   r ;
{
    u->lu_iorb.io_error = r ;
    phd_iocompl( u ) ;
}

/********************************************************************
 * phd_iocompl -
 *
 *
 */

VOID    phd_iocompl( u )
LUTE    *u ;
{

    P2LM[u->lu_driveno].seqisr = dummyisr ;
    /* error logging hook */
    HDERR( &u->lu_iorb , &HTF[u->lu_unitno] ) ;

    FLAGSET( u->lu_ioflagno , u->lu_pdaddr , u->lu_iorb.io_error ) ;
   hdunsync(DriveSync[u->lu_driveno]);
    /* reset global addresses  */
}


/*********************************************************************
* dummyisr - return false (no dispatch)
*/
ISR dummyisr()
{
    return(FALSE) ;
}


/*********************************************************************
*  phd_bldtf -
*   build the task file and do some hardware manupulation
*/

VOID    phd_bldtf( hdtfp , u )
HDTF    *hdtfp ;
LUTE    *u ;
{

    IORB    *i ;                                    /*  i/o request block pointer */
    BYTE    ss ;                                    /*  temp for sector size indicator */
    BYTE    cmd ;                                   /* controller specific cmd */
    BYTE    *tfptr ;                                /* pointer to current task file element */

    i = &u->lu_iorb ;                               /* i/o request block */
    tfptr = &hdtfp->hd_cmd[6] ;

    /* set hardware latch for 8 or 16 head hard disk option */
   
    if( u->lu_pudp->pu_nheads > 8 )  {
        OUTP(HDCREG,HDC_H16) ;
    }
    else  {
        OUTP(HDCREG,HDC_H8) ;
    }

    /* setup the task file */

    cmd = hcmdcode((BYTE)i->io_op) ;

    if( *tfptr == HDCFORMAT )  {
        if( cmd == HDCREAD )  {                     /* assume same head,cyl */
            cmd = cmd | HDCRETRY ;                  /* no retries post fmt read */
        }
    }
    *tfptr-- = cmd ;                                /*  hdc opcode */

    ss = bpsx( u->lu_lddp->ld_mdb.md_secsiz ) ;
    *tfptr-- = HDCECC | ( ss << 5 ) | (u->lu_driveno << 4) | i->io_dkaddr.hsc_head ;
                        /*  select byte */

    *tfptr-- = (BYTE) ((i->io_dkaddr.hsc_cyl & 0xff00) >> 8 ) ;
    *tfptr-- = (BYTE) (i->io_dkaddr.hsc_cyl & 0xff) ;

    *tfptr-- = i->io_dkaddr.hsc_sector ;
    *tfptr-- = (BYTE) i->io_nsecs ;

    *tfptr = (BYTE) (u->lu_pudp->pu_precomp >> 2) ;

}


/*********************************************************************
*  BuildLiersFormat -
*   build the task file and do some hardware manupulation
*/

VOID    BuildLiersFormat( hdtfp , u )
HDTF    *hdtfp ;
LUTE    *u ;
{

    IORB    *i ;                                    /*  i/o request block pointer */
    BYTE    ss ;                                    /*  temp for sector size indicator */
    BYTE    cmd ;                                   /* controller specific cmd */
    BYTE    *tfptr ;                                /* pointer to current task file element */

    i = &u->lu_iorb ;                               /* i/o request block */
    tfptr = &hdtfp->hd_cmd[6] ;

   
        OUTP(HDCREG,HDC_H8) ;

    /* setup the task file */

    cmd = hcmdcode((BYTE)i->io_op) ;

    if( *tfptr == HDCFORMAT )  {
        if( cmd == HDCREAD )  {                     /* assume same head,cyl */
            cmd = cmd | HDCRETRY ;                  /* no retries post fmt read */
        }
    }
    *tfptr-- = cmd ;                                /*  hdc opcode */

    ss = bpsx( u->lu_lddp->ld_mdb.md_secsiz ) ;
    *tfptr-- = HDCECC | ( ss << 5 ) | (u->lu_driveno << 4) |
            (i->io_dkaddr.hsc_head >> 1) ;
                        /*  select byte */

    *tfptr-- = (BYTE) ((i->io_dkaddr.hsc_cyl & 0xff00) >> 8 ) ;
    *tfptr-- = (BYTE) (i->io_dkaddr.hsc_cyl & 0xff) ;

    *tfptr-- = i->io_dkaddr.hsc_sector ;
    *tfptr-- = (BYTE) (i->io_nsecs << 1);

    *tfptr = (BYTE) (u->lu_pudp->pu_precomp >> 2) ;

}


/**************************************************************************
 *  phd_bsnbi - build sector number/block indicator from either the
 *  starting sector number and the number of sectors/track -or-
 *  the sector list.
 *
 */

VOID    phd_bsnbi( u )
LUTE    *u ;
{
    WORD    *rptr ;                                 /* pointer to sector list or starting sector */
    IORB    *i ;                                    /* pointer to i/o request block */
    BYTE    *b ;                                    /* pointer to our buffer */
    BYTE    sn ;                                    /* incremental sector # */
    BYTE    stsn ;                                  /* starting sector # */
    WORD    nsecs ;                                 /* number of sectors */
    WORD    count ;                                 /* loop count (#sectors) */

    i = &u->lu_iorb ;
    b = (BYTE*) i->io_buffer ;
    count = nsecs = i->io_nsecs ;
    nsecs += stsn = i->io_dkaddr.hsc_sector ;

    if( (BYTE)i->io_op == DKFMT | (BYTE)i->io_op == DKFMTBB ) {  /* no list format */
        for( sn = stsn ; count-- ; )  {
            *b++ = ( ((BYTE)i->io_op == DKFMTBB) ? 0x80 : 0x0 ) ;
            *b++ = (BYTE) sn++ ;
            if( stsn )  {
                if( sn > nsecs )  {
                    sn = stsn ;
                }
            }
            else  {
                if( sn >= nsecs )  {
                    sn = stsn ;
                }
            }
        }
    }
    else  {
        for( rptr = (WORD*)i->io_totnsecs ; count-- ; )  {
            *b++ = ( ((BYTE)i->io_op == DKFMTBBLST) ? 0x80 : 0x0 ) ;
            *b++ = (BYTE) *rptr++ ;
        }
    }
}
/**************************************************************************
 *  LiersBsnbi - build sector number/block indicator from either the
 *  starting sector number and the number of sectors/track -or-
 *  the sector list.
 *
 */

VOID    LiersBsnbi( u )
LUTE    *u ;
{
    WORD    *rptr ;                                 /* pointer to sector list or starting sector */
    IORB    *i ;                                    /* pointer to i/o request block */
    BYTE    *b ;                                    /* pointer to our buffer */
    BYTE    sn ;                                    /* incremental sector # */
    BYTE    stsn ;                                  /* starting sector # */
    WORD    nsecs ;                                 /* number of sectors */
    WORD    count ;                                 /* loop count (#sectors) */
    static WORD    LiersSkew[34] =
            {1,24,13,2,25,14,3,26,15,4,27,16,5,28,17,6,29,18,
             7,30,19,8,31,20,9,32,21,10,33,22,11,34,23,12};
    
    i = &u->lu_iorb ;
    b = (BYTE*) i->io_buffer ;
    count = nsecs = 34;
    nsecs += stsn = i->io_dkaddr.hsc_sector ;

    if( (BYTE)i->io_op == DKFMT | (BYTE)i->io_op == DKFMTBB ) {  /* no list format */
        for( sn = stsn ; count-- ; )  {
            *b++ = ( ((BYTE)i->io_op == DKFMTBB) ? 0x80 : 0x0 ) ;
            *b++ = (BYTE) sn++ ;
            if( stsn )  {
                if( sn > nsecs )  {
                    sn = stsn ;
                }
            }
            else  {
                if( sn >= nsecs )  {
                    sn = stsn ;
                }
            }
        }
    }
    else  {
        for( rptr = &LiersSkew[0] ; count-- ; )  {
            *b++ = ( ((BYTE)i->io_op == DKFMTBBLST) ? 0x80 : 0x0 ) ;
            *b++ = (BYTE) *rptr++ ;
        }
    }
}


/************************************************************************
 *                                                                      *
 *          Hard Disk Driver ISR                                        *
 *                                                                      *
 ************************************************************************/



/*************************************************************************
 * seekasr - gets execution in asr contexrt after a seek command is issued
 * and the seek complete interrupt is received.
 */
ASR seekasr(u)
LUTE	*u ;
{
    LONG 	i ;
    P2LM[u->lu_driveno].curcyl = u->lu_iorb.io_dkaddr.hsc_cyl ;
    hd_cleartimer( u ) ;
    phd_io( u ) ;
    return ( 0 ) ;
}

/***********************************************************************
 * seekisr - gets execution in isr context after a seek command
 * is issued.
 */

ISR seekisr(drno)
WORD    drno ;
{
    LONG    r ;
    LUTE    *u ;
    u = P2LM[drno].physlute ;

    if(( r = DKASR( seekasr , u , 0L )) < 0L)  {/* re queue the request */
        phd_terminate( u , r ) ;                /* error on asr */
    }
    return( TRUE ) ;                            /* dispatch */
}

/***********************************************************************
 * headisr - gets execution in isr context after a restore or seek command
 * is issued.
 */

ISR headisr(drno)
WORD    drno ;
{
    LONG    r ;
    LUTE    *u ;
    u = P2LM[drno].physlute ;

    if(( r = DKASR( headasr , u , 0L )) < 0L)  {
        phd_terminate( u , r ) ;
    }
    return( TRUE ) ;                                /* dispatch */

}

/***********************************************************************
 * headasr - executed after headisr to check and save status in a global
 * location and terminate the req (no system functions in isr context) .
 */

ASR headasr(i)
IORB    *i ;
{
    LUTE    *u ;

    u = (LUTE *)i ;
    if( ! HDCBUSY  )  {
        i->io_error = chk_stat() ;
    }
    else  {
        i->io_error = (ED_DISK | E_DKATTACH) ;
    }
    hd_cleartimer(u) ;
    phd_iocompl( u ) ;
    return(0);
}


/*********************************************************************
 * readisr - gets execution in isr context during a read command. This
 * routine fires off an asr to do the work required.
 */

ISR readisr(drno)
WORD    drno ;
{
    LONG    r ;
    LUTE    *u ;

    u = P2LM[drno].physlute ;

    if(( r = DKASR( readasr , u , 0L )) < 0L)  {
        phd_terminate( u , r ) ;
    }
    return(TRUE) ;
}

/*********************************************************************
 * readasr - checks for command completion and terminates the request
 * if so, or inputs a buffer load of data and sets up for the next
 * interation.
 *
 */

ASR readasr(i)
IORB    *i ;
{
    BYTE    status ;                                /* temp status byte */
    LUTE    *u ;

    u = (LUTE *)i ;

    status = INP( HDCSTAT ) ;

    if( !( status & HDC_BSY ) )  {
        if( (i->io_error = chk_stat()) != NULLPTR ) {
            phd_iocompl( u ) ;
            return(0) ;
        }
    }
    if( status & HDC_BSY || status & HDC_DRQ )  {

        /* if we get here hdc could be busy */
        /* so we can't check status, but    */
        /* is o.k. to input data            */

        if( u->lu_flags & DKF_UADDR ) MAPU( u->lu_b_pdaddr ) ;

        i->io_buffer = INPSTR( HDCDATA , i->io_buffer , u->lu_lddp->ld_mdb.md_secsiz ) ;

        if( u->lu_flags & DKF_UADDR ) UNMAPU() ;

        status = INP( HDCSTAT ) ;
        if( status & HDC_BSY || status & HDC_DRQ )  {
            return(0) ;
        }
    }

    i->io_error  = chk_stat() ;
    phd_io1(i) ;
    return(0) ;        /* dispatch when finished */

}


/********************************************************************
 * writeisr - gets execution in isr context during a write command.
 * This routine fires off an asr to do work required.
 */

ISR writeisr(drno)
WORD    drno ;
{
    LONG r ;
    LUTE    *u ;

    u = P2LM[drno].physlute ;

    if(( r = DKASR( writeasr , u , 0L )) < 0L)  {
        phd_terminate( u , r ) ;
    }
    return(TRUE) ;                                  /* dispatch */
}

/********************************************************************
 * writeasr - This routine checks for command completion and terminates
 * the request if so, or inputs a buffer of data and sets up for the next
 * interation.
 *
 */

ASR writeasr(i)
IORB    *i ;
{
    BYTE    status ;
    LUTE    *u ;

    u = (LUTE *)i ;

    status = INP( HDCSTAT ) ;
    if( status & HDC_BSY ) {
        phd_terminate( u , (ED_DISK | E_DKATTACH) ) ;
    }
    else {
        if( ( i->io_error = chk_stat()) != NULLPTR )  {
            phd_iocompl( u ) ;
        }
        else  {
            if( status & HDC_DRQ )  {               /* output data  */
                if( u->lu_flags & DKF_UADDR ) MAPU( u->lu_b_pdaddr ) ;
                i->io_buffer =  OUTPSTR( HDCDATA , i->io_buffer , u->lu_lddp->ld_mdb.md_secsiz ) ;
                if( u->lu_flags & DKF_UADDR ) UNMAPU() ;
            }
            else  {
                phd_io1(i) ;
            }
        }
    }
    return (0);
}

/************************************************************************
 * verifyisr - gets executed in isr context after a verify command has
 * completed. this routine fires off an asr to do the work required.
 */

ISR verifyisr(drno)
WORD    drno ;
{
    LONG    r ;
    LUTE    *u ;

    u = P2LM[drno].physlute ;

    if(( r = DKASR( verifyasr , u , 0L )) < 0L)  {
        phd_terminate( u , r ) ;
    }
    return( TRUE ) ;                                /* dispatch */
}

/***********************************************************************
 * verifyasr - this routine checks for errors and terminates the request.
 */

ASR verifyasr(i)
IORB    *i ;
{

    if( ! HDCBUSY )  {
        i->io_error = chk_stat() ;
    }
    else  {
        i->io_error = (ED_DISK | E_DKATTACH) ;
    }
    phd_io1(i) ;
    return(0);
}


/*************************************************************************
* formatisr - gets executed after a format command is completed.
*/

ISR formatisr(drno)
WORD    drno ;
{
    LONG    r ;
    LUTE    *u ;

    u = P2LM[drno].physlute ;

    if(( r = DKASR( formatasr , u , 0L )) < 0L)  {
        phd_terminate( u , r ) ;
    }
    return(TRUE) ;
}

/****************************************************************************
* formatasr - executed after format interrupt to check and save status in a
* global location , turn off the watch dog timer , deallocates format buffer
* space, and terminates the req.
* ( no system functions in isr context) .
*/

ASR formatasr(i)
IORB    *i ;
{

    if( ! HDCBUSY )  {
        i->io_error = chk_stat() ;
    }
    else  {
        i->io_error = (ED_DISK | E_DKATTACH) ;
    }
    hd_cleartimer((LUTE *)i) ;
    SFREE( i->io_buffer ) ;
    phd_iocompl( (LUTE *) i ) ;
    return(0);
}


/**************************************************************************
 *  outputcmd -
 *  Output a high level disk command to the hdc.
 *  The hdtfp parm is a pointer to the appropriate
 *  task file.
 *
 */

BOOLEAN outputcmd( hdtfp )
HDTF    *hdtfp ;                                    /* ptr to Task File */
{

    BYTE    *c ;                                    /*  ptr to command task file string */
    WORD    ccnt ;                                  /* command count */
    WORD    port ;                                  /* port number of hdc task register    */

    for(c = &hdtfp->hd_cmd[0], port = HDCPRECOMP, ccnt = CFMAX ; ccnt-- ; )  {
        if( ! (outputbyte(port++ , *c++))  )  {
            /* output failed */
            return( FALSE ) ;
        }
    }
    return( TRUE ) ;
}


/************************************************************************
*  phd_setparm -
*   output parameters to hdc for multi-track, multi-cylinder i/o.
*   ( # heads, # sectors/track )
*/

ERROR   phd_setparm( u )
LUTE    *u ;
{
    HDTF    *cf ;
    LDD     *l ;                                    /* current ldd */
    BYTE    driveno ;                               /* drive number */
    BYTE    ss ;
    WORD    temp;				    /* slow down the process */

    l = u->lu_lddp ;                                /* init ptr */
    driveno = u->lu_driveno ;                       /* init drive anbr */

    cf = &HTF[u->lu_unitno] ;
    cf->hd_cmd[1] = (BYTE) ( l->ld_mdb.md_sectrk ) ;

    ss = bpsx( l->ld_mdb.md_secsiz ) ;

    cf->hd_cmd[5] =  HDCECC | (ss << 5) | (driveno << 4) | (u->lu_pudp->pu_nheads-1) ;

    cf->hd_cmd[6] = HDCSET ;                        /* set parameter hdc command */

    for ( temp = 0; temp < 2000 ; temp++) ;
    if( !outputcmd( cf ) )  {
	for ( temp = 0; temp < 2000 ; temp++) ;
        return( ED_DISK | E_DKATTACH ) ;
    }
    else  {
	for ( temp = 0; temp < 2000 ; temp++) ;
        return( E_SUCCESS ) ;
    }
}


/**************************************************************************
*   hd_settimer - set up watch dog timer
*
*
*/
VOID    hd_settimer( time,i )
LONG    time ;
IORB    *i ;
{
    LUTE    *u ;
    DKTIMPB *timpb ;

    u = (LUTE *)i ;
    timpb = &HDTIMPB ;
    timpb->tim_time = time ;                        /* requested timer setting */

    timemask[u->lu_unitno] = SUPIF( F_TIMER , timpb ) ;

    NEXTASR( timemask[u->lu_unitno], hd_timex, i, 0L, DKASRPRI+10 );
}

/*************************************************************************
*   hd_cleartimer - cancel timer
*
*/
VOID    hd_cleartimer(u)
LUTE    *u ;
{
    timemask[u->lu_unitno] = SUPIF( F_CANCEL , timemask[u->lu_unitno] ) ;
    timeouts[u->lu_unitno] = 0 ;
}

/*************************************************************************
*   hd_timex - watch dog timer has barked
*
*/

ASR hd_timex(i)
IORB    *i ;
{
    LUTE    *u ;                                    /* LUTE pointer to get the drive number */

    /* if the timer has been canceled do nothing */
    u = (LUTE *)i ;
    if (!(timemask[u->lu_unitno])) return( 0 ) ;

    SUPIF( F_RETURN , timemask[u->lu_unitno] ) ;    /* get the timer return code */
    if (P2LM[u->lu_unitno].seqisr == (ISR(*)() )seekisr)  {
        if(timeouts[u->lu_unitno]++ < tortry)  {
            phd_io(u) ;
        }
        else  {
            i->io_error = (ED_DISK | 0xbeef /*E_DKATTACH*/) ;
            phd_iocompl( u ) ;
        }
    }
    else  {
        i->io_error = (ED_DISK | 0xbeef /*E_DKATTACH*/) ;
        phd_iocompl( u ) ;
    }
    return(0); 
}


/************************************************************************
 *                                                                      *
 *          Hard Disk Driver Primitives                                 *
 *                                                                      *
 ************************************************************************/



/**************************************************************************
 *  outputbyte -
 *  Output a single command/parm byte to the hdc.
 *
 */

BOOLEAN outputbyte( port , byte )
BYTE    byte ;                                      /* data to output */
WORD    port ;                                      /* port number to output to */
{ 
    if( ! cmdquery()  )  {
        return( FALSE ) ;
    }
    OUTP( port , byte ) ;
    return( TRUE ) ;
}



/**************************************************************************
 *  cmdquery -
 *
 *  exit:  TRUE:    hdc is ready for cmd (not busy, seek comp., ready)
 *      FALSE:  hdc is not ready for command (busy)
 */

BOOLEAN cmdquery()
{
WORD    s ;                                         /* contents of hdc status register for bit masking */

    s = INP(HDCSTAT) ;
    /* removed seek check KPB */
    if( !(s & HDC_BSY) && (s & HDC_RDY) )  {
        return( TRUE ) ;
    }
    else  {
        return( FALSE ) ;
    }
}


/**************************************************************************
*  hcmdcode -
*   translate a command code into an hdc operation code.
*/

BYTE    hcmdcode(op)
BYTE    op ;
{
    switch(op)  {
        case DKREAD:    return( HDCREAD ) ;         /* code for hdc read */
        case DKWRITE:   return( HDCWRITE ) ;        /* code for hdc write*/
        case DKVERIFY:  return( HDCVERIFY ) ;       /* code for hdc verify*/
        case DKFMT:
        case DKFMTBB:
        case DKFMTLST:
        case DKFMTBBLST:
            return( HDCFORMAT ) ;                   /* code for hdc format*/

        default:    return( 0 ) ;
    }
}


/*************************************************************************
 *  bpsx -
 *  returns a byte per sector indicator that is controller specific,
 *  given a word value for the number of bytes per sector.
 *  default is 512 bytes per sector.
 */
BYTE    bpsx( n )
WORD    n ;
{
    static  BYTE    bytsec[9] = { 1,3,0,-1,1,-1,-1,-1,2 } ;

    return(bytsec[n/128]) ;
}

/**************************************************************************
 *  fgap -
 *  returns a format gap length that is controller specific,
 *  formula for minimum gap:
 *
 *  gap = 2*M*S + K + E
 *
 *  M = motor speed variation ( we will use 2% )
 *  S = sector length in bytes
 *  K = 25 for interleave factor of 1
 *  K = 0 for any other interleave factor
 *  E = 7 if the sector is to be extended
 */

BYTE    fgap( u )
LUTE    *u;
{
    BYTE    r ;
    static  BYTE    fgaps[4] = { 10 , 20 , 41 , 5 } ;


    r = bpsx( u->lu_lddp->ld_mdb.md_secsiz ) ;
    r = fgaps[ r ] ;

    /* don't check for interleave factor of one with this hardware  */

    return( r ) ;
}


/***********************************************************************
*  chk_stat -
*   analyzes result hdc status register and error register
*   and returns the appropriate return code
*   note: should not be called if the busy bit is set
*/

ERROR   chk_stat( )
{
    BYTE    status ;                                /* hdc status/error register */

    status =  INP( HDCSTAT ) & 0xf1 ;               /* mask off unused bits */
                                                    /* drq, ecc, index  */

    if( status == (HDC_RDY+HDC_SC)) {               /* quick o.k. check */
        return( E_SUCCESS ) ;                       /* good status */
    }
    else  {
        if( !(status & HDC_RDY ))  {                /* not ready ? */
            return( ED_DISK | E_READY ) ;
        }
        else  {
            if( status & HDC_WF )  {                /* write fault ? */
                return( ED_DISK | E_WRITEFAULT ) ;
            }
            else  {
                if( !( status & HDC_SC ))  {        /* seek not complete ?  */
                    return( ED_DISK | E_SEEK ) ;           /* not an error if overlapped seeks */
                }
                else  {
                    if( !( status & HDC_ERR ))  {
                        return( ED_DISK | E_DKATTACH ) ;      /* busy slipped in ?    */
                    }
                    else  {
                        status = INP( HDCERR ) ;
                        if( status & HDC_BB )  {    /* bad block detect ?   */
                            return( ED_DISK | E_SEC_NOTFOUND ) ;
                        }
                        else  {
                            if( status & HDC_CRC )  {     /* uncorrectable ecc ?  */
                                return( ED_DISK | E_CRC ) ;
                            }
                            else  {
                                if( status & HDC_ID )  {     /* id mark not found ?  */
                                    return( ED_DISK | E_MISADDR ) ;
                                }
                                else  {
                                    if( status & HDC_AC ) {   /* aborted command ?    */
                                        return( ED_DISK | E_INVCMD ) ;
                                    }
                                    else {
                                        if( status & HDC_TK )  {  /* track 0 not found ?  */
                                            return( ED_DISK | E_SEEK ) ;
                                        }
                                        else  {
                                            if( status & HDC_DM )  {  /* data mark not found ?*/
                                                return( ED_DISK | E_SEC_NOTFOUND ) ;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
