/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service_ngen_qic.c                                                   */
/*                                                                           */
/*  Workstation QIC-02 device procedures for CTOS Sequential Access service. */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/17/91  121J.06  P. Johansson  /  Don't allow buffered mode to change  */
/*                                      in SeqAccessModeSet; HB-001 clears   */
/*                                      DMA count registers if blank tape is */
/*                                      encountered before ANY data is read, */
/*                                      so force actual transfer count to 0. */
/*  04/19/91  121J.05  P. Johansson  /  Don't write a filemark if the count  */
/*                                      is zero.                             */
/*  03/29/91  121H.04  P. Johansson  /  Fixes to checkpoint logic at EOM.    */
/*  01/29/91  121F.03  P. Johansson  /  Add watchdog for tape timeouts.      */
/*  01/15/91  121F.02  P. Johansson  /  Device buffers have been transferred */
/*                                      to medium after EOT exception.       */
/*  01/04/91  121F.01  P. Johansson  /  New procedure to return default      */
/*                                      device buffer configuration details. */
/*  12/17/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/*****************************  MODULE HEADER  *******************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define CloseRTClock
#define Delay
#define MapBusAddress
#define OpenRTClock
#define PaFromP
#define QueryRequestInfo
#define Request
#define Send
#define ShortDelay
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define CtQIC

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "ngen_qic.h"
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void disable(void);
extern void enable(void);
extern char input(unsigned io_address);
extern void output(unsigned io_address, char value);
extern void QicISR(void);

/* Error return codes used by this module */

#define RqErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned default_resp_exch;
extern unsigned internal_os_version;
extern unsigned own_user_num;
extern qic_iob_type *qic_iob;
extern unsigned qic_base_addr;
extern char *qic_control;
extern unsigned serv_exch;

/* Static variables global within this manuscript */

private dcb_qic_type *dcb;
private watchdog_type watchdog = {0, 10, 0, 0, 0, 0xFFFF};

/* Function prototypes defined before the functions themselves are declared */

unsigned issue_qic_command(char command);
unsigned map_qic_status(void);
unsigned qic_ready(void);
unsigned read_qic_status(void);
unsigned reset_qic_firmware(void);
unsigned reset_xbus_misr(void);
unsigned select_qic(Boolean lock_cartridge);

pragma Page(1);
/*-----------------------------------------------------------------------------
 The initialization procedure is called once when the Sequential Access
 service is installed.  The HB-001 hardware does not need any particular
 initialization to eliminate, for example, unwanted interrupts from floating
 I/O signals, so this code merely establishes some of the data structure
 contents for subsequent use when the device is opened.  NB: Because only one
 HB-001 (or Flemington QIC module) is permitted to be attached to the X-Bus at
 a time, there are no reentrancy worries with the use of a static variable
 ('dcb') to make references to the QIC Dcb faster throughout the rest of this
 manuscript. */

void ngen_qic_init(dcb_type *uninitialized_dcb, unsigned base_address) {

   unsigned i;

   dcb = (dcb_qic_type *) (watchdog.dcb = uninitialized_dcb);
   watchdog.exch_resp = serv_exch;
   qic_base_addr = base_address;
   qic_control = &dcb->qic_control;
   qic_iob = (qic_iob_type *) &dcb->own_rq;
   for (i = 0; i <= last(dcb->qic_reg_addr); i++)
      dcb->qic_reg_addr[i] = (base_address | (QIC_REG_ADDR + (2 * i)));

}

/*-----------------------------------------------------------------------------
 This function provides device specific information about buffer capacity so
 that SeqAccessOpen may allocate buffers intelligently. */

unsigned ngen_qic_defaults(default_info_type *default_info) {

   default_info->device_buffer_size = 2048;
   default_info->recommended_buffers = 4;
   default_info->recommended_buffer_size = 8192;
   default_info->recommended_buffer_pool_size = 32768;
   default_info->recommended_write_buffer_full_threshold = 2048;
   default_info->recommended_read_buffer_empty_threshold = 2048;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Information that is in the "erasable" part of the Dcb must be reestablished
 each time the device is opened (the information in the device dependent
 portion, which is set up by the initialization procedure, is safe from
 erasure).  In addition, a real-time clock for the gross timing of QIC
 operations must be established, the X-Bus interrupt mediator must be notified
 that we wish to receive interrupts and the HB-001 interface firmware must be
 reset. */

unsigned ngen_qic_open(dcb_qic_type *dcb) {

   unsigned erc, rq_info;
   void *data_segment = &qic_base_addr, *response_rq;
   set_xbus_misr_rq_type set_xbus_misr_rq;

   dcb->own_rq.s_cnt_info = 4;
   dcb->own_rq.exch_resp = serv_exch;
   dcb->own_rq.user_num = own_user_num;
   dcb->min_record_size = dcb->max_record_size = dcb->block_size = PAGE_SIZE;
   if (watchdog.user_count++ == 0) {
      watchdog.counter = watchdog.counter_reload;
      watchdog.events = 0;
      if ((erc = OpenRTClock(&watchdog)) != ercOK) {
         watchdog.user_count--;
         return(erc);
      }
   }
   memset(&set_xbus_misr_rq, 0, sizeof(set_xbus_misr_rq));
   set_xbus_misr_rq.s_cnt_info = 6;
   set_xbus_misr_rq.n_resp_pb_cb = 1;
   set_xbus_misr_rq.exch_resp = default_resp_exch;
   set_xbus_misr_rq.erc_ret = 0xFFFF;
   if (QueryRequestInfo(set_xbus_misr_rq.rq_code = SET_XBUS_MISR_RQ_CODE,
                        &rq_info, sizeof(rq_info)) != ercOK || rq_info == 0)
      set_xbus_misr_rq.rq_code = OLD_SET_XBUS_MISR_RQ_CODE;
   set_xbus_misr_rq.interrupt_service = (void *) QicISR;
   set_xbus_misr_rq.data_segment = selector_of(data_segment);
   set_xbus_misr_rq.interrupt_handle_ret = &dcb->xbus_handle;
   set_xbus_misr_rq.size_interrupt_handle_ret = sizeof(dcb->xbus_handle);
   if ((erc = Request(&set_xbus_misr_rq)) == ercOK) {
      while (   (erc = Wait(default_resp_exch, &response_rq)) == ercOK
             && &set_xbus_misr_rq != response_rq)
         ;
      if (erc == ercOK)
         erc = set_xbus_misr_rq.erc_ret;
   }
   if (erc != ercOK)
      return((erc == ercServiceNotAvail) ? ercXbifNotInstalled : erc);
   reset_qic_firmware();
   return(select_qic(TRUE));	/* Select the drive and lock the cartridge */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When the HB-001 QIC is closed, perform a SELECT DRIVE command without the
 LOCK CARTRIDGE bit: this causes the LED on the drive itself (not the LED on
 the module) to be extinguished.  Then deassert any control signals to the
 module, close the real-time clock and release the shared X-Bus interrupt. */

unsigned ngen_qic_close(dcb_qic_type *dcb) {

   unsigned erc = ercOK;

   select_qic(FALSE);				/* Unlock the cartridge */
   output(dcb->qic_control_reg, dcb->qic_control = 0);
   if (watchdog.user_count == 0)
      erc = reset_xbus_misr();
   else if (   --watchdog.user_count == 0
            && (erc = CloseRTClock(&watchdog)) != ercOK)
      reset_xbus_misr();
   else
      erc = reset_xbus_misr();
   return(erc);

}

private unsigned reset_xbus_misr(void) {

   unsigned erc, rq_info = 0;
   void *response_rq;
   reset_xbus_misr_rq_type reset_xbus_misr_rq;

   memset(&reset_xbus_misr_rq, 0, sizeof(reset_xbus_misr_rq));
   reset_xbus_misr_rq.s_cnt_info = 2;
   reset_xbus_misr_rq.exch_resp = default_resp_exch;
   reset_xbus_misr_rq.erc_ret = 0xFFFF;
   if (QueryRequestInfo(reset_xbus_misr_rq.rq_code = RESET_XBUS_MISR_RQ_CODE,
                        &rq_info, sizeof(rq_info)) != ercOK || rq_info == 0)
      reset_xbus_misr_rq.rq_code = OLD_RESET_XBUS_MISR_RQ_CODE;
   reset_xbus_misr_rq.interrupt_handle = dcb->xbus_handle;
   if ((erc = Request(&reset_xbus_misr_rq)) == ercOK)
      while (   (erc = Wait(default_resp_exch, &response_rq)) == ercOK
             && &reset_xbus_misr_rq != response_rq)
         ;
   if (erc == ercOK)
      erc = reset_xbus_misr_rq.erc_ret;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 There is nothing controllable or user configurable about the QIC-02 tape
 backup module.  However, the generic Sequential Access service is expecting
 reasonable answers to be provided in the mode parameters block.  It arrives
 zero filled, which simplifies the job here tremendously! */

unsigned ngen_qic_mode_query(dcb_qic_type *dcb,
                             seq_parameters_type *seq_parameters) {

   if (read_qic_status() == ercOK)
      seq_parameters->write_protected = dcb->unit_status.write_protected;
   seq_parameters->density = 0x05;		/* QIC-24 */
   seq_parameters->block_size =			/* Not selectable, 512 bytes */
      seq_parameters->min_record_size = 
      seq_parameters->max_record_size =  PAGE_SIZE;
   seq_parameters->disable_error_correction = TRUE;
   seq_parameters->read_retry_limit =		/* Built into firmware */
      seq_parameters->write_retry_limit = 16;
   seq_parameters->data_buffer_nonrecoverable = TRUE;
   return(ercOK);

}

unsigned ngen_qic_mode_set(dcb_qic_type *dcb,
                           seq_parameters_type *seq_parameters) {

   dcb->operating_mode.variable_length = FALSE;
   dcb->operating_mode.buffer_recoverable = FALSE;
   dcb->operating_mode.data_compression = FALSE;
   dcb->block_size = dcb->min_record_size =  dcb->max_record_size =  PAGE_SIZE;
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For each of the different SeqAccessControl functions, select the appropriate
 QIC-02 command and establish a timeout limit (in tenths of seconds) long
 enough for the operation to complete.  Note that record spacing has to be
 implemented by QIC-02 read data operations---in this case, the fact that
 'data_transfer' is not set in the Dcb state indicates that data read is to be
 discarded, not transferred to a (nonexistent!) user buffer. */

unsigned ngen_qic_ctrl(dcb_qic_type *dcb) {

   char command;
   seq_access_rq_type *rq;

   rq = dcb->rq;
   switch (dcb->ctrl_function = rq->ctrl_function) {
      case CTRL_REWIND:
      case CTRL_UNLOAD:
         command = REWIND;
         dcb->timeout = 100;
         break;

      case CTRL_RETENSION:
         command = RETENSION;
         dcb->timeout = 200;
         break;

      case CTRL_ERASE_MEDIUM:
         command = ERASE;
         dcb->timeout = 200;
         break;

      case CTRL_WRITE_FILEMARK:
         if (rq->ctrl_qualifier < 0)
            return(ercTapeCommandIllegal);
         command = WRITE_FILEMARK;
         if ((dcb->ctrl_qualifier = rq->ctrl_qualifier) == 0)
            return(ercOK);
         dcb->timeout = 60;
         break;

      case CTRL_SCAN_FILEMARK:
         if (rq->ctrl_qualifier != 1)
            return(ercTapeCommandIllegal);
         command = READ_FILEMARK;
         dcb->ctrl_qualifier = rq->ctrl_qualifier;
         dcb->timeout = 900;
         break;

      case CTRL_ERASE_GAP:
         return(ercOK);

      default:
         return(ercTapeCommandIllegal);
   }
   return(issue_qic_command(command));

}

/*-----------------------------------------------------------------------------
 If the six QIC-02 status bytes are already available because of an earlier
 QIC EXCEPTION condition, it is sufficient to map them into an error return
 code and set the device independent status information in the Dcb.
 Otherwise, issue a READ STATUS command and map its results.  Note that
 because of an interface peculiarity it is necessary to request the status
 twice: some exception conditions (i.e. NOT READY and NO CARTRIDGE) do not
 seem to clear on the first READ STATUS after the condition has been
 corrected. */

unsigned ngen_qic_status(dcb_qic_type *dcb) {

   unsigned erc;

   if (dcb->state.status_available)
      erc = map_qic_status();
   else {	/* Note strange QIC_02 firmware bug --- get status twice! */
      read_qic_status();
      erc = read_qic_status();
   }
   dcb->state.pending_status = dcb->state.status_available = FALSE;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Data transfer operations for the HB-001 module require that the DMA and
 counter/timer circuitry in the module be programmed.  The DMA transfer is
 controlled by a 24-bit bus address and by a 16-bit transfer count.  There are
 obscure rules pertaining to HB-001 hardware design that differentiate reads
 from writes and the first of a series of data transfer operations from the
 subsequent ones.  Essentially, on the first READ DATA the count register is
 to be loaded with transfer size less one while all subsequent read transfers
 require that the count be programmed to the number of bytes less two.  For a
 WRITE DATA command, the count register is always loaded with the transfer
 size less one but the second counter register must be programmed as a
 one-shot with the page size less two on the first WRITE DATA, only.  Other
 than that, everything is pretty much straight-forward. */

unsigned ngen_qic_io(dcb_qic_type *dcb) {

   unsigned long bus_address;
   char command;
   unsigned erc, garbage, xfer_count;

   if (      dcb->device_buffer->bus_address == 0
          && (erc = (internal_os_version >= 0x0C00)
                     ? MapBusAddress(dcb->io_data = dcb->device_buffer->data,
                                     dcb->device_buffer->size, NO_DMA_FENCE,
                                     &dcb->device_buffer->bus_address,
                                     &garbage)
                     : PaFromP(dcb->io_data = dcb->device_buffer->data, 0,
                               &dcb->device_buffer->bus_address)) != ercOK)
      return(erc);
   if ((erc = qic_ready()) != ercOK)
      return((erc == ercQicException) ? read_qic_status() : erc);
   if (dcb->command == READ_DATA)
      xfer_count = (dcb->io_xfer_count = dcb->device_buffer->available) - 2;
   else
      xfer_count = (dcb->io_xfer_count = dcb->device_buffer->available) - 1;
   dcb->ctrl_function = 0xFFFF;	/* Make sure it's not a real function */
   dcb->timeout = 60;		/* 60 seconds derived empirically */
   output(dcb->qic_timer_mode_reg, 0x30);
   output(dcb->qic_counter0_reg, xfer_count);
   output(dcb->qic_counter0_reg, xfer_count >> 8);
   if (!dcb->state.inbound && dcb->command != WRITE_DATA) {
      output(dcb->qic_timer_mode_reg, 0x72);
      output(dcb->qic_counter1_reg, PAGE_SIZE - 2);
      output(dcb->qic_counter1_reg, (PAGE_SIZE - 2) >> 8);
   }
   bus_address = dcb->device_buffer->bus_address >> 1;
   disable();
   output(dcb->qic_dma_addr_low_reg, bus_address);
   output(dcb->qic_dma_addr_mid_reg, bus_address >> 8);
   output(dcb->qic_dma_addr_high_reg, bus_address >> 16);
   enable();
   if (dcb->state.inbound) {
      command = READ_DATA;
      dcb->qic_control |= QIC_DMA_WRITE;
   } else {
      command = WRITE_DATA;
      dcb->qic_control |= QIC_DMA_READ;
   }
   return(issue_qic_command(command));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Most of the time, the READY signal is asserted very quickly by the QIC-02
 interface.  Occasionally, though, when a lengthy error recovery or retry
 process is underway in the hardware it may take MUCH longer for READY to be
 present.  The maximum limit of 40 seconds for the secondary wait was arrived
 at empirically with a few error-laden QIC tapes. */

private unsigned qic_ready(void) {

   char ctrl_lines;
   unsigned ctrl_line_reg, i;

   ctrl_line_reg = dcb->qic_ctrl_line_reg;
   for (i = 1; i <= 5000; i++)
      if ((ctrl_lines = input(ctrl_line_reg)) & QIC_EXCEPTION)
         return(ercQicException);
      else if (ctrl_lines & QIC_READY)
         return(ercOK);
   for (i = 1; i <= 400; i++) {
      Delay(1);
      if ((ctrl_lines = input(ctrl_line_reg)) & QIC_EXCEPTION)
         return(ercQicException);
      else if (ctrl_lines & QIC_READY)
         return(ercOK);
   }
   return(ercTapeNotReady);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The QIC-02 READ STATUS command is essentially an immediate command because
 its timing is so fast.  Wait in a tight loop until the status bytes are
 available and then read all six.  Subsequently map the status information to
 an error return code and set the generic device status information in the
 Dcb. */

private unsigned read_qic_status(void) {

   unsigned erc, i;

   dcb->state.status_available = FALSE;
   qic_iob->erc_ret = REQUEST_IN_PROGRESS;
   qic_iob->send_msg = qic_iob->interrupt = FALSE;
   ShortDelay(10);
   output(dcb->qic_command_reg, dcb->command = READ_STATUS);
   ShortDelay(10);
   output(dcb->qic_control_reg, dcb->qic_control |= QIC_SELECT);
   for (i = 1; i <= 100; i++) {
      if (!qic_iob->interrupt)
         Delay(1);		/* Wait circa ten seconds for interrupt... */
      else if (qic_iob->qic_ctrl_lines & QIC_EXCEPTION)
         return(ercTapeStatusUnavailable);
      else {
         for (i = 0; i <= last(dcb->qic_status); i++) {
            if ((erc = qic_ready()) != ercOK)
               return(ercTapeStatusUnavailable);
            dcb->qic_status[i] = input(dcb->qic_status_reg);
         }
         dcb->state.status_available = TRUE;
         return(map_qic_status());
      }
   }
   output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_SELECT);
   return(ercTapeStatusUnavailable);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The HB-001 interface firmware is normally reset when a SeqAccessOpen is
 performed or if a catastrophic error condition (e.g. I/O operation timeout)
 occurs.  After performing the reset (and awaiting the interrupt that signals
 its completion) read the QIC-02 status TWICE to clear the power-on reset bit
 from the status bytes.  At other times, this bit is considered abnormal. */

private unsigned reset_qic_firmware(void) {

   unsigned i;

   dcb->command = RESET_QIC_FIRMWARE;
   dcb->state.status_available = FALSE;
   qic_iob->erc_ret = REQUEST_IN_PROGRESS;
   qic_iob->send_msg = qic_iob->interrupt = FALSE;
   output(dcb->qic_control_reg,
          (dcb->qic_control = (QIC_ONLINE | QIC_SELECT)) | QIC_RESET);
   for (i = 1; i <= 100; i++) {
      if (!qic_iob->interrupt)
         Delay(1);		/* Wait circa ten seconds for interrupt... */
      else {
         read_qic_status();	/* Read status once to clear reset bit... */
         ShortDelay(10);
         return(read_qic_status());	/* ...then return real status */
      }
   }
   output(dcb->qic_control_reg, dcb->qic_control = 0);
   return(ercTapeHardwareError);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 During SeqAccessOpen and SeqAccessClose, as SELECT DRIVE command sequence is
 performed to respectively lock or unlock the cartridge.  This does not
 physically prevent removal of the medium, it only ensures that if the medium
 is removed that an exception status is generated on the next command to the
 drive.  In this way, we can discover if the medium has been changed while the
 drive happened to be quiescent. */

private unsigned select_qic(Boolean lock_cartridge) {

   unsigned erc, i;

   dcb->state.status_available = FALSE;
   qic_iob->erc_ret = REQUEST_IN_PROGRESS;
   qic_iob->send_msg = qic_iob->interrupt = FALSE;
   if ((erc = qic_ready()) != ercOK)
      return(erc);
   output(dcb->qic_control_reg, dcb->qic_control |= QIC_SELECT);
   output(dcb->qic_command_reg,
          dcb->command = (lock_cartridge) ? SELECT_DRIVE | LOCK_CARTRIDGE
                                          : SELECT_DRIVE);
   for (i = 1; i <= 100; i++)
      if (!qic_iob->interrupt)
         Delay(1);		/* Wait circa ten seconds for interrupt... */
      else if (qic_iob->qic_ctrl_lines & QIC_EXCEPTION)
         return(read_qic_status());
      else
         return(ercOK);
   output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_SELECT);
   return(ercTapeHardwareError);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 All QIC-02 commands that require tape motion (and hence an appreciable amount
 of time for their completion) are processed by this routine.  After
 determining that the interface is ready to accept a command, the HB-001
 interrupts are enabled and the LED turned on to let the user know the drive
 is active.  Then, if the command is not a data transfer command (or if this
 is the first time a READ DATA or WRITE DATA command has been issued after a
 non-data transfer command), the command is written to the control register to
 commence operations. */

private unsigned issue_qic_command(char command) {

   unsigned erc;

   dcb->state.status_available = FALSE;	/* Status cleared by new command */
   dcb->io_xfer_actual = 0;
   qic_iob->erc_ret = REQUEST_IN_PROGRESS;
   qic_iob->send_msg = TRUE;
   qic_iob->interrupt = FALSE;
   if ((erc = qic_ready()) != ercOK)
      return((erc == ercQicException) ? read_qic_status() : erc);
   output(dcb->qic_control_reg, dcb->qic_control |= QIC_SELECT);
   if (     command != dcb->command
         || ((command != READ_DATA) && (command != WRITE_DATA)))
      output(dcb->qic_command_reg, dcb->command = command);
   return(REQUEST_IN_PROGRESS);

}

/*-----------------------------------------------------------------------------
 If data transfer had been in progress, read the counter/timer registers in
 the HB-001 module to determine the transfer residual and calculate the actual
 quantity of data transferred accordingly (this is used by the higher level,
 device independent procedure that called us).  Also, if an exception
 condition occurred, read the QIC-02 status avaialable from the interface.
 Note that the HB-001 DMA circuitry, for some reason, clears the counter
 registers when blank tape is encountered before any data at all is
 transferred---so we can't trust the computed transfer count in this case and
 must force it to zero. */

unsigned ngen_qic_rq_done(dcb_qic_type *dcb) {

   unsigned erc;

   dcb->timeout = 0;			/* OK or in error, it's still done */
   if ((erc = qic_iob->erc_ret) != ercOK) {
      dcb->state.status_available = FALSE;	/* No QIC-02 status... */
      dcb->state.pending_status = TRUE;		/* ..but an error is pending */
   } else {
      if (dcb->state.data_transfer) {
         output(dcb->qic_timer_mode_reg, 0x30);	/* Read residual from 8253 */
         dcb->io_xfer_actual = dcb->io_xfer_count
                            - (  (unsigned) input(dcb->qic_counter0_reg)
                               | (unsigned) input(dcb->qic_counter0_reg) << 8);
      }
      if (qic_iob->qic_ctrl_lines & QIC_EXCEPTION) {	/* Status available? */
         dcb->state.status_available = TRUE;
         dcb->state.pending_status = ((erc = read_qic_status()) != ercOK);
         if (     erc == ercTapeBlank
               && dcb->state.data_transfer
               && dcb->io_xfer_actual == dcb->io_xfer_count)
            dcb->io_xfer_actual = 0;		/* DMA peculiarity of HB-001 */
      }
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The watchdog procedure is periodically activated by a timer request block.
 Because the counter reload field is set to 10 and the timer has a granularity
 of 100 milliseconds, the 'events' field holds the number of seconds that have
 elapsed since the last invocation.  This value is used to decrement the
 timeout field in the Dcb, which is also expressed in one second units.
 If time has run out, interrupts are disabled to guard the critical section
 with the interrupt service routine and the HB-001 interface firmware is
 reset.  After this reset, the timeout error code is inserted into the Iob
 which is then sent to the service exchange for the normal completion
 processing.  While the QIC device is open by a user, the watchdog is always
 running (even if no QIC operations are in progress). */

void ngen_qic_watchdog(watchdog_type *watchdog) {

   if (dcb->timeout > watchdog->events)	/* Time still left on the clock? */
      dcb->timeout -= watchdog->events;	/* Yes, just decrement it */
   else if (dcb->timeout != 0) {	/* Is the timer enabled? */
      dcb->timeout = 0;			/* Not any more! */
      disable();			/* Critical section with ISR... */
      if (qic_iob->interrupt)
         enable();			/* Whew! Got in just under the wire */
      else {
         output(dcb->qic_control_reg, dcb->qic_control &= ~QIC_SELECT);
         enable();			/* OK now 'cause interrupts disabled */
         qic_iob->send_msg = FALSE;	/* We wish to ignore reset interrupt */
         reset_qic_firmware();		/* Reset known state of the world */
         qic_iob->erc_ret = ercTapeTimeout;
         Send(qic_iob->exch_resp, qic_iob);	/* Cause "done" code to run */
      }
   }
   watchdog->events = 0;		/* Ensure a subsequent reactivation */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Synthesize the six QIC-02 status bytes into the virtual status information
 maintained in the Dcb.  After that is done, see if an error return code has
 to be generated unique to the status condition. */

private unsigned map_qic_status(void) {

   unsigned erc = ercOK;		/* Innocent until proven guilty */

   dcb->soft_errors += ((dcb->qic_status[2] << 8) + dcb->qic_status[3]);
   dcb->underruns += ((dcb->qic_status[4] << 8) + dcb->qic_status[5]);
   dcb->unit_status.reset = ((dcb->qic_status[1] & POWER_UP_RESET) != 0);
   dcb->unit_status.busy = ((dcb->qic_ctrl_lines & QIC_READY) == 0);
   dcb->unit_status.ready = ((dcb->qic_status[0] & NO_CARTRIDGE) == 0);
   dcb->unit_status.write_protected =
         (dcb->unit_status.ready && (dcb->qic_status[0] & WRITE_PROTECT) != 0);
   dcb->unit_status.on_line = ((dcb->qic_status[0] & NOT_ONLINE) == 0);
   dcb->position.beginning_of_medium = ((dcb->qic_status[1] & BOM) != 0);
   dcb->position.end_of_data = ((dcb->qic_status[1] & EOD) != 0);
   dcb->position.end_of_medium = ((dcb->qic_status[0] & EOM) != 0);
   dcb->command_status.file_mark = ((dcb->qic_status[0] & FILEMARK) != 0);
   dcb->command_status.illegal_operation =
                                 ((dcb->qic_status[1] & ILLEGAL_COMMAND) != 0);
   dcb->command_status.recovered_error =
                                    (dcb->qic_status[1] & EIGHT_RETRIES)
                                 && !(dcb->qic_status[0] & UNRECOVERABLE_DATA);
   dcb->command_status.hard_error =
                              ((dcb->qic_status[0] & UNRECOVERABLE_DATA) != 0);
   if (!dcb->unit_status.on_line)
      erc = ercTapeHardwareError;
   else if (dcb->unit_status.reset)
      erc = ercTapeReset;
   else if (dcb->command_status.illegal_operation)
      erc = ercTapeCommandIllegal;
   else if (!dcb->unit_status.ready)
      erc = ercTapeNotReady;
   else if (dcb->qic_status[1] & NO_DATA || dcb->position.end_of_data)
      erc = ercTapeBlank;
   else if (dcb->command_status.file_mark)
      erc = (dcb->ctrl_function == CTRL_SCAN_FILEMARK) ? ercOK
                                                       : ercTapeFileMark;
   else if (dcb->command_status.hard_error)
      erc = ercTapeIoError;
   else if (dcb->position.end_of_medium && !dcb->state.ignore_EOM)
      erc = ercTapeEomWarning;
   else if (dcb->unit_status.write_protected && dcb->mode == MODE_MODIFY)
      erc = ercTapeWriteProtected;
   return(erc);

}
