/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*                                                                           */
/*  seq_service_init.c                                                       */
/*                                                                           */
/*  Initialization procedures for CTOS Sequential Access service.            */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  04/25/91  121J.07  P. Johansson  /  All BTOS name SCSI sequential access */
/*                                      devices "S0", "S1", etc.             */
/*  03/22/91  121H.06  P. Johansson  /  Check request codes 0x200A and       */
/*                                      old 0x801D for XBIS installation;    */
/*                                      for BTOS 8.2 and CTOS 9.10, do Exit  */
/*                                      immediately after ConvertToSys.      */
/*  03/07/91  121H.05  P. Johansson  /  Fix bug when installed twice on SRP. */
/*  02/13/91  121G.04  P. Johansson  /  Permit installation under CTOS 9.10, */
/*                                      CTOS/VM 2.3, CTOS/VM 2.4, BTOS 8.2.1,*/
/*                                      BTOS II 3.0.1 and BTOS II 3.2.0 plus */
/*                                      (mainstream) CTOS 3.0 and above.     */
/*  02/01/91  121F.03  P. Johansson  /  SeqAccessDiscardBufferData request.  */
/*  01/22/91  121F.02  P. Johansson  /  SuperGen support.                    */
/*  01/07/91  121F.01  P. Johansson  /  Modified to support Pertec 1/2".     */
/*  12/12/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <ctype.h>
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* Suppress C run-time (only CTOS functionality needed) */

pragma Off(List);
#include <stub.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocExch
#define AllocMemorySL
#define ChangePriority
#define CheckErc
#define ConfigurationQuery
#define ConvertToSys
#define CSubparams
#define CurrentOsVersion
#define ErrorExit
#define FSrp
#define GetModuleId
#define GetPStructure
#define GetScsiInfo
#define GetUserNumber
#define NlsULCmpB
#define ParseFileSpec
#define QueryDefaultRespExch
#define QueryRequestInfo
#define RgParam
#define ScsiManagerNameQuery
#define ServeRq
#define SetMsgRet
#define SetPartitionName
#define Exit

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define scsi_info_type
#define sdType
#define Syscom
#define SysConfigType

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include <scsi.h>
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern void initialize_buffer_pool(unsigned buffer_pages);
extern void initialize_msgs(void);
extern void register_exit_procedure(void (*procedure)(void));
extern unsigned seq_access(void);
extern sdType *standard_msg(unsigned msg_num);
extern unsigned update_route_table(dcb_type *dcb, char operation);

/* Error return codes used by this module */

#define InitErc
#define FsErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern void (*device_init[])(dcb_type *dcb, unsigned device_info);
extern void (*device_open[])(dcb_type *dcb);
extern unsigned n_dcb;
extern dcb_type *dcb_array[];

/* Global variables exported by this manuscript */

unsigned default_resp_exch, serv_exch, term_exch;
unsigned internal_os_version;
unsigned own_user_num;
unsigned seq_access_rq_code[] = {SEQ_ACCESS_TERMINATION_RQ_CODE,
   SEQ_ACCESS_READ_RQ_CODE, SEQ_ACCESS_WRITE_RQ_CODE,
   SEQ_ACCESS_CONTROL_RQ_CODE, SEQ_ACCESS_STATUS_RQ_CODE,
   SEQ_ACCESS_CHECKPOINT_RQ_CODE, SEQ_ACCESS_RECOVER_BUFFER_DATA_RQ_CODE,
   SEQ_ACCESS_DISCARD_BUFFER_DATA_RQ_CODE, SEQ_ACCESS_OPEN_RQ_CODE,
   SEQ_ACCESS_CLOSE_RQ_CODE, SEQ_ACCESS_MODE_QUERY_RQ_CODE,
   SEQ_ACCESS_MODE_SET_RQ_CODE, SEQ_ACCESS_VERSION_RQ_CODE,
   SEQ_ACCESS_DEINSTALL_RQ_CODE};
unsigned seq_access_rq_code_last = last(seq_access_rq_code);
unsigned save_rq_info[last(seq_access_rq_code) + 1];
SysConfigType *sysConfig;

/* Static variables global within this manuscript */

private unsigned buffer_pages = 0, n_device;
private char device_name[MAX_SEQ_DEVICES][13] = {"\3QIC"}, qic_slot;

/* Function prototypes defined before the functions themselves are declared */

void allocate_dcb(dcb_type **dcb, char device_name[], char password[],
                  char device_class, unsigned device_info);
void allocate_scsi_dcb(dcb_scsi_type **scsi_dcb, char device_name[],
                       char password[], char host_adapter, char scsi_id,
                       char lun, char scsi_manager_name[]);
void convert_to_sys(void);
void delete_route_table_entries(void);
void scan_config_file(void);
void scan_hardware(void);
void serve_requests(void);
void verify_parameters(void);
void verify_requests(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Sequential Access service installation sequence does the usual housekeeping
 (obtains native language messages, if present, allocates exchanges, etc.) and
 then gets down to scanning the hardware configuration to determine which
 sequential access devices to take under our wing.  If no such devices are
 found, exit (but let the user know!).  Otherwise, serve the requests, convert
 to a system service and leave to wait on the service exchange. */

void main(void) {

   unsigned os_revision, os_version;

   initialize_msgs();		/* Native language support, if present */
   CheckErc(CurrentOsVersion(&os_version, &os_revision));
   internal_os_version = (os_version << 8) | os_revision;
   if (     os_version < 0x0C	/* All newest CTOS are OK.... */
         && (   !FSrp(NULL)	/* Older versions on workstations, only */
             && (   os_version != 0x09
                 || (   os_revision != 0x07	/* BTOS 8.2.1 */
                     && os_revision != 0x08	/* BTOS II 3.0.1 Real */
                     && os_revision != 0x0A	/* CTOS 9.10 */
                     && os_revision != 0x0B	/* BTOS II 3.2.0 Real */
                     && os_revision != 0x0D))	/* CTOS 3.3 Real */
             && (   os_version != 0x0B
                 || (   os_revision != 0x03	/* CTOS/VM 2.3 */
                     && os_revision != 0x04	/* BTOS II 3.0.1 Protected */
                     && os_revision != 0x06	/* CTOS/VM 2.4 */
                     && os_revision != 0x07))))	/* BTOS II 3.2.0 Protected */
      exit_with_msg(ercOldOS, NLS_WRONG_OS);
   CheckErc(GetPStructure(ATpConfiguration, 0, &sysConfig));
   verify_parameters();		/* Consistency checking, only */
   verify_requests();		/* In case we're already installed! */
   CheckErc(QueryDefaultRespExch(&default_resp_exch));
   CheckErc(AllocExch(&serv_exch));
   CheckErc(AllocExch(&term_exch));
   initialize_buffer_pool(buffer_pages);	/* Allocate for use later */
   if (FSrp(NULL))
      scan_config_file();	/* Consult SrpConfig.Sys for devices */
   else
      scan_hardware();		/* See how many devices are attached */
   if (n_dcb == 0)		/* Any sequential access devices found? */
      exit_with_msg(ercOK, NLS_NO_DEVICES);
   serve_requests();		/* Almost ready to begin */
   convert_to_sys();		/* The last hurdle! */
   seq_access();		/* Wait forever at service exchange */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Validate the parameters supplied by the user in the Executive command form
 (or otherwise passed by the VLPB).
 
 	Install Sequential Access Service
 	    [Device(s) ([QIC])]
 	    [Buffer pool size (in Kb segments, default 64)]
 	    [QIC interface slot (SRP/XE 530 only, default 77)]

 On the workstations, with CTOS 4.0 and above, multiple sequential access
 devices are supported and are assigned the names specified by the user in the
 order they are encountered on the SCSI bus and/or X-Bus, from left to right.
 On the SRP/XE the configuration file allows the description of multiple
 sequential access devices and the user must specify, at the time of
 installation, which devices this particular instance of the Sequential Access
 Service is to control. */

private void verify_parameters(void) {

   unsigned erc, i, j, k, node_len, position_len, text_len;
   sdType sd_param;
   char text[MAX_DEVICE_LENGTH];

   if ((i = _min(CSubparams(DEVICE_NAME_PARAM), MAX_SEQ_DEVICES)) == 0)
      n_device = 1;
   else
      for (n_device = 0; n_device < i; n_device++) {
         CheckErc(RgParam(DEVICE_NAME_PARAM, n_device, &sd_param));
         erc = ParseFileSpec(0, sd_param.pb, sd_param.cb, FALSE, NULL,
                             &node_len, &text, &text_len, NULL, NULL, NULL,
                             &position_len, NULL, NULL, FALSE, 3);
         if (     erc != ercOK || node_len != 0 || position_len != 0)
            ErrorExit(ercBadDevSpec);
         for (j = 0; j < n_device; j++)
            if (     device_name[j][0] == text_len
                  && NlsULCmpB(NULL, &device_name[j][1], text,
                               text_len, &k) == ercOK
                  && k == 0xFFFF)
               ErrorExit(ercBadDevSpec);/* Duplicate device names verboten */
         memcpy(&device_name[n_device][1], text,
                device_name[n_device][0] = text_len);
      }
   if ((i = CSubparams(BUFFER_POOL_PARAM)) > 1)
      ErrorExit(ercSyntax);
   if (i == 0)
      buffer_pages = 128;		/* Default buffer pool size, 64 Kb */
   else {
      CheckErc(RgParam(BUFFER_POOL_PARAM, 0, &sd_param));
      if (sd_param.cb > 4)		/* 9,999 Kb, maximum */
         ErrorExit(ercSyntax);
      memcpy(text, sd_param.pb, sd_param.cb);
      text[sd_param.cb] = 0;
      for (j = 0; j <sd_param.cb; j++)
         if (!isdigit(text[j]))
            ErrorExit(ercSyntax);
      buffer_pages = (1024 / PAGE_SIZE) * atoi(text);
   }
   if ((i = CSubparams(QIC_SLOT_PARAM)) > 1)
      ErrorExit(ercSyntax);
   if (FSrp(NULL))
      if (i == 0)
         qic_slot = 0x77;		/* Usual position for QICI on SRP */
      else {
         CheckErc(RgParam(QIC_SLOT_PARAM, 0, &sd_param));
         if (sd_param.cb != 2)		/* Must be two hexadecimal digits */
            ErrorExit(ercSyntax);
         text[0] = sd_param.pb[0];
         text[1] = sd_param.pb[1];
         if (   (text[0] == '7' && text[1] == '7')
             || (text[0] >= '2' && text[0] <= '6' && text[1] == '0'))
            qic_slot = ((text[0] & 0x0F) << 4) | (text[1] & 0x0F);
         else
            ErrorExit(ercSyntax);
      }
   else if (i != 0)
      ErrorExit(ercSyntax);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 On the SRP/XE 530, consult [Sys]<Sys>SrpConfig.Sys to ascertain the names and
 physical addressing information of the sequential access devices.  These are
 then matched against the list of device names provided in the VLPB so this
 instance of the service can decide which devices it should control.  Some
 future release of workstation CTOS will probably implement the same
 functionality for multiple sequential access devices on workstations, too. */

private void scan_config_file(void) {

#define MASTER_FP "MasterFp"

   unsigned erc, i, j, k;
   static unsigned nls_device_class[] = {NLS_QIC36_CLASS, NLS_HALF_INCH_CLASS,
                                         NLS_SCSI_CLASS};
   struct {
      unsigned ctrl_code;
      unsigned reserved;
      char device_name[MAX_DEVICE_LENGTH + 1];
   } query_control;
   char scsi_manager_name[MAX_DEVICE_LENGTH + 1];
   sdType *sd_device_class;
   struct {
      char device_name[MAX_DEVICE_LENGTH + 1];
      char device_class[MAX_DEVICE_LENGTH + 1];
      char password [MAX_PASSWORD_LENGTH + 1];
      char processor_name[4];
      char scsi_adapter;
      char target_id;
      char lun;
      char unit;
   } seq_config;

   register_exit_procedure(delete_route_table_entries);
   for (i = 0; i < n_device; i++) {
      memset(&query_control, 0, sizeof(query_control));
      query_control.ctrl_code = 1;
      memcpy(query_control.device_name, device_name[i], device_name[i][0] + 1);
      if (     ConfigurationQuery(MASTER_FP, last(MASTER_FP), &query_control,
                                  sizeof(query_control), &seq_config,
                                  sizeof(seq_config), &j) != ercOK
            || j != sizeof(seq_config))
         exit_with_msg(ercInitFileFormat, 0);
      if (seq_config.device_name[0] == 0xFF)
         seq_config.device_name[0] = 0;
      if (seq_config.device_class[0] == 0xFF)
         seq_config.device_class[0] = 0;
      if (seq_config.password[0] == 0xFF)
         seq_config.password[0] = 0;
      if (seq_config.scsi_adapter == 0xFF)
         seq_config.scsi_adapter = 0;
      if (seq_config.target_id == 0xFF)
         seq_config.target_id = 0;
      if (seq_config.lun == 0xFF)
         seq_config.lun = 0;
      if (seq_config.unit == 0xFF)
         seq_config.unit = 0;
      for (j = 0; j <= last(nls_device_class); j++) {
         sd_device_class = standard_msg(nls_device_class[j]);
         if (     seq_config.device_class[0] == sd_device_class->cb
               && NlsULCmpB(NULL, &seq_config.device_class[1],
                            sd_device_class->pb, sd_device_class->cb,
                            &k) == ercOK
               && k == 0xFFFF) {
            switch (j) {
               case 0:		/* QIC-02 interface to QIC-36 device */
                  if (device_open[srp_QIC_class] == 0)
                     exit_with_msg(ercInitFileFormat, 0);
                  allocate_dcb(&dcb_array[n_dcb++], seq_config.device_name,
                               seq_config.password, srp_QIC_class, qic_slot);
                  break;

               case 1:		/* Pertec interface to half-inch tape */
                  if (device_open[Pertec_class] == 0)
                     exit_with_msg(ercInitFileFormat, 0);
                  allocate_dcb(&dcb_array[n_dcb++], seq_config.device_name,
                               seq_config.password, Pertec_class,
                               seq_config.unit);
                  break;

               case 2:		/* SCSI interface to device */
                  if (device_open[SCSI_class] == 0)
                     exit_with_msg(ercInitFileFormat, 0);
                  if (ScsiManagerNameQuery(seq_config.processor_name,
                                           sizeof(seq_config.processor_name),
                                           scsi_manager_name,
                                           sizeof(scsi_manager_name)) != ercOK)
                     exit_with_msg(ercInitFileFormat, 0);
                  allocate_scsi_dcb((dcb_scsi_type **) &dcb_array[n_dcb++],
                                    seq_config.device_name,
                                    seq_config.password,
                                    seq_config.scsi_adapter,
                                    seq_config.target_id, seq_config.lun,
                                    scsi_manager_name);
                  break;
            }
            if ((erc = update_route_table(dcb_array[n_dcb - 1],
                                          ROUTE_TABLE_ADD)) != ercOK) {
               n_dcb--;		/* Don't need to remove the Dcb that failed! */
               exit_with_msg(erc, 0);
            }
            break;		/* Back to outer loop of device names */
         }
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 At this point, the parameters supplied by the user in the command form (or in
 the JCL file) have passed muster.  Now take a look at the actual hardware
 that is present on this workstation and determine how many sequential access
 devices are present.  */

private void scan_hardware(void) {

   unsigned i = 0, module = 2, module_id, scsi_adapter = 0, scsi_info_size;
   scsi_info_type scsi_info[7];

   while (GetModuleId(XBUS, module++, &module_id) == ercOK)
      if ((module_id == FLEMINGTON_QIC || module_id == NGEN_QIC) && i++ != 0)
         exit_with_msg(ercWrongHardware, NLS_WRONG_HARDWARE);
   GetModuleId(XBUS, module = 1, &module_id);
   if (     module_id == SERIES_286I || module_id == SERIES_386I
         || module_id == SUPERGEN) {
      if (GetScsiInfo(0, 0, 0, &scsi_info, sizeof(scsi_info),
                      &scsi_info_size) == ercOK)
         for (i = 0;    i < (scsi_info_size / sizeof(*scsi_info))
                     && n_dcb < n_device; i++)
            if (scsi_info[i].device_type == SEQUENTIAL_ACCESS_DEVICE)
               allocate_scsi_dcb((dcb_scsi_type **) &dcb_array[n_dcb],
                                 device_name[n_dcb++], NULL, scsi_adapter,
                                 scsi_info[i].scsi_id, scsi_info[i].lun, NULL);
      scsi_adapter++;
   }
   while (n_dcb < n_device && GetModuleId(XBUS, ++module, &module_id) == ercOK)
      switch (module_id >> 8) {
         case (FLEMINGTON_QIC >> 8):
            allocate_dcb(&dcb_array[n_dcb], device_name[n_dcb++], NULL,
                         flemington_QIC_class, (module - 1) << 8);
            break;

         case (NGEN_QIC >> 8):
            allocate_dcb(&dcb_array[n_dcb], device_name[n_dcb++], NULL,
                         ngen_QIC_class, (module - 1) << 8);
            break;

         case (NGEN_SCSI >> 8):
         case (NGEN_SCSI_PLUS_FLOPPY >> 8):
            if (GetScsiInfo(module - 1, 0, 0, &scsi_info, sizeof(scsi_info),
                            &scsi_info_size) == ercOK)
               for (i = 0;    i < (scsi_info_size / sizeof(*scsi_info))
                           && n_dcb < n_device; i++)
                  if (scsi_info[i].device_type == SEQUENTIAL_ACCESS_DEVICE)
                     allocate_scsi_dcb((dcb_scsi_type **) &dcb_array[n_dcb],
                                       device_name[n_dcb++], NULL,
                                       scsi_adapter, scsi_info[i].scsi_id,
                                       scsi_info[i].lun, NULL);
            scsi_adapter++;
            break;

      }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When valid devices have been recognized (on either the SRP or a workstation),
 allocate a device control block and copy into it some of the pertinent
 information for its subsequent use by the Sequential Access service.  Note
 that SCSI devices require a different set of information than the eariler,
 more hardware-oriented interfaces. */

private void allocate_dcb(dcb_type **dcb, char device_name[],
                          char password[], char device_class,
                          unsigned device_info) {

   unsigned erc, rq_info;

   if (     (   device_class == flemington_QIC_class
             || device_class == ngen_QIC_class)
         && (   (   QueryRequestInfo(SET_XBUS_MISR_RQ_CODE, &rq_info,
                                     sizeof(rq_info)) != ercOK
                 && QueryRequestInfo(OLD_SET_XBUS_MISR_RQ_CODE, &rq_info,
                                     sizeof(rq_info)) != ercOK)
             || rq_info == 0))
      exit_with_msg(ercNoXbif, NLS_NO_XBIF);
   if ((erc = AllocMemorySL(sizeof(dcb_type), dcb)) != ercOK)
      exit_with_msg(erc, 0);
   memset(*dcb, 0, sizeof(dcb_type));
   memcpy((*dcb)->device_name, device_name, device_name[0] + 1);
   if (password != NULL)
      memcpy((*dcb)->device_password, password, password[0] + 1);
   if (device_init[(*dcb)->device_class = device_class] != 0)
      device_init[(*dcb)->device_class](*dcb, device_info);

}

private void allocate_scsi_dcb(dcb_scsi_type **scsi_dcb, char device_name[],
                               char password[], char host_adapter,
                               char scsi_id, char lun,
                               char scsi_manager_name[]) {

   static char direct_io_name[2] = "Z0";
   unsigned erc;

   if ((erc = AllocMemorySL(sizeof(dcb_type), scsi_dcb)) != ercOK)
      exit_with_msg(erc, 0);
   memset(*scsi_dcb, 0, sizeof(dcb_type));
   (*scsi_dcb)->device_class = SCSI_class;
   memcpy((*scsi_dcb)->device_name, device_name, device_name[0] + 1);
   if (password != NULL)
      memcpy((*scsi_dcb)->device_password, password, password[0] + 1);
   (*scsi_dcb)->host_adapter = host_adapter;
   (*scsi_dcb)->scsi_id = scsi_id;
   (*scsi_dcb)->lun = lun;
   if (internal_os_version >= 0x0C00) {
      if (scsi_manager_name != NULL)
         memcpy((*scsi_dcb)->scsi_manager_name, scsi_manager_name,
                scsi_manager_name[0] + 1);
   } else {
      if (     internal_os_version == 0x0B04	/* SCSI devices in BTOS II.. */
            || internal_os_version == 0x0B07)
         direct_io_name[0] = 'S';		/* ...named S0, S1 et cetera */
      memcpy((*scsi_dcb)->direct_io_name, direct_io_name,
             sizeof((*scsi_dcb)->direct_io_name));
      direct_io_name[1]++;			/* Bump to next device name */
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 One of the preliminary consistency checks---see if all needed request codes
 are defined!  Then see if anybody is already serving them.  Either condition
 prevents us from going any further. */

private void verify_requests(void) {

   unsigned erc, i;

   for (i = 0; i <= seq_access_rq_code_last; i++) {
      if ((erc = QueryRequestInfo(seq_access_rq_code[i], &save_rq_info[i],
                                  sizeof(save_rq_info[i]))) != ercOK)
         exit_with_msg(erc, 0);
      if (save_rq_info[i] != 0 && save_rq_info[i] != CLUSTER_AGENT_SERV_EXCH)
         exit_with_msg(ercOK, NLS_ALREADY_INSTALLED);
   }

}

/*-----------------------------------------------------------------------------
 All the preliminary consistency checks have been passed, Dcb's have been
 allocated, devices initialized and so on.  Now signal our intention to serve
 the requests */

private void serve_requests(void) {

   unsigned erc, i;

   for (i = 0; i <= seq_access_rq_code_last; i++) {
      if ((erc = QueryRequestInfo(seq_access_rq_code[i], &save_rq_info[i],
                                  sizeof(save_rq_info[i]))) != ercOK)
         exit_with_msg(erc, 0);
      if (save_rq_info[i] != 0 && save_rq_info[i] != CLUSTER_AGENT_SERV_EXCH)
         exit_with_msg(ercOK, NLS_ALREADY_INSTALLED);
   }
   for (i = 0; i <= seq_access_rq_code_last; i++)
      if ((erc = ServeRq(seq_access_rq_code[i], serv_exch)) != ercOK)
         exit_with_msg(erc, 0);

}

/*-----------------------------------------------------------------------------
 Nearly done!  Just change our priority and make permanent via a
 ConvertToSys.  The installation complete message displays only if the Exit()
 procedure is successful (and at that point we don't care because Exit has
 become a NOP). */

private void convert_to_sys(void) {

   unsigned erc;
   sdType *sd_msg;

   if ((erc = ChangePriority(SEQ_SERVICE_PRIORITY)) != ercOK)
      exit_with_msg(erc, 0);
   sd_msg = standard_msg(NLS_INSTALLED_OK);
   SetMsgRet(sd_msg->pb, sd_msg->cb);
   if ((erc = ConvertToSys()) != ercOK)
      exit_with_msg(erc, (erc == ercBadUserNum) ? NLS_CANT_INSTALL : 0);
   Exit();
   GetUserNumber(&own_user_num);
   sd_msg = standard_msg(NLS_PARTITION_NAME);
   SetPartitionName(0, sd_msg->pb, sd_msg->cb);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 If, during installation on the SRP (or any other future product that
 maintains a routing table), an error occurs after route table updates have
 been issued, these must be backed out---or else a subsequent installation
 attempt will collide with its own footprints from the earlier failure!  This
 procedure is registered as an "exit procedure" and is invoked if
 exit_with_msg() is called. */

void delete_route_table_entries(void) {

   unsigned i;

   for (i = 0; i < n_dcb; i++)
      update_route_table(dcb_array[i], ROUTE_TABLE_DELETE);

}
