/*
	 sbus.c
*/
#include "types.h"
#include "menu.h"
#include "routines.h"
#include "globl.h"
#include "spm.h"
#include "global.h"
#include "rtc.h"
#include "disp.h"
#include "ipcc.h"
#include "dev.h"
#include "novram.h"
#include "misc.h"
#include "ioa0.h"
#include "icb_config.h"
#include "iosba_addr.h"

#define	MOD_FLAG	1
#define	IF_FLAG		2

extern  char	emulate;
extern  char	ignore_it;
extern  char	got_berr;
extern	uchar	myslot;
extern	uchar	css_slot;
extern	uchar	sub_slot;
extern	uchar	phys_dev;
extern	uchar	log_dev;
extern	struct	slotst sf;
extern	uchar	bdhere[];	 /* slot id for CSS type boards */
extern	struct	iosb_config	ios_conf[][MAX_CSS_SLOT];


/*------------------------------------------------------------------------------
		iready: increase ready count in the arbiter for the SPM
                        toggle ready bit
-----------------------------------------------------------------------------*/
iready()
{
	*WRCNTL1 &= ~WR1_IRDY;	  /*  1 - 0 edge causes incr ready count  */
	*WRCNTL1 |= WR1_IRDY;
}

/*------------------------------------------------------------------------------
		rready() :  reset ready count in the arbiter for the SPM
                            toggle reset bit
-----------------------------------------------------------------------------*/
rready()
{

	*WRCNTL1 &= ~WR1_RRDY;	
	*WRCNTL1 |= WR1_RRDY;
}


/*------------------------------------------------------------------------------
-----------------------------------------------------------------------------*/
css_reset()
{

	*WRCNTL1 &= ~WR1_CSSRST;		/* and assert it. */
	wait_cnt(3000);
	*WRCNTL1 |= WR1_CSSRST;			/* and assert it. */
	wait_cnt(3000);
	rready();
	iready();
}


int
init_css()
{
	register uint  inv_slot;

	uint	i;

	myslot = (*STATUSREG & STAT_SLOTMASK) >> STAT_SLOTSH;
	inv_slot = ~myslot & 0x0f; 
	*WRCNTL0 = (myslot << 8) | (myslot << 12) | (myslot << 16) | 
			(inv_slot << 20);
	*WRCNTL1 = WR1_FRC_RST_ACK | WR1_IRDY | WR1_FMOD
			| WR1_FDEST | WR1_DATYPE | WR1_FINTREC | WR1_DREQ
			| WR1_IDLERST | WR1_RSTREC | WR1_TPOL 
			| WR1_UFTYPE | WR1_ID_RST | WR1_4BYTE_RD
			| WR1_MOTOR | WR1_FDACK | WR1_FRES | WR1_TP
			| (myslot << WR1_DEST_SH)
			| WR1_POL; 
	for (i = 0; i != 0x6000;i++);
#ifdef	REMOVE
	wait_cnt(400);
#endif
	
	*WRCNTL1 |= WR1_RRDY;
	*WRCNTL1 |= WR1_CSSRST;
	*WRCNTL0 |= WR0_FRC_INH7;

}


int
css_bus_init()
{
	
	init_css();

	dram();

	pwr_nov_tst();	/* Test it here.....why not... */

	rready();

	iready();

	ctinten();
#ifdef	REMOVE
	init_css();
	iready();

	dram();
	ctinten();
#endif
}




/*
**	cssmap
*/

void
cssmap(mapnum,slotnum,slotadd)
register uint mapnum;
register uchar slotnum,slotadd;
{

	register uchar *map_ptr;
 
	map_ptr=(uchar*)((uint)MAPBASE | (mapnum << 28));
	*map_ptr = (slotnum << 4) | slotadd;
}

/*
**	cssmap_save
*/

uchar
cssmap_save (mapnum)
register  uchar  mapnum;
{
	register uchar	*map_ptr;
	
	map_ptr = (uchar *)((unsigned)MAPBASE | (mapnum << 28));
	return (*(uchar *)map_ptr);
}

/* 
**	cssmap_restore
*/
uint
cssmap_restore (mapnum, map_value)
register  uchar  mapnum;
register  uchar  map_value;
{
	register uchar *map_ptr;
	if (mapnum > NUM_CSS_MAPS)
		return (1);

	map_ptr = (uchar *)((unsigned)MAPBASE | (mapnum << 28));
	*map_ptr = (uchar)map_value;
	return (0);
}


/*------------------------------------------------------------------------------
	cpu_idle : If 'flg' is a non zero then use 'slt' for target.
		   else use the table and do all the found CPUs for target.
		   Of course the idea is to 'idle the cpus', but you already
		   know that... right!!!.
------------------------------------------------------------------------------*/
cpu_idle(flg,slt)
char flg,slt;
{
	int i;

	if(!flg) {
		for(i=0;i<Sbus_Num_Slot;i++) {		/* loop for cpu's. */
			if(bdhere[i] == CPUHERE) {	/* if a cpu, send SPM
							   slot and the idle
							   command.   */
				ipccsend(i, IPCC_SM_SLOT, myslot);
				ipccsend(i, IPCC_PM_PUTR, -1);
			}
		}
	}
	else {
		if((slt>15) || (slt<0) || (bdhere[slt] !=
			CPUHERE)) {

			printf("\nInvalid PM slot number.\n");
			return(1);
		}	
		ipccsend(slt, IPCC_SM_SLOT, myslot);	/* SPM slot. */
		ipccsend(slt, IPCC_PM_PUTR, -1);	/* idle command. */
	}
	return(0);
}


void
cmd_write(slot, command)
uchar  slot;
uint   command;
{
	uint	wrcntl1_save;
	uint	wrcntl0_save;
	uchar   saved_map = cssmap_save(0);
	int	*cmd_ptr = (int *) 0x8ffffffc;	/* ptr for sending commands */

	int	i;
/*
**	As a utility, this routine needs to restore these 
**	global variables to their previous values
*/
	char	emulate_save = emulate;
	char	ignore_save = ignore_it;
	char	berr_save = got_berr;

	cssmap(0, slot, (uchar)0x0f);
	_cache(0);		/* disable 68020 cache to bypass 68020 bug */
	emulate = ignore_it = 1;    /* ignore bus errors */
	wrcntl1_save = *WRCNTL1;
	wrcntl0_save = *WRCNTL0;

	*WRCNTL1 &= ~(WR1_FRES | WR1_UFTYPE); /* clear the force response bit */

	/* disable interface */
	*WRCNTL0 = (*WRCNTL0 & ~WR0_FTYPE_MASK) | command; 

	*cmd_ptr = 0;	                      /* send any data, it's ignored */

	for (i = 0; i != 1000; i++);
#ifdef	REMOVE
	wait_cnt(10);
#endif

	*WRCNTL1 = wrcntl1_save; 	/* restore wrcntl regs  */
	*WRCNTL0 = wrcntl0_save; 	/* restore wrcntl regs  */

	emulate = emulate_save;
	ignore_it = ignore_save;
	got_berr = berr_save;
	_cache(1);		/* enable 68020 cache to bypass 68020 bug */
	cssmap_restore (0, saved_map);
}


/*------------------------------------------------------------------------------
	ifenable :	Enable the slot and interface to a slot.
			flg == 0:  enable module and i/f
			flg == 1:  enable module
			flg == 2:  enable i/f
------------------------------------------------------------------------------*/
ifenable(slot, flg)
uchar	slot;
uchar	flg;
{

	if (flg==0 || flg==2)  /* enable interface */
		cmd_write (slot, IF_ENABLE);

	if (flg==0 || flg==1)  /* enable module */
		cmd_write (slot, MODULE_ENABLE);

}

/*------------------------------------------------------------------------------
	ifdisable :	Disable the slot and interface to a slot.
			flg == 0:  disable module and i/f
			flg == 1:  disable module
			flg == 2:  disable i/f
------------------------------------------------------------------------------*/
ifdisable(slot, flg)
uchar	slot;
uchar	flg;
{

	if(flg==0 || flg==1)  /* enable module */
		cmd_write (slot, MODULE_DISABLE);

	if(flg==0 || flg==2)  /* enable interface */
		cmd_write (slot, IF_DISABLE);

}


int
cmd_cntl (comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	uchar	command = atox(comm_args[1]);
	uchar	c_slot = atoi(comm_args[2]);

	cmd_write (c_slot, command);
	return(0);
}
	
/*------------------------------------------------------------------------------
	slot_cntl:	
			comm_args[1] == all:  enable module and i/f
			comm_args[1] == mod:  enable module
			comm_args[1] == if:  enable i/f
------------------------------------------------------------------------------*/
int
slot_cntl (comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	
	register  uchar  enable;
	register  uchar  ctl_flag;

	if (*comm_str == 'e' || *comm_str == 'E')
		enable = 1;
	else
		enable = 0;

	switch (*comm_args[1]) {
		case 'a':
			ctl_flag = (MOD_FLAG | IF_FLAG);
			break;
		case 'i':
			ctl_flag = IF_FLAG;
			break;
		case 'm':
			ctl_flag = MOD_FLAG;
			break;
		default:
			printf ("%s:  Illegal argument %s, <all-if-mod>.\n",
				comm_args[0], comm_args[1]);
			return(0);
	}

	if (*comm_str == 'E' || *comm_str == 'D')
		return(define_slot (enable, ctl_flag, arg_cnt));
	else
		return(control_slot (enable, ctl_flag, arg_cnt));
}


/*
**	The parameter passing here is a little flakey, the slot(s) is in
**	a global variable comm_args, and as such is not passed, this
**	is not intuitive
*/
int
control_slot (enable, ctl_flag, arg_cnt)
uint	enable;
uint	ctl_flag;
uint	arg_cnt;
{

	uchar  c_slot = (uchar)atoi(comm_args[2]);
	uchar  s_slot;

	if (arg_cnt == 3) {
		s_slot = atoi(comm_args[3]);
		if (logiom(c_slot) == NOBOARD) {
			printf ("No IOM in slot %x (%d).\n", c_slot, c_slot);
			return(1);
		}
			
		if ((uchar)test_for_adapter(c_slot) != TYPE_IOSBA) {
			printf ("Slot control for CSS and IOSB.\n");
			return(1);
		}

		if (enable) {
			if (ctl_flag & IF_FLAG)
				io_control (INTERFACE_ENABLE, c_slot, s_slot);
			if (ctl_flag & MOD_FLAG)
				io_control (MOD_ENABLE, c_slot, s_slot);
		}
		else {
			if (ctl_flag & MOD_FLAG)
				io_control (MOD_DISABLE, c_slot, s_slot);
			if (ctl_flag & IF_FLAG)
				io_control (INTERFACE_DISABLE, c_slot, s_slot);
		}	
	}
	else {
		if (enable) {
			if (ctl_flag & IF_FLAG)
				cmd_write (c_slot, IF_ENABLE);
			if (ctl_flag & MOD_FLAG)
				cmd_write (c_slot, MODULE_ENABLE);
		}
		else {
			if (ctl_flag & MOD_FLAG)
				cmd_write (c_slot, MODULE_DISABLE);
			if (ctl_flag & IF_FLAG)
				cmd_write (c_slot, IF_DISABLE);
		}	
	}
	return(0);
}

int
define_slot (enable, ctl_flag, arg_cnt)
uint	enable;
uint	ctl_flag;
uint	arg_cnt;
{
	uchar	bd_id;
	uchar	c_slot;
	uchar	s_slot;
	uchar	iom_num;

	bd_id = atox(comm_args[2]);
	c_slot = atoi(comm_args[3]);
	if (arg_cnt == 4) {
		s_slot = atoi(comm_args[4]);
		if ((iom_num = logiom(c_slot)) == NOBOARD) {
			printf ("No IOM in slot %x (%d).\n", c_slot, c_slot);
			return(1);
		}
			
		if ((uchar)test_for_adapter(c_slot) != TYPE_IOSBA) {
			printf ("Slot control for CSS and IOSB.\n");
			return(1);
		}

		if (enable) {
			if (ctl_flag & IF_FLAG)
				io_control (INTERFACE_ENABLE, c_slot, s_slot);
			if (ctl_flag & MOD_FLAG)
				io_control (MOD_ENABLE, c_slot, s_slot);
			ios_conf[iom_num][s_slot].slot_id = (uchar)bd_id;
		}
		else {
			if (ctl_flag & MOD_FLAG)
				io_control (MOD_DISABLE, c_slot, s_slot);
			if (ctl_flag & IF_FLAG)
				io_control (INTERFACE_DISABLE, c_slot, s_slot);
			ios_conf[iom_num][s_slot].slot_id = (uchar)0xFF;
		}

	}
	else {
		if (enable) {
			if (ctl_flag & IF_FLAG)
				cmd_write (c_slot, IF_ENABLE);
			if (ctl_flag & MOD_FLAG)
				cmd_write (c_slot, MODULE_ENABLE);
			bdhere[c_slot] = bd_id;
			sf.on[c_slot] = 1;
		}
		else {
			if (ctl_flag & MOD_FLAG)
				cmd_write (c_slot, MODULE_DISABLE);
			if (ctl_flag & IF_FLAG)
				cmd_write (c_slot, IF_DISABLE);
			bdhere[c_slot] = 0xFF;
			sf.on[c_slot] = 0;
		}
	}	
	return(0);
}


