
/*------------------------------------------------------------------------------
	margin.c
------------------------------------------------------------------------------*/

#include "types.h"
#include "spm.h"
#include "rwi.h"
#include "rwicio.h"
#include "novram.h"
#include "rtc.h"
#include "global.h"

#define	MARGIN_HIGH	1
#define MARGIN_LOW	2
#define	MARGIN_NORM	3

#define	RELAY_ON	1
#define	RELAY_OFF	2

#define	MS_LIMIT	10000

extern char *gets ();

static char high_margin[] = {
	PS_0_MARGIN_POS,
	PS_1_MARGIN_POS,
	PS_2_MARGIN_POS,
	PS_3_MARGIN_POS,
	PS_4_MARGIN_POS
	};

static char low_margin[] = {
	PS_0_MARGIN_NEG,
	PS_1_MARGIN_NEG,
	PS_2_MARGIN_NEG,
	PS_3_MARGIN_NEG,
	PS_4_MARGIN_NEG
	};

static char ssrelay[] = {
	SSR_0_ON,
	SSR_1_ON,
	SSR_2_ON,
	SSR_3_ON,
	SSR_4_ON,
	SSR_5_ON,
	SSR_6_ON
	};

/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/

int
margin(comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	int	ps_num;

	if (arg_cnt == 1) {
		display_cios_status();
		return(0);
	}

	/*  This is the cabinet number, it doesn't matter whether
	**  this is decimal or hex in systems less than 10 cabinets
	*/
	ps_num = atoi(comm_args[1]);
	switch (*comm_args[2]) {
		case 'l':
		case 'L':
			do_margin (ps_num, MARGIN_LOW);
			break;
		case 'h':
		case 'H':
			do_margin (ps_num, MARGIN_HIGH);
			break;
		case 'n':
		case 'N':
			do_margin (ps_num, MARGIN_NORM);
			break;
		default:
			printf ("Bad margin option: %s, no change performed.\n",
				comm_args[2]);
			return(1);
	}
	return(0);
}


int
do_margin (ps_num, margin_type)
int	ps_num;
int	margin_type;
{
	struct	cio *cio_ptr = CIO_1;
	int	lport;

	lport = (ps_num > 2) ? PORTC : PORTB;
	switch (margin_type) {
		case MARGIN_NORM:
			if (cio_ptr->portdata[PORTB].reg & high_margin[ps_num]) 
				cio_ptr->portdata[PORTB].reg &= 
						(char)(~high_margin[ps_num]);
			wait_cnt(500);
			if (cio_ptr->portdata[lport].reg & low_margin[ps_num]) 
				cio_ptr->portdata[lport].reg &= 
						(char)(~low_margin[ps_num]);
			break;

		case MARGIN_HIGH:
			if (cio_ptr->portdata[lport].reg & low_margin[ps_num]) 
				cio_ptr->portdata[lport].reg &= 
						(char)(~low_margin[ps_num]);
			wait_cnt(500);
			cio_ptr->portdata[PORTB].reg |= high_margin[ps_num];
			break;

		case MARGIN_LOW:
			if (cio_ptr->portdata[PORTB].reg & high_margin[ps_num]) 
				cio_ptr->portdata[PORTB].reg &= 
						(char)(~high_margin[ps_num]);
			wait_cnt(500);
			cio_ptr->portdata[lport].reg |= low_margin[ps_num];
			break;
	}
	return(0);
}	

/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/
display_cios_status()  
{
	
	uchar porta_0;
	uchar portb_0;
	uchar portc_0;
	uchar porta_1;
	uchar portb_1;
	uchar portc_1;
	int i;
	int j;
	char ssr[4];
	char marg_plus[4];
	char marg_minus[4];
	char ps_pf[4];
	char version[12];
	char on_sw[4];
	char off_sw[4];
	char sys_on_lit[4];
	char warning_lit[4];
	char ups_off[4];
	char ups_ac_fail[4];
	char ups_low_battery[4];
	
	int num_cabinets;
	int num_margin;
	int num_power;
	int short_count;
	int long_count;

	struct	cio	*cio_0_ptr = CIO_0;
	struct	cio	*cio_1_ptr = CIO_1;

	/* 
	 * Read all cio port in
	 */
	porta_0 = cio_0_ptr->portdata[PORTA].reg;
	portb_0 = cio_0_ptr->portdata[PORTB].reg;
	portc_0 = cio_0_ptr->portdata[PORTC].reg;
	porta_1 = cio_1_ptr->portdata[PORTA].reg;
	portb_1 = cio_1_ptr->portdata[PORTB].reg;
	portc_1 = cio_1_ptr->portdata[PORTC].reg;
	
	if (ACRW_HERE) {	/* Mask off unwanted data */
		porta_0 &= (unsigned char)0xE3;
		portb_0 &= (unsigned char)0x83;
		portb_1 &= (unsigned char)0x21;
		portc_1 &= (unsigned char)0x0C;

		num_cabinets = 2;
		num_margin = 1;
		num_power = 2;
		long_count = 28;
		short_count = 28;
	}
	else {
		num_cabinets = CAB_PER_SYS;
		num_margin = PS_PER_SYS;
		num_power = PS_PER_SYS;
		long_count = 73;
		short_count = 55;
	}

	switch (porta_1 & 0x07) {
		case REV1A:
			strcpy(version,"RWI Rev. 1A");
			break;
	        case REV2B:
			strcpy(version,"RWI Rev. 2B");
			break;
		case ACRW:
			strcpy(version,"ACRW Rev. 2");
			break;
	} 

	if ( portc_0 & 0x01 ) {
		strcpy(on_sw,"ON");
	} else {
		strcpy(on_sw,"OFF");
	}

	if ( portc_0 & 0x02 ) {
		strcpy(off_sw,"ON");
	} else {
		strcpy(off_sw,"OFF");
	}

	if ( portc_1 & 0x04 ) {
		strcpy(sys_on_lit,"OFF");
	} else {
		strcpy(sys_on_lit,"ON");
	}

	if ( portc_1 & 0x08 ) {
		strcpy(warning_lit,"OFF");
	} else {
		strcpy(warning_lit,"ON");
	}


	if ( portb_0 & 0x80 ) {
		strcpy(ups_off,"OFF");
	} else {
		strcpy(ups_off,"ON");
	}
	
	if ( porta_0 & 0x20 ) {
		strcpy(ups_ac_fail,"ON");
	} else {
		strcpy(ups_ac_fail,"OFF");
	}

	if ( porta_0 & 0x40 ) {
		strcpy(ups_low_battery,"ON");
	} else {
		strcpy(ups_low_battery,"OFF");
	}
	printf("\f\t\tStatus of CIO Output Bits %s\n",version);
	printf("\n");
	printf("\tON.SW            %3s   OFF.SW          %3s\n",on_sw,off_sw);
	printf("\tSYS.ON.LITE:     %3s   WARNING.LITE:   %3s\n",sys_on_lit,warning_lit);
	printf("\tUPS.OFF:         %3s   UPS.AC.FAIL:    %3s\n",ups_off,ups_ac_fail);
	printf("\tUPS.LOW.BATTERY: %3s\n ",ups_low_battery);
	printf("\n");
	printf("CABINET     ");
	for(i=0; i<num_cabinets; i++) 
		printf("    %d    ",i);
	printf("\n");

	printf("%c",'+');
	pline ('-', long_count);
	printf("%c",'+');
	printf("\n");

	printf("%s","|          ");
	for (j=0; j<num_cabinets; j++) 
		printf("%s","|        ");
	printf("%s","|");
	printf("\n");

	printf("%s","| SSR      ");
	for (j=0; j<num_cabinets; j++) {
		if( portb_0 & (0x01 << j)) {
			strcpy(ssr,"ON");
		} else {
			strcpy(ssr,"OFF");
		}
		printf("|  %3s   ",ssr); 
	}
	printf("%s","|");
	printf("\n");

	printf("%s","|          ");
	for (j=0; j<num_cabinets; j++) 
		printf("%s","|        ");
	printf("%s","|");
	printf("\n");

	printf("%c",'+');
	pline ('-', long_count);
	printf("%c",'+');
	printf("\n");

	printf("%s","|          ");
	for (j=0; j<num_margin; j++) 
		printf("%s","|        ");
	printf("%s","|");
	printf("\n");

	printf("%s","| MARGIN + ");
	for (j=0; j<num_margin; j++) {
		if( portb_1 & (0x01 << j)) {
			strcpy(marg_plus,"ON");
		} else {
			strcpy(marg_plus,"OFF");
		}
		printf("|  %3s   ",marg_plus); 
	}
	printf("%s","|");
	printf("\n");

	printf("%s","|          ");
	for (j=0; j<num_margin; j++) 
		printf("%s","|        ");
	printf("%s","|");
	printf("\n");

	printf("%s","| MARGIN - ");
	for (j=0; j<num_margin; j++) {
		if( j < 3 ) {
			if (portb_1 &( 0x01 << j + 5)) {
				strcpy(marg_minus,"ON");
			} else {
				strcpy(marg_minus,"OFF");
			}
		} else {
			if (portc_1 &( 0x01 << j - 3)) {
				strcpy(marg_minus,"ON");
			} else {
				strcpy(marg_minus,"OFF");
			}
		}
		printf("|  %3s   ",marg_minus); 
	} 
	printf("%s","|");
	printf("\n");

	if (ACRW_HERE) {
		printf("%s","|          ");
		for (j=0; j<num_margin; j++) 
			printf("%s","|        ");
		printf("%s","|");
		printf("\n");

		printf("%c",'+');
		pline ('-', short_count);
		printf("%c",'+');
		printf("\n");
	}

	printf("%s","|          ");
	for (j=0; j<num_power; j++) 
		printf("%s","|        ");
	printf("%s","|");
	printf("\n");

	printf("%s","| PS.PF    ");
	for (j=0; j<num_power; j++) { 
		if( porta_0 & (0x01 << j)) {
			strcpy(ps_pf,"ON");
		} else {
			strcpy(ps_pf,"OFF");
		}
		printf("|  %3s   ",ps_pf); 
	}
	printf("%s","|");
	printf("\n");

	printf("%s","|          ");
	for (j=0; j<num_power; j++) 
		printf("%s","|        ");
	printf("%s","|");
	printf("\n");

	printf("%c",'+');
	pline ('-', short_count);
	printf("%c",'+');
	printf("\n");
}


/*------------------------------------------------------------------------------
	n_ps_m : Normalize Power Supply Margins.
------------------------------------------------------------------------------*/
n_ps_m()
{
	struct	cio  *cio_1_ptr = CIO_1;

	cio_1_ptr->portdata[PORTB].reg = 0x00;	/* initial data for port B*/
	cio_1_ptr->portdata[PORTC].reg &= 0x0C;	/* initial data for port C */
}

/*------------------------------------------------------------------------------

------------------------------------------------------------------------------*/

int
relay(comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	int	ps_num;

	if (arg_cnt == 1) {
		display_cios_status();
		return(0);
	}

	/*  This is the cabinet number, it doesn't matter whether
	**  this is decimal or hex in systems less than 10 cabinets
	*/
	ps_num = atoi(comm_args[1]);

	if ((strcmp(comm_args[2],"on") == 0) || 
				(strcmp(comm_args[2],"ON") == 0)) {
		do_relay (ps_num, RELAY_ON);
		return(0);
	}
	
	if ((strcmp(comm_args[2],"off") == 0) || 
				(strcmp(comm_args[2],"OFF") == 0)) {
		do_relay (ps_num, RELAY_OFF);
		return(0);
	}

	return(1);
}


int
do_relay (ps_num, relay_cmd)
int	ps_num;
int	relay_cmd;
{

	struct	cio  *cio_ptr = CIO_0;

	if (relay_cmd == RELAY_ON) {
		cio_ptr->portdata[PORTB].reg |= (char)ssrelay[ps_num];
		return(0);
	}

	if (relay_cmd == RELAY_OFF) {
		cio_ptr->portdata[PORTB].reg &= (char)(~ssrelay[ps_num]);
		return(0);
	}

	return(1);
}

