
#include  "types.h"
#include  "iopm_comm.h"
#include  "spm.h"
#include  "novram.h"
#include  "dev.h"
#include  "sdk_disk.h"
#include  "disp.h"
#include  "global.h"
#include  "icb_config.h"
#include  "iosba_addr.h"
#include  "iopm_def.h"
#include  "scsi_comm.h"

#define		BLK_SIZE	0x400		/* 1K block */
#define		MAGIC_TIMEOUT	100		

#define		MAX_MEM_ERR	16

#define		IOPM_WRITES	8
#define		IOPM_SETTLE	0x50000

#define		FAIL		1
#define		PASS		0

extern  uint    tst_pat[];
#define	TST_PAT_SIZ	32
#define	TEST_INCR	0x04030201
#define	IOPM_START_ADDR	0xA8000000
#define	IOPM_MEM_SIZE	0x00001000

#define IOPM_CMD_TIMEOUT	0xFFFF
#define	IOPM_DISK_TIMEOUT	5000
#define	IOPM_TAPE_TIMEOUT	50000
#define BLK_NUM_DELAY		500
#define	XFER_DELAY		50000
#define	VER_DELAY		50
#define TYPE_DELAY		5000
#define	WASTE_LIMIT		0xC00

extern	uchar	css_slot;
extern	uchar	sub_slot;
extern	uchar	phys_dev;
extern	uchar	log_dev;
extern	uint	boot_type;

extern	struct	slotst sf;

extern	char	got_berr;		/* Bus error flag */
extern	char	emulate;		/* Bus error emulation flag */
extern	char	ignore_it;		/* If set, ignore CSS bus errors. */

extern	uchar	bdhere[];
extern	struct	iosb_config	ios_conf[][MAX_CSS_SLOT];

uint get_iopm_blk_num ();
int xfer_iopm_dev ();
void show_iopm_error();

extern  void    fill32 ();
extern  int	check32 ();

int
iopm_bootable (c_slot, s_slot)
char c_slot;
char s_slot;
{
	uchar  device_id;
	uchar  sav_berr;
	char	emulate_save = emulate;
	char	ignore_save = ignore_it;

	if (s_slot == NO_SUB_SLOT)
		cssmap (MAP02, c_slot, 0x0F);
	else
		cssmap (MAP02, c_slot, s_slot);

	sav_berr = got_berr;
	got_berr = 0;
	emulate = ignore_it = 1;

	device_id = *(uchar *)IOPM_DEV_BD_ID;	/* assumes MAP02 used */

	emulate = emulate_save;
	ignore_it = ignore_save;
	if (got_berr) {
		got_berr = sav_berr;
		return(0);			/* not bootable */
	}
	got_berr = sav_berr;

	if (device_id & IOPM_BOOTABLE_DEV)
		return (1);			/* bootable device board */
	else
		return (0);			/* device board not bootable */
}

uint
get_iopm_blk_num (type)
char	type;
{

	uchar  sav_berr;
	char	emulate_save = emulate;
	char	ignore_save = ignore_it;

	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;

	sav_berr = got_berr;
	got_berr = 0;
	emulate = ignore_it = 1;

	if (sub_slot == NO_SUB_SLOT)
		cssmap (MAP02, css_slot, 0x0F);
	else
		cssmap (MAP02, css_slot, sub_slot);

	if (iopm_cmd->istatus != (uint)CSS_OWNED) {
		printf ("IOPM Stucture in unexpected state - not CSS\n");
		printf ("Expected %d; Got %d\n", CSS_OWNED, iopm_cmd->istatus);
		return(0xFFFFFFFF);
	}

	iopm_cmd->phys_dev = (uint)phys_dev;
	iopm_cmd->log_dev = (uint)log_dev;
	iopm_cmd->icmd = (uint)IOPM_BLOCK_NUM;

	if (do_iopm_cmd (BLK_NUM_DELAY) == FAIL) {
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return(0xFFFFFFFF);
	}
	got_berr = sav_berr;
	emulate = emulate_save;
	ignore_it = ignore_save;

	if (iopm_cmd->ret_code) {
		show_iopm_error(iopm_cmd->ret_code);
		return(0xFFFFFFFF);
	}

	if (type)		/* slice type is important */
		if (iopm_cmd->dev_type != type) {
			printf ("Slice type not as desired.\n");
			printf ("Expected %d, Got %d.\n", type, 
				iopm_cmd->dev_type);
			return(0xFFFFFFFF);
		}

	return(iopm_cmd->st_block);
}

int
xfer_iopm_dev (block, size, read)
uint	block;
uint	size;
uint	read;
{
	uchar  sav_berr;
	char	emulate_save = emulate;
	char	ignore_save = ignore_it;

	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;

	if (sub_slot == NO_SUB_SLOT)
		cssmap (MAP02, css_slot, 0x0F);
	else
		cssmap (MAP02, css_slot, sub_slot);

	sav_berr = got_berr;
	got_berr = 0;
	emulate = ignore_it = 1;

	if (iopm_cmd->istatus != (uint)CSS_OWNED) {
		printf ("IOPM Stucture in unexpected state - not CSS\n");
		printf ("Expected %d; Got %d\n", CSS_OWNED, iopm_cmd->istatus);
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return (1);
	}

	iopm_cmd->phys_dev = (uint)phys_dev;
	iopm_cmd->st_block = block;
	iopm_cmd->num_block = size;
	if (read)
		iopm_cmd->icmd = (uint)IOPM_R_DEV;
	else
		iopm_cmd->icmd = (uint)IOPM_W_DEV;

	if (do_iopm_cmd (XFER_DELAY) == FAIL) {
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return(FAIL);
	}

	got_berr = sav_berr;
	emulate = emulate_save;
	ignore_it = ignore_save;
	return(iopm_cmd->ret_code);
}

int
get_iopm_dev_type ()
{
	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;
	SCSI_COMM_PTR  scsi_cmd;
	uint	sav_berr;
	char	emulate_save = emulate;
	char	ignore_save = ignore_it;
	uint	version;

	if (sub_slot == NO_SUB_SLOT)
		cssmap (MAP02, css_slot, 0x0F);
	else
		cssmap (MAP02, css_slot, sub_slot);

	sav_berr = got_berr;
	got_berr = 0;
	emulate = ignore_it = 1;

	if (iopm_cmd->istatus != (uint)CSS_OWNED) {
		printf ("IOPM Stucture in unexpected state - not CSS\n");
		printf ("Expected %d; Got %d\n", CSS_OWNED, iopm_cmd->istatus);
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return(FAIL);
	}

	iopm_cmd->icmd = (uint)IOPM_EEPROM_VER;

	if (do_iopm_cmd (VER_DELAY) == FAIL) {
		printf ("Error getting DSDB EEPROM version data.\n");
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return(FAIL);
	}

	if (iopm_cmd->ret_code) {
		printf ("Error getting DSDB EEPROM version data.\n");
		return(1);
	}
	version = ((iopm_cmd->fname[0] * 10) + iopm_cmd->fname[1]);

	if (version < 26) {
		printf ("DSDB EEPROM version %d.%d doesn't support dev_type.\n",
			iopm_cmd->fname[0], iopm_cmd->fname[1]);
		return(1);
	}
	if (version < 29)
		scsi_cmd = (struct scsi_comm *)SCSI_COMM_ADDR2;
	else
		scsi_cmd = (struct scsi_comm *)SCSI_COMM_ADDR8;

	iopm_cmd->phys_dev = (uint)phys_dev;
	iopm_cmd->icmd = (uint)GET_DEV_TYPE;

	if (do_iopm_cmd (TYPE_DELAY) == FAIL) {
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return(FAIL);
	}

	got_berr = sav_berr;
	emulate = emulate_save;
	ignore_it = ignore_save;

	if (iopm_cmd->ret_code)
		show_iopm_error (iopm_cmd->ret_code);
	else
		show_inq_info (css_slot, sub_slot, scsi_cmd);

	return(iopm_cmd->ret_code);
}

static char *dev_types[] = {
	"Magnetic Disk",
	"Magnetic Tape",
	"Printer",
	"Processor",
	"WORM",
	"Read only device"
};

show_inq_info (c_slot, s_slot, scsi_cmd)
uint	c_slot;
uint	s_slot;
SCSI_COMM_PTR  scsi_cmd;
{
	uint	count = 0;

	if (s_slot == NO_SUB_SLOT)
		cssmap (MAP02, c_slot, 0x0F);
	else
		cssmap (MAP02, c_slot, s_slot);

	if (scsi_cmd->result_buffer[0] < (sizeof(dev_types)/sizeof(char *)))
		printf ("Device type:  %s\n", 
					dev_types[scsi_cmd->result_buffer[0]]);

	printf ("Vendor:  ");
	for (count = 8; count != 16 && scsi_cmd->result_buffer[count]; count++)
		putchar(scsi_cmd->result_buffer[count]);
	printf ("\nProduct:  ");
	for (count = 16; count != 32 && scsi_cmd->result_buffer[count]; count++)
		putchar(scsi_cmd->result_buffer[count]);
	printf ("\nRevision:  ");
	for (count = 32; count != 36 && scsi_cmd->result_buffer[count]; count++)
		putchar(scsi_cmd->result_buffer[count]);
	
	putchar('\n');
}


void
check_iopm_dev_type (type_expected)
uint	type_expected;
{
	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;

	if (sub_slot == NO_SUB_SLOT)
		cssmap (MAP02, css_slot, 0x0F);
	else
		cssmap (MAP02, css_slot, sub_slot);

	switch (type_expected) {
	case DV_IOPM_MT:
	case DV_IOPM_9T:
		if (iopm_cmd->dev_type & TAPE_DEVICE)
			printf ("Device is tape type as requested\n");
		else
			printf ("Device is not tape type as expected\n");
		break;
	case DV_IOPM_DK:
	case DV_IOPM_RV:
		if (iopm_cmd->dev_type & DISK_DEVICE)
			printf ("Device is disk type as requested\n");
		else
			printf ("Device is not disk type as expected\n");
		break;
	default:
		printf ("find_iopm_dev_type: bad boot_type.\n");
		break;
	}
}

int
iopm_init (c_slot, s_slot)
uchar	c_slot;
uchar	s_slot;
{
	uint err = 0;
	uint cnt;
	uint *iopm_start= (uint *)IOP_RAM_START;
	uint *iopm_end = (uint *)(IOP_RAM_START + IOPM_MEM_SIZE);
	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;

	if (s_slot == NO_SUB_SLOT)
		cssmap (MAP02, c_slot, 0x0F);
	else {
		if (ios_conf[logiom(c_slot)][s_slot].slot_id == IOPM_CONSOLE)
			return(0);

		cssmap (MAP02, c_slot, s_slot);
	}

	for (cnt = 0; cnt != IOPM_WRITES; cnt++)
		*iopm_start = (uint)0;
	
	wait_cnt(1800);
         
	for (cnt = 0; cnt != 8; cnt++) {
		fill32 (iopm_start, iopm_end, tst_pat[cnt], TEST_INCR);
		if ((err += check32 (iopm_start, iopm_end, tst_pat[cnt], TEST_INCR)) > MAX_MEM_ERR)
			return(1);
	}
	fill32 (iopm_start, iopm_end, 0, 0);

	iopm_cmd->iop_magic = SPM_MAGIC_NUM;

	return(0);
}

int
iopm_prom_ver (comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	uchar  sav_berr;
	char	device[8];
	uint	iom_num;
	char	emulate_save = emulate;
	char	ignore_save = ignore_it;

	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;

	switch (arg_cnt) {
		case 0:
			break;
		case 1:
			css_slot = (uchar)atoi(comm_args[1]);
			sub_slot = NO_SUB_SLOT;
			break;
		case 2:
			css_slot = (uchar)atoi(comm_args[1]);
			sub_slot = (uchar)atoi(comm_args[2]);
			break;
	}

	if (sub_slot == NO_SUB_SLOT) {
		if (bdhere[css_slot] != IOPHERE) {
			printf ("No IOPM in slot %x (%d).\n",css_slot,css_slot);
			return(1);
		}
	}
	else {
		iom_num = logiom(css_slot);
		if (ios_conf[iom_num][sub_slot].slot_id != IOPHERE) {
			printf ("No IOPM in slot %x/%x (%d/%d).\n", 
				css_slot, sub_slot, css_slot, sub_slot);
			return(1);
		}
	}

	sav_berr = got_berr;
	got_berr = 0;
	emulate = ignore_it = 1;

	if (sub_slot == NO_SUB_SLOT)
		cssmap (MAP02, css_slot, 0x0F);
	else
		cssmap (MAP02, css_slot, sub_slot);

	if (iopm_cmd->istatus != (uint)CSS_OWNED) {
		printf ("IOPM Stucture in unexpected state - not CSS\n");
		printf ("Expected %d; Got %d\n", CSS_OWNED, iopm_cmd->istatus);
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return (1);
	}

	if (comm_str[1] == 'e') {
		iopm_cmd->icmd = (uint)IOPM_EEPROM_VER;
		strcpy (device, "EEPROM");
	}
	else {
		iopm_cmd->icmd = (uint)IOPM_PROM_VER;
		strcpy (device, "EPROM");
	}

	if (do_iopm_cmd (VER_DELAY) == FAIL) {
		printf ("Error getting IOPM %s version data.\n", device);
		got_berr = sav_berr;
		emulate = emulate_save;
		ignore_it = ignore_save;
		return(FAIL);
	}
	got_berr = sav_berr;
	emulate = emulate_save;
	ignore_it = ignore_save;

	if (iopm_cmd->ret_code) {
		printf ("Error getting IOPM %s version data.\n", device);
		return(1);
	}

	printf ("IOPM %s version:  %d.%d\n", 
		device,	iopm_cmd->fname[0], iopm_cmd->fname[1]);
	return(0);
}

void
iopm_recover (c_slot, s_slot)
uchar	c_slot;
uchar   s_slot;
{

	if (s_slot == NO_SUB_SLOT) {
		cssmap(MAP00, (unsigned char)c_slot, 0x0f); /* setup IOPM map */
		ifdisable(c_slot, 0);			/* reset board */

		wait_cnt(400);		/* 100 ms  + 10% */

		ifenable(c_slot, 2);			/* enable interface */
		iopm_init(c_slot, NO_SUB_SLOT);	  /* initialization routine */
		ifenable(c_slot, 0);			/* enable module */
	}
	else {
		printf ("don't do this yet\n");
	}

	return;
}

int
do_iopm_cmd (iopm_timeout)
uint	iopm_timeout;
{
	uint	timer = 0;

	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;

	if (sub_slot == NO_SUB_SLOT)
		cssmap (MAP02, css_slot, 0x0F);
	else
		cssmap (MAP02, css_slot, sub_slot);

	iopm_cmd->ret_code = IOPM_CMD_TIMEOUT;
	iopm_cmd->istatus = (uint)IOPM_OWNED;

	while (iopm_cmd->istatus != (uint)CSS_OWNED && !got_berr) {
		wait_cnt(WASTE_LIMIT / 100);

		if (got_berr) {
			printf ("IOPM in slot %x (%d) failed.\n", 
					css_slot, css_slot);
			printf ("Attempting recovery...reseting IOPM.\n");
			iopm_recover (css_slot, sub_slot);
			return(FAIL);
		}

		if (timer++ == iopm_timeout)
			return(FAIL);
	}
	return(PASS);		
}

int
check_iopm_magic ()
{
	uchar	c_slot;
	uint	s_slot;
	IOP_COMM_PTR  iopm_cmd = (struct iop_comm *)IOPM_COMM_ADDR;
	uint	iom_num = 0;
	uint	magic_number;
	uint	timeout = 4 * 20000; /* 4 times as long to allow for 4 Mb iopm */
#ifdef	REMOVE
	wait_cnt (22000);
#endif

	for (c_slot = 0; c_slot != Sbus_Num_Slot; ) {
		if (bdhere[c_slot] == IOPHERE) {
			cssmap (MAP02, c_slot, 0x0F);
			if (iopm_cmd->iop_magic != (uint)IOPM_MAGIC_NUM) {
				wait_cnt(1);
				if (--timeout)
					continue;
				else {
					printf ("\nIOPM in slot %x (%d)", 
						c_slot, c_slot);
					printf ("does not respond, Halted.\n");
					cssmap(MAP00, (uchar)c_slot, 0x0f);

					ifdisable(c_slot, 0);	/* disable bd */
					bdhere[c_slot] = NOBOARD;
					sf.on[c_slot] = 0; /* Reset slot flag */
				}
			}
		}
		c_slot++;
	}

	for (c_slot = 0; c_slot != Sbus_Num_Slot; c_slot++) {
	    if ((bdhere[c_slot] & BDTYPEMASK) == IOMTYPE) {
		if (ios_conf[iom_num][0].iomslot != c_slot)
		    continue;
	
	    for (s_slot = 0; s_slot != Sbus_Num_Slot; ) {
		if (ios_conf[iom_num][s_slot].slot_id == (uchar)IOPHERE) {

		    cssmap (MAP02, c_slot, s_slot);
		    if ((magic_number=iopm_cmd->iop_magic) != 
						(uint)IOPM_MAGIC_NUM) {
			wait_cnt(1);
			if (--timeout)
			    continue;
			else {
		      	    printf("got = %x, exp %x\n", 
					magic_number, IOPM_MAGIC_NUM);
		      	    printf ("\nIOPM in slots %x/%x (%d/%d)",
				c_slot, s_slot, c_slot, s_slot);
		      	    printf ("does not respond, Halted\n");
		      	    io_control(MOD_DISABLE, c_slot, s_slot);
		      	    io_control(INTERFACE_DISABLE, c_slot, s_slot);
		      	    ios_conf[iom_num][s_slot].slot_id = (uchar)NOBOARD;
		        }
		    }
	        }
	        s_slot++;
	     }
             iom_num++;
	  }
	}
}


int
check_iopm_console (c_slot, s_slot)
uint	c_slot;
uint	s_slot;
{
	
	cssmap (MAP07, c_slot, s_slot);

	if (*STATUS_CTRL_REG & CONSOLE_ON)
		return(1);
	
	return(0);
}

int
iopm_no_css_reset(c_slot, s_slot)
uint	c_slot;
uint	s_slot;
{
	unsigned short conf_reg;

	cssmap (MAP07, c_slot, s_slot);

	conf_reg = *CONFIG_REG;

	conf_reg |= (unsigned short)(IOPM_CSS_RESET);

	*CONFIG_REG = (unsigned short)conf_reg;

}


static  char *cmd_err[] = {
	"",
	"Device ID Invalid.\n",
	"Device not Ready.\n",
	"Disk device Read Failed.\n",
	"Disk device Write Failed.\n",
	"Tape device Read Failed.\n",
	"Tape device Write Failed.\n",
	"Tape rewind Failed.\n",
	"Tape skip Failed.\n",
	"Device not supported.\n",
	"Unit initialization Failed.\n",
	"Requested command not supported.\n",
	"Device Inquiry Failed.\n"
};


static  char *scsi_err[] = {
	"",
	"DMA count.\n",
	"Device not on line.\n",
	"Command timeout.\n",
	"Illegal instruction.\n",
	"Xfer flag, improper state.\n",
	"Unexpected Disconnect.\n",
	"Unexpected Interrupt.\n",
	"Unexpected message.\n",
	"Check device condition.\n",
	"Device Busy.\n",
	"Unexpected Status.\n",
	"Phase error.\n",
	"Fatal SCSI error.\n",
};

void
show_iopm_error(stat)
uint stat;
{
	register  uint	err;

	putchar('\n');
	if (err = (stat & 0x00FF))
		if (err < (sizeof(cmd_err)/sizeof(char *))) 
			printf ("Command Error:  %s", cmd_err[err]);
		else
			printf ("Unknown command error %x.\n", err);
	
	if (err = ((stat & 0xFF00) >> 8))
		if (err < (sizeof(scsi_err)/sizeof(char *))) 
			printf ("SCSI Error:  %s", scsi_err[err]);
		else
			printf ("Unknown SCSI error %x.\n", (err << 8));
	
}

