
/*
** file: flp_exec.c
**		Routines in this file are responsible for controlling the 
**		complete floppy command execution.  This file includes the
**		individual commands rdsec, wrsec, rdtrack, seek, recal, mon,
**		moff.  The command execution is controlled by do_flp_cmd (),
**		which uses send_cmd () (in this file), then calls routines
**		in flp_result.c and flp_check.c to complete the command.
**
**  890819  Redo floppy driver 
**
**	flp.c		--  User interface
**	flp_exec.c	--  Floppy command start
**	flp_result.c	--  Process floppy command
**	flp_check.c	--  Analyze floppy response
**	flp_test.c	--  Floppy tests -- from i_flop_test menu
**	head/flp.h	--  General defines used by all
**	head/flp_chk.h  --  Floppy result checking include
**/

#include "types.h"
#include "global.h"
#include "flp.h"			/* definition of floppy command block */

#define	TIMMY	1000 

extern	struct	flp_cmd	 flp_cmd;
extern	union	flp_res	 flp_res;
extern	uint	flp_errors[];		/* error array */
extern	uchar	*flp_buf;		/* address used by read and write
					   commands.			*/
extern	uchar	spinning;		/* motor status, checked by mon/moff
					   set/reset by flp_motor_on and
					   flp_motor_off respectively	*/
extern	uchar	current_cyl;		/* accurate cylinder, initialized to
					   0xFF on powerup, checked by seek,
					   set by result phase of recal or
					   seek only, set to 0xFF when 
					   ready change detected 	*/
extern	uchar	flp_status;		/* Status returned by a sense drive
					   status command		*/
extern	uint	xfer_size;		/* Size of xfer performed by the 
					   assembly code in flp_xfer.s	*/
extern	uint	flp_int;		/* flag used to indicate a floppy
					   interrupt has occured.	*/
extern	uint	flp_err;		/* flag used to indicate a floppy
					   error has occured during the 
					   read, write or result phase.	*/

uint	flp_error;			/* DEBUG */
uint	res_cnt;			/* number of bytes in the result
					   block.			*/
uchar	char_data;

void 	mon();
void	moff();
int	recal();
int	seek();
int	rdsec();
int	rdtrack();
int	wrsec();
#ifdef	REMOVE
int	format_track();
#endif
int	sense_int_status();
void	floppy_init();
void	flp_dev_init();
void	clear_fdc();

int	do_flp_cmd();
int	send_flp_cmd();

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
mon()
{
	if(!spinning) {		/* Check if command may be ignored */
		*FDBITS = M_ON;
		wait_cnt(5000);

		spinning = 1;
	}
	return;
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
moff()
{
	if (spinning) {	/* Check if command may be ignored */
		*FDBITS = M_OFF;
		wait_cnt(5000);
	
		spinning = 0;
	}
	return;
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
recal()
{
	int	do_flp_ret = 0;

	mon();			/* Turn on the motor. */

#ifdef	DEBUG
	printf ("recal:  Begin execution.\n");
#endif
	flp_cmd.cmd1 = (uchar)FD_RECAL;
	flp_cmd.cmd2 = (uchar)FD_HEAD0;
		
	res_cnt = SIS_RES;
	expect_sk_int();
	if (do_flp_ret = do_flp_cmd(SEND2, SEEK_TIMEOUT, 0)) {
		current_cyl = (uchar)NEED_RECAL;
		SKERR++;
	}
	else {
		current_cyl = flp_res.recal.cyl;
	}
	res_flp_int();

	return(do_flp_ret | (int)current_cyl);
}
	
/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
seek(cyl)
uchar	cyl;
{
	int	recal_ret = 0;
	int	do_flp_ret = 0;

	if (cyl == current_cyl)
		return((int)current_cyl);

	mon ();

#ifdef	DEBUG
	printf ("seek:  Begin execution.  cyl:  %d.\n", cyl);
#endif
	if (current_cyl == (uchar)NEED_RECAL) {
		printf ("seek:  Recal command needed on floppy drive.\n");
		if(recal_ret = recal())
			return(recal_ret | RECAL_FAIL);
	}

	flp_cmd.cmd1 = (uchar)FD_SEEK;
	flp_cmd.cmd2 = (uchar)FD_HEAD0;
		
	flp_cmd.args.seek.new_cyl = (uchar)cyl;

	res_cnt = SIS_RES;
	expect_sk_int();
	if (do_flp_ret = do_flp_cmd(SEND3, SEEK_TIMEOUT, 0)) {
		current_cyl = (uchar)NEED_RECAL;
		SKERR++;
	}
	else {
		current_cyl = flp_res.seek.cyl;
	}
	res_flp_int();

	return(do_flp_ret | (int)current_cyl);
}
	
/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
rdsec(cyl, head, sect)
uchar	cyl;
uchar	head;
uchar	sect;
{
	int	seek_ret = 0;
	int	do_flp_ret = 0;

	mon ();

	if((seek_ret = seek(cyl)) != (int)cyl)
		return(seek_ret | SEEK_FAIL);

#ifdef	DEBUG
	printf ("rdsec:  Begin execution.  cyl:  %d; head:  %d;  sct:  %d.\n",
		cyl, head, sect);
#endif
	flp_cmd.cmd1 = (uchar)FD_RDSEC;
	if (head == (uchar)1)
		flp_cmd.cmd2 = (uchar)FD_HEAD1;
	else
		flp_cmd.cmd2 = (uchar)FD_HEAD0;
		
	flp_cmd.args.rd_wr.cyl = (uchar)cyl;
	flp_cmd.args.rd_wr.head = (uchar)head;
	flp_cmd.args.rd_wr.sect = (uchar)sect;
	flp_cmd.args.rd_wr.nbyte = (uchar)NBYTES;
	flp_cmd.args.rd_wr.eot = (uchar)EOT;
	flp_cmd.args.rd_wr.gpl = (uchar)GPL;
	flp_cmd.args.rd_wr.dtl = (uchar)DTL;

	xfer_size = (uint)RW_SEC;
	res_cnt = RES7;
	expect_rd_int();
	if (do_flp_ret = do_flp_cmd(SEND9, RD_WR_TIMEOUT, RES_TIMEOUT))
		RDERR++;

	res_flp_int();

	return(do_flp_ret);
}

#ifdef	REMOVE
/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
rdtrack(cyl, head)
uchar	cyl;
uchar	head;
{
	int	seek_ret = 0;
	int	do_flp_ret = 0;

	mon();

	if((seek_ret = seek(cyl)) != (int)cyl)
		return(seek_ret | SEEK_FAIL);

#ifdef	DEBUG
	printf ("rdtrack:  Begin execution.  cyl: %d; head: %d.\n", cyl, head);
#endif
	flp_cmd.cmd1 = (uchar)FD_RDTRACK;
	if (head == (uchar)1)
		flp_cmd.cmd2 = (uchar)FD_HEAD1;
	else
		flp_cmd.cmd2 = (uchar)FD_HEAD0;
		
	flp_cmd.args.rd_wr.cyl = (uchar)cyl;
	flp_cmd.args.rd_wr.head = (uchar)head;
	flp_cmd.args.rd_wr.sect = (uchar)0x00;
	flp_cmd.args.rd_wr.nbyte = (uchar)NBYTES;
	flp_cmd.args.rd_wr.eot = (uchar)EOT;
	flp_cmd.args.rd_wr.gpl = (uchar)GPL;
	flp_cmd.args.rd_wr.dtl = (uchar)DTL;

	xfer_size = (uint)RW_TRACK;
	res_cnt = RES7;
	expect_rd_int();
	if (do_flp_ret = do_flp_cmd(SEND9, RD_WR_TIMEOUT, RES_TIMEOUT))
		RDERR++;

	res_flp_int();

	return(do_flp_ret);
}
#endif

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
wrsec(cyl, head, sect)
uchar	cyl;
uchar	head;
uchar	sect;
{
	int	seek_ret = 0;
	int	do_flp_ret = 0;

	mon();

	if((seek_ret = seek(cyl)) != (int)cyl)
		return(seek_ret | SEEK_FAIL);

#ifdef	DEBUG
	printf ("wrsec:  Begin execution.  cyl:  %d; head:  %d; sect:  %d.\n",
			cyl, head, sect);
#endif
	flp_cmd.cmd1 = (uchar)FD_WRSEC;
	if (head == (uchar)1)
		flp_cmd.cmd2 = (uchar)FD_HEAD1;
	else
		flp_cmd.cmd2 = (uchar)FD_HEAD0;
		
	flp_cmd.args.rd_wr.cyl = (uchar)cyl;
	flp_cmd.args.rd_wr.head = (uchar)head;
	flp_cmd.args.rd_wr.sect = (uchar)sect;
	flp_cmd.args.rd_wr.nbyte = (uchar)NBYTES;
	flp_cmd.args.rd_wr.eot = (uchar)EOT;
	flp_cmd.args.rd_wr.gpl = (uchar)GPL;
	flp_cmd.args.rd_wr.dtl = (uchar)DTL;

	xfer_size = (uint)RW_SEC;
	res_cnt = RES7;
	expect_wr_int();
	if (do_flp_ret = do_flp_cmd(SEND9, RD_WR_TIMEOUT, RES_TIMEOUT))
		WRERR++;

	res_flp_int();

	return(do_flp_ret);
}

#ifdef	REMOVE
/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
format_track(cyl, head)
uchar	cyl;
uchar	head;
{
	int	seek_ret = 0;
	int	do_flp_ret = 0;

	mon();

	if((seek_ret = seek(cyl)) != (int)cyl)
		return(seek_ret | SEEK_FAIL);

#ifdef	DEBUG
	printf ("format_track:  Begin execution.  cyl:  %d; head:  %d.\n",
		cyl, head);
#endif
	flp_cmd.cmd1 = (uchar)FD_FMT;
	if (head == (uchar)1)
		flp_cmd.cmd2 = (uchar)FD_HEAD1;
	else
		flp_cmd.cmd2 = (uchar)FD_HEAD0;
		
	flp_cmd.args.format.nbyte = (uchar)NBYTES;
	flp_cmd.args.format.spt = (uchar)SPT;
	flp_cmd.args.format.fmt_gpl = (uchar)FMT_GPL;
	flp_cmd.args.format.fbyte = (uchar)FFB;

	printf ("Format cylinder %2d, head %d ",
		cyl, head);

	res_cnt = RES7;
	expect_fresult();
	if (do_flp_ret = do_flp_cmd(SEND6, 0, FMT_TIMEOUT))
		WRERR++;

	res_flp_int();

	return(do_flp_ret);
}
#endif

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
specify()
{
	int	do_flp_ret = 0;

#ifdef	DEBUG
	printf ("specify:  Begin execution.\n");
#endif
	flp_cmd.cmd1 = (uchar)FD_SPEC;
	flp_cmd.cmd2 = (uchar)SPEC1;
		
	flp_cmd.args.specify.hlt = (uchar)SPEC2;

	res_cnt = 0;
	do_flp_ret = do_flp_cmd(SEND3, 0, 0);

	return(do_flp_ret);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
do_flp_cmd(send_cnt, int_timeout, res_timeout)
uint	send_cnt;
register  uint	int_timeout;
register  uint	res_timeout;
{
	uint	done = 0;
	uint	retry = FLP_RETRY;
	uint	flp_error = 0;

	do {
		if (flp_error = Do_flp_cmd(send_cnt, int_timeout, res_timeout)){
			flp_dev_init();
			if (!(--retry))
				done = 1;
		}
		else {
			done = 1;
		}
	} while (!done);

	if (flp_error != RES_INT_FAIL)		/* No results to check */
		flp_error |= check_result ();

	return(flp_error);
}

int
Do_flp_cmd(send_cnt, int_timeout, res_timeout)
uint	send_cnt;
register  uint	int_timeout;
register  uint	res_timeout;
{
	int error = 0;

	flp_err = 0;			/* reset floppy error flag */
	flp_int = 0;			/* reset interrupt occured flag */

#ifdef	DEBUG
	flp_error = 0;		/* DEBUG variable */

	printf ("do_flp_cmd:  Begin execution.  snd: %d; int: %d; res: %d.\n",	
		send_cnt, int_timeout, res_timeout);
#endif

/*
**	Send command phase
*/
	if (error = send_flp_cmd(send_cnt))
		return(error);			/* return, no command sent....
						   no reason to continue */

/*
**	Wait for floppy drive to execute passed command.  Interrupt routines
**	will read data, write data, or complete the specified command.  If a
**	special interrupt handling routine is to be used, it must be placed
**	in the vector table prior to this time. (before calling do_flp_cmd)
**
*/
	if (int_timeout) {	/* If need to wait for interrupt */

		while (!(flp_int & 0xFFFF0000) && --int_timeout) {
			wait_cnt(10);
		}

#ifdef	DEBUG
		printf ("do_flp_cmd:  CMD_INT>>  cnt: %d; flp_error: %d.\n",
				int_timeout, flp_error);
		flp_error = 0;
#endif
		if (!int_timeout) {
			printf ("Time-out waiting for floppy command.\n");
			printf ("No interrupt received.\n");
			return(CMD_INT_FAIL);
		}
	}
	
/*	
**	The result phase is completed in this loop.  The floppy controller
**	will interrupt the CPU when result data is ready.  The interrupt
**	routine is in flp_xfer.s
*/
	if (res_timeout) {	/* If result phase interrupt driven */

		while (!(flp_int & 0x0000FFFF) && --res_timeout) {
			wait_cnt(10);
		}

#ifdef	DEBUG
		printf ("do_flp_cmd:  RES_INT>>  cnt:  %d, flp_error %d.\n",
			res_timeout, flp_error);
#endif
		if (!res_timeout) {
			printf ("Time-out waiting for floppy results.\n");
			printf ("No interrupt received.\n");
			return(RES_INT_FAIL);
		}
	}

	if (flp_err & 0xFFFF0000) 	/* did error occur during cmd */
		error |= CMD_FAIL;
	
	if (flp_err & 0x0000FFFF) 	/* did error occur during result */
		error |= RES_FAIL;

	return(error);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
send_flp_cmd(send)
uint	send;
{
	uchar	*command;		/* pointer to floppy command array */
	uchar	sent = 0;		/* byte sent to fdc count */
	uint	timer;			/* timeout counter */
	
	command = (uchar *)(&flp_cmd);	

#ifdef	DEBUG
	printf ("send_flp_cmd:  Sending a command of %2x.\n", flp_cmd.cmd1);
#endif

	timer = (uint)EXM_TIMEOUT;
	while (EXM_NOT_DONE && --timer) 
		wait_cnt(10);

	if (!timer) {
		printf ("send_flp_cmd:  Time-out waiting for exm_done.\n");
		return(SEND_EXM_FAIL);
	}

	timer = (uint)SEND_TIMEOUT;
	while ((sent != send) && --timer) {
		if((*MSR & MSRMSK) == MSRcmd)  {
			*DATA = command[sent++];
			timer = (uint)SEND_TIMEOUT;
		}
		else {
			wait_cnt(10);
		}
	}

	if (!timer) {
		printf ("send_flp_cmd:  Time-out sending command.\n");
		return(SEND_FAIL);
	}

	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
flp_dev_init()
{
	unsigned int int_level;

	moff();
	int_level = chgint(7);		/* set to 7, saving old level */

	*OPS = (uchar)OPS_RESET;	/* Generate soft reset */

	char_data = *CNTL;		/* Read CNTL -- fdc in special mode */
	wait_cnt(1); 

	*OPS = (uchar)OPS_RESET;	/* Generate SRST */
	wait_cnt(1);

	char_data = *CNTL;		/* Read CNTL -- fdc in AT mode */
	wait_cnt(1); 

	*CNTL = 0x00;			/* Set RPM in control register */
	wait_cnt(1);

	*OPS = (uchar)CNTRLwd;		/* Setup mode as desired */
	wait_cnt(1); 

	clear_fdc();			/* Clear fdc if needed */
	wait_cnt(5); 

	chgint(int_level);		/* restore interrupt level */
	return;
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
clear_fdc()
{
	uchar stat;
	uchar quit = 0;
	uchar cnt0 = 0;

	do {
		wait_cnt(1);
		stat = *MSR;
		switch(stat & 0xf1) {
			case 0x70:
			case 0x71:
				printf("Floppy drive not ready.\n");
				quit = 1;
				break;
			case 0x80: 
				quit = 1; 
				break;
			case 0x81:
			case 0xd1: 
				sense_int_status(); 
				break;
			case 0x90:
			case 0xd0: 
			case 0xc0: 
				char_data = *DATA; 
				break;
			default: 
				if (cnt0++ == 20) 
					quit = 1;
				break;
		}
	} while(!quit);
	return;
}


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
sense_int_status()
{

	uint	error = 0;

#ifdef	DEBUG
	printf ("sense_int_status:  Begin execution.\n");
#endif

	flp_cmd.cmd1 = (uchar)FD_SEEK;

	res_cnt = SIS_RES;

	if (error = send_flp_cmd(1))
		return(error);

	error |= flp_cmd_complete();

	error |= check_result();

	return(error);
}

