
/*
 * debug.c
 *
 */

#include "types.h"
#include "menu.h"
#include "routines.h"
#include "globl.h"
#include "spm.h"
#include "global.h"
#include "rtc.h"
#include "disp.h"
#include "dev.h"
#include "novram.h"
#include "misc.h"
#include "ioa0.h"
#include "icb_config.h"

#define MAP00	00	/* map ram 0 */
#define MM_INIT 0x78	/* Turn green LED, EDAC and error interrupt bits */
#define MaxSlots 16

extern	uchar	byte_throw;
extern	ushort	word_throw;
extern	uint	long_throw;
extern	uchar	HndlBerr;
extern	char	emulate;
extern	char	got_berr; 	/* set if we got bus error */
extern	uchar	bdhere[];	/* slot id for CSS type boards */
extern	uchar	css_slot;
extern	uchar	sub_slot;
extern	uchar	phys_dev;
extern	uchar	log_dev;
extern	struct	iosb_config	ios_conf[][MAX_CSS_SLOT];
extern	uchar	*CTL_REG;

static	uint	em_addr;

extern	uint get_iopm_blk_num ();
extern	int iopm_prom_ver ();
extern	int slot_cntl();
extern	int cmd_cntl();
extern  int nov_tst();
int exemem();
int iopm_tst ();
int iopm_blk_num();
int iopm_dev_type();
int rready_prt();
int iready_prt();
int init_prt_css();
int init_css_prt();
int set_berr_loop();
int reset_berr_loop();

struct init_menu i_debugger [] = {
	
	"jsr", "addr [arg1 [arg2 [arg3 [arg4 [arg5 [arg6]]]]]], args in HEX",
	"Jump to subroutine",'c',1,7,jsr,    (char *)0,
	
	"memd", "",
	"Memory Debugger Utility Menu",'m',0,0,(PFI)i_debugger_mem,  (char *)0,
	
	"flagd", "",
	"Debugger Flag Set/Reset Menu",'m',0,0,(PFI)i_debugger_flag,  (char *)0,
	
	"busd", "",
	"Debugger CSS Bus Utility Menu",'m',0,0,(PFI)i_debugger_bus,  (char *)0,
	
	"iod", "",
	"Debugger I/O Utility Menu",'m',0,0,(PFI)i_debugger_io,  (char *)0,

	"", "", "", '\0', 0, 0, 0, (char *)0,
	};
	
	
struct init_menu i_debugger_flag [] = {
	"ca", "<on|off>",
	"Turn CPU cache on or off", 'c',1,1, cache020, (char *)0,
	
	"me","<on|off>",
	"Enter memory simulation mode", 'c',1,1,mem_simulate, (char *)0,
	
	"sbl", "",
	"Set buserr loop flag ", 'c', 0, 0, set_berr_loop, (char *)0,
	
	"rbl", "",
	"Reset buserr loop flag ", 'c', 0, 0, reset_berr_loop, (char *)0,
	
	"em", "",
	"Set buserr emulate flag ", 'c', 0, 0, berr_em, (char *)0,
	
	"emn", "",
	"Reset buserr emulate flag ", 'c', 0, 0, berr_em_n, (char *)0,
	
	"", "", "", '\0', 0, 0, 0, (char *)0,
	};
	
	
struct init_menu i_debugger_bus [] = {
	"ic", "",
	"Initialize CSS activity", 'c', 0,0,init_css_prt, (char *)0,
	
	"id","[d] (diag mode)",
	"Initialize Dispatcher",'c',0,1,init_dispatcher,(char *)0,
	
	"ir", "",
	"Increase ready count in arbiter", 'c',0,0,iready_prt, (char *)0,
	
	"rr", "",
	"Reset ready count in arbiter", 'c',0,0,rready_prt, (char *)0,
	
	"rc", "",
	"Reset CSS bus", 'c', 0, 0, css_reset1, (char *)0,
	
	"eslot","<if|mod|all> css_slot [sub_slot], slots in decimal",
	"Enable CSS IF or MODULE ", 'c', 2,3, slot_cntl, (char *)0,
	
	"dslot","<if|mod|all> css_slot [sub_slot], slots in decimal",
	"Disable CSS IF or MODULE ", 'c', 2,3, slot_cntl, (char *)0,
	
	"Eslot","<if|mod|all> board_id css_slot [sub_slot], slots in decimal",
	"Enable CSS IF or MODULE ", 'c', 3,4, slot_cntl, (char *)0,
	
	"Dslot","<if|mod|all> board_id css_slot [sub_slot], slots in decimal",
	"Disable CSS IF or MODULE ", 'c', 3,4, slot_cntl, (char *)0,
	
	"con(trol)","cmd  css_slot, cmd in hex, slot in decimal",
	"Send a control write to specified slot",'c',2,2,cmd_cntl,(char *)0,
	
	"", "", "", '\0', 0, 0, 0, (char *)0,
	};
	
	
struct init_menu i_debugger_mem [] = {
	"map", "[map]  [slot]  [addr], args in hex",
	"Map slot and address", 'c', 0,3,mslot, (char *)0,
	
	"dm<.[bwl]>", "addr  [size], args in hex",
	"Display memory", 'c', 1, 2, dmem, (char *)0,
	
	"fm<.[bwl]>", "addr  data  [size], args in hex",
	"Fill memory block", 'c', 2,3,fmem, (char *)0,
	
	"mm<.[bwl]>", "addr  [data], args in hex",
	"Modify memory", 'c', 1, 2, mmem, (char *)0,
	
	"tm<.[bwl]>", "start_addr end_addr [seed] [incr], args in hex",
	"Test memory", 'c', 2, 4, memtest, (char *)0,
	
	"ltm<.[bwl]>", "start_addr end_addr [seed] [incr], args in hex",
	"Loop test memory", 'c', 2, 4, memtest, (char *)0,
	
	"nov_tst", "",
	"Test Novram for read and write", 'c', 0, 0, nov_tst, (char *)0,
	
	"lnov_tst", "",
	"Loop test Novram for read and write", 'c', 0, 0, nov_tst, (char *)0,
	
	"rdb<.[bwl]>", "addr  [size], args in hex",
	"Read block loop", 'c', 1, 2, rdblock, (char *)0,
 	
	"wrb<.[bwl]>","addr  data  [size], args in hex",
	"Write block loop",'c',2,3,wrblock,(char *)0,
	
	"mov<.[bwl]>", "src  dest  size, args in hex",
	"Move memory", 'c', 3,3,move_mem, (char *)0,
	
	"lmov<.[bwl]>","src  dest  size, args in hex",
	"Loop move memory",'c',3,3,move_mem,(char *)0,
	
	"cmp<.[bwl]>","src  dest  size, args in hex",
	"Compare memory",'c',3,3,compare_mem,(char *)0,
	
	"lcmp<.[bwl]>", "src  dest  size, args in hex",
	"Loop compare memory", 'c', 3,3,compare_mem, (char *)0,
	
	"ecmp<.[bwl]>", "src  dest  size, args in hex",
	"Loop compare memory, stop on error", 'c', 3,3,compare_mem, (char *)0,
	
	"sea<.[bwl]>", "pat [addr] [leng], args in hex",
	"Search for pattern in memory", 'c', 1,3,search_mem, (char *)0,
#ifdef	REMOVE	
	"lmem","",
	"Local SPM memory test.",'c', 0, 0,lramtst,(char *)0,
#endif	
	"exm","css_slot  [Mbytes], slot in decimal, Mbytes in hex",
	"Execute test from main memory",'c',1,2,exemem,(char *)0,
	
	"", "", "", '\0', 0, 0, 0, (char *)0,
	};
	
	
struct init_menu i_debugger_io [] = {
	"rd_iopm", "css_slot phys_dev block count [sub_slot], args in decimal",
	"Read block(s) from an IOPM device",'c', 4, 5, iopm_tst, (char *)0,

	"blk_num", "css_slot  phys_dev  log_dev  [sub_slot], args in decimal",
	"Find starting block number",'c', 3, 4, iopm_blk_num, (char *)0,
	
	"dev_type", "css_slot  phys_dev  [sub_slot], args in decimal",
	"Display inquiry information",'c', 2, 3, iopm_dev_type, (char *)0,
	
	"e_i(opm_ver)", "[css_slot  [sub_slot]], args in decimal",
	"Display IOPM EPROM Version",'c', 0, 2, iopm_prom_ver, (char *)0,
	
	"ee_i(opm_ee_ver)", "[css_slot  [sub_slot]], args in decimal",
	"Display IOPM EEPROM Version",'c', 0, 2, iopm_prom_ver, (char *)0,
	
	"", "", "", '\0', 0, 0, 0, (char *)0,
	};

struct menu *debugger = (struct menu *)i_debugger;
struct menu *debugger_mem = (struct menu *)i_debugger_mem;
struct menu *debugger_flag = (struct menu *)i_debugger_flag;
struct menu *debugger_io = (struct menu *)i_debugger_io;
struct menu *debugger_bus = (struct menu *)i_debugger_bus;

/*
 *	Memory related commands
 *
 *	dmem(comm_str, arg_cnt)
 *	char *comm_str;
 *	int arg_cnt;
 *
 *	dm(emory)<.[bwl]> <addr <count>> - Display memory
 *		This is the ubiquitous display memory command. The minimum
 *		required to operate this command is 'dm xxxx', where xxxx is
 *		the desired address. The result is that the memory is dumped
 *		to the screen with the applicable ascii equivalents in the
 *		right hand margin for a page, waiting for a <RET> for another
 *		page, <SP> for another line, or any other key to quit back to
 *		the prompt.
 *		When the count is specified, just that amount is displayed
 *		with no paging.
 */

extern char	mref_siz;	/* Memory reference size, defaults to long */
extern int	pagsize;
extern char	comm_args[][MAXARGSIZE];

static uchar	*lastmem;
static char	linbuf[17];
static int	nline;
static uint	count;


dmem(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register int j;

	linbuf[16] = '\0';

	/* Arg_cnt will be either 0, 1 or 2 */
	switch (arg_cnt) {
		case 0:
			lastmem = (unsigned char*)0x00;
			count = 0x0fffffff; /* max size */
			break;
		case 1:
			lastmem = (unsigned char*)atox(comm_args[1]);
			count = 0x0fffffff; /* max size */
			break;
		case 2:
			lastmem = (unsigned char*)atox(comm_args[1]);
			count = atox(comm_args[2]);/* get the number of bytes */
			break;
	}

	if((unsigned)lastmem & (unsigned)0x80000000 ) {	/* if css check clock */
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return(0);
	}

	/* determine how many fields to print per line */
	switch(mref_siz) {
		case 0  :		/* no mref_siz specified by user */
		case 'l': 
			nline = 4; 
			break;
		case 'w':
			nline = 8;
			break;
		case 'b':
			nline = 16;
			break;
	}
	/* Main loop */
	while(1) {
		for(j=0; (j<pagsize) && (count);j++) {
			one_lne(); /* show one page from current pointers */
			if(!count)			/* done */
				return(0);
		}
		while(1) {
			linbuf[0] = Getchar(); /* wait for key input */
			switch(linbuf[0]) {    /* read the input */
				case '\r': 
				case '\n':
					break;
				case ' ':
					one_lne(); /* show one more line */
					continue;
				default:
					return(0);
			}
			break;
		}
	}
}

one_lne()
{
int i;
register	uchar	rd_char;
		printf("%08x  ", lastmem); /* Beginning addr of line */
		for( i=0; i<nline; i++) {
			switch(mref_siz) {		
				case 0  :
				case 'l':
					printf("%08x ", *(unsigned *)lastmem);
					lastmem += 4;
					count = (count >= 4)?(count - 4):0;
					break;
				case 'w':
					printf("%04x ", *(unsigned short*)lastmem);
					lastmem += 2;
					count = (count >= 2)?(count - 2):0;
					break;
				case 'b':
					printf("%02x ", 
					(rd_char = *(uchar *)lastmem));
					if((rd_char >= ' ') && (rd_char <= '~'))
						linbuf[i] = rd_char;
					else
						linbuf[i] = '.';
					lastmem++;
					count--;
					break;
			}
			if(!count) {
				if(mref_siz == 'b') /* if done and byte mode */
					for(i++; i<nline; i++) {
						linbuf[i] = ' ';
						printf("   ");
					}
				break;
			}
		}
		if(mref_siz == 'b') {
			linbuf[i] = 0;  /* terminate */
			printf(" %s",linbuf);
		}
	putchar('\n');
}


/*
 *	mmem(comm_str, arg_cnt)
 *	char *comm_str;
 *	int arg_cnt;
 *
 *	mm(emory)<.[bwl]> <addr <data>>  -  Modify memory
 */

mmem(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	char inline[64];
	register unsigned data;
	register int val;

	/* Arg cnt will be either 0, 1 or 2 */
	if(arg_cnt == 2)
		data = atox(comm_args[2]);
	if(arg_cnt>=1)
		lastmem = (unsigned char *)atox (comm_args[1]);
        /* if in CSS RAM check which clock we are on */
	if((uchar *)lastmem >= (uchar *)0x80000000) 
	{
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return;
	}
	do {
		printf("%08x  ",lastmem);
/*		if(*comm_str == 'm') {		*/
			switch(mref_siz) {
				case 0  : /* no mref_siz specified by user */
				case 'l':
					printf("%08x ", *(unsigned*)lastmem);
					break;
				case 'w':
					printf("%04x ", *(unsigned short *)lastmem);
					break;
				case 'b':
					printf("%02x ", *lastmem);
					break;
			}
/*		}		*/
		if(arg_cnt == 2) {
			switch(mref_siz) {
				case 'l':
					printf("< %08x\n", (unsigned)data);
					break;
				case 'w':
					printf("< %04x\n", (unsigned short)data);
					break;
				case 'b':
					printf("< %02x\n", (unsigned char)data);
					break;
			}
		}
		else {
			val = (int)gets(inline,64);
			if(val == -1)
				return(0);
			if(*inline == '\0')
				goto nextspot;
			if(*inline == '-') {
				switch(mref_siz) {
					case 0  :
					case 'l':
						lastmem -= 4;
						break;
					case 'w':
						lastmem -= 2;
						break;
					case 'b':
						lastmem--;
						break;
				}
				continue;
			}
			data = atox(inline);
		}
		switch(mref_siz) {
			case 0  :
			case 'l':
				*(unsigned *)lastmem = (unsigned)data;
				break;
			case 'w':
				*(unsigned short *)lastmem = (unsigned short)data;
				break;
			case 'b':
				*lastmem = (unsigned char)data;
				break;
		}
		if(arg_cnt != 2) {
nextspot:		switch(mref_siz) {
				case 0  :
				case 'l':
					lastmem += 4;
					break;
				case 'w':
					lastmem += 2;
					break;
				case 'b':
					lastmem++;
					break;
			}
		}
	}
	while(arg_cnt <= 1);
}

#define	ONE_MEG	0x100000
				
int
memtest(comm_str, arg_cnt)
char  *comm_str;
int   arg_cnt;
{
	char  *start = (char *)atox(comm_args[1]);
	char  *end = (char *)atox(comm_args[2]);
	uint  seed = 0;
	uint  incr = 1;
	uint  loops = 0;
	uint  error = 0;
	uint  passes = 0;
	uint  print_dot = (ONE_MEG / ((unsigned)end - (unsigned)start));
	
	switch (arg_cnt) {
		case 2:
			break;
		case 3:
			printf ("Illegal argument count; must be 2 or 4.\n");
			return(1);
		case 4:
			seed = atox(comm_args[3]);
			incr = atox(comm_args[4]);
			break;
	}

	if (*comm_str == 'l') {
		printf ("Depress ESC to exit\n");
		loops = 1;
	}
	
	do {
		switch (mref_siz) {
			case 'b':
				fill8 (start, end, seed, incr);
				error += check8 (start, end, seed, incr);
				break;
			case 'w':
				fill16 (start, end, seed, incr);
				error += check16 (start, end, seed, incr);
				break;
			case 0  :	/* no mref_siz specified by user */
			case 'l':
				fill32 (start, end, seed, incr);
				error += check32 (start, end, seed, incr);
				break;
		}
		if ((++passes % print_dot) == 0)
			printf(".");

		if (ifesc())
			loops = 0;

	} while (loops);
	printf ("\nMemory testing statistics:  Passes:  %d;   Errors:  %d.\n",
		passes, error);
	printf ("Starting address:  ");
	print_addr(start);
	printf ("\nEnding address:  ");
	print_addr(end);
	putchar('\n');
}

/*
 * rdloop - continuous read of one location
 */

rdloop(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register int i;
	register short j;
	register char k;

	lastmem = (uchar *)atox(comm_args[1]);

	/* if in CSS RAM, check which clock we are on */ 
	if((unsigned)lastmem & (unsigned)0x80000000) {
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return(1);
	}

	switch(mref_siz) {
		case 0  :
		case 'l':
			while(con_in() == -1)
				long_throw = *(uint *)lastmem;
			break;
		case 'w':
			while(con_in() == -1)
				word_throw = *(ushort *)lastmem;
			break;
		case 'b':
			while(con_in() == -1)
				byte_throw = *(uchar *)lastmem;
			break;
	}
	return(0);
}

/*
 * wrloop - write loop on one location
 */

wrloop(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	register uint i;
	register ushort j;
	register uchar k;

	i = (uint)atox(comm_args[1]);
	j = (ushort)i;
	k = (uchar)i;

	lastmem = (uchar *)atox(comm_args[2]);

	/* if in CSS RAM, check which clock we are on */ 
	if((unsigned)lastmem & (unsigned)0x80000000) {
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return(1);
	}

	switch(mref_siz) {
		case 0  :
		case 'l':
			while(con_in() == -1)
				*(uint *)lastmem = i;
			break;
		case 'w':
			while(con_in() == -1)
				*(ushort *)lastmem = j;
			break;
		case 'b':
			while(con_in() == -1)
				*(uchar *)lastmem = k;
			break;
	}
	return(0);
}
	
/*
 * rdblock - read block loop
 */

rdblock(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register unsigned char *dp, *end;
	register int i;
	register short j;
	register char k;
	register unsigned int size = 0;
	register unsigned l,m;

	switch (arg_cnt) {
	case 0:
		break;
	case 1:
		lastmem = (unsigned char *)atox(comm_args[1]);
		size = 1;
		break;
	case 2:
		lastmem = (unsigned char *)atox(comm_args[1]);
		size = atox(comm_args[2]);
		break;
	}

	if((unsigned)lastmem & (unsigned)0x80000000) { /* if CSS, check clock */
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return(1);
	}

	if(!size)
		size = 1;

	printf ("Press ESC to exit.\n");
	end = (uchar *)((unsigned)lastmem + (unsigned)size);
	while(con_in() == -1) {
		dp = lastmem;
		switch(mref_siz) {
			case 0  :	/* no mref_siz specified by user */
			case 'l':
				while(dp < end) {
					i = *(uint *)dp;
					dp += 4;
				}
				break;
			case 'w':
				while(dp < end) {
					j = *(ushort *)dp;
					dp += 2;
				}
				break;
			case 'b':
				while(dp < end) {
					k = *(uchar *)dp;
					dp++;
				}
				break;
		}
	}
	return(0);
}


/*
 * wrblock - write block loop
 */

wrblock(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register uchar *dp, *end;
	register int i;
	register short j;
	register char k;
	register uint size = 1;
	register uint l,m;

	lastmem = (uchar *)atox(comm_args[1]);
	if((uint)lastmem & (uint)0x80000000) { /* if CSS, check clock */
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return(1);
	}

	i = (uint)atox(comm_args[2]);
	j = (ushort)i;
	k = (uchar)i;

	if (arg_cnt == 3)
		size = atox(comm_args[3]);

	if (!size) 
		size = 0x10000;

	printf ("Press ESC to exit.\n");
	end = (uchar *)((uint)lastmem + (uint)size);
	while (con_in() == -1) {
		dp = lastmem;
		switch (mref_siz) {
			case 0  :	/* no mref_siz specified by user */
			case 'l':
				while (dp < end) {
					*(uint *)dp = i;
					dp += 4;
				}
				break;
			case 'w':
				while (dp < end) {
					*(ushort *)dp = j;
					dp += 2;
				}
				break;
			case 'b':
				while (dp < end) {
					*(uchar *)dp = k;
					dp++;
				}
				break;
		}
	}
	return(0);
}

iready_prt()
{
	printf ("Increase ready count in arbiter.\n");
	iready();
}

/*
 * fmem - fill memory block
 */

fmem(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register unsigned char *dp, *end;
	register int i;
	register short j;
	register char k;
	unsigned size;

	size=0;
	lastmem = (unsigned char *)atox(comm_args[1]);
	/* if in CSS RAM check which clock we are on */
	if(lastmem >= (unsigned char*)0x80000000) 
	{
		if(clokpr(1))	 /* and if it returns bad, so do we. */
			return;
	}
	i=atox(comm_args[2]);
	j=i;
	k=i;
	if(arg_cnt ==3)
		size = atox(comm_args[3]);
	end = lastmem + size;
	dp = lastmem;
	switch(mref_siz) {
		case 0  :		/* no mref_siz specified by user */
		case 'l':
			while(dp < end) {
				*(unsigned *)dp =i;
				dp += 4;
			}
			break;
		case 'w':
			while(dp < end) {
				*(short *)dp =j;
				dp += 2;
			}
			break;
		case 'b':
			while(dp < end) {
				*(char *)dp =k;
				dp++;
			}
			break;
	}
}

rready_prt()
{
	printf ("Reset ready count in arbiter.\n");
	rready();
}

/*------------------------------------------------------------------------------
-----------------------------------------------------------------------------*/
css_reset1()
{
	int i;

	printf ("Reset CSS Bus.\n");
	css_reset();
	slotson();	/* enable boards who passed first time */
	wait_cnt(100);
	cpu_idle(0,0);	/* Go thru the whole table and put the CPUs in idle. */
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
init_css_prt ()
{
	printf ("Initialize CSS activity.\n");
	init_css ();
}

mslot(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	register uint mapnum;
	register uchar slotnum;
	register uchar slotadd;
	register uchar *i;
 
	if(clokpr(1))
		return(1);

	if(!arg_cnt) {	/* display map ram contents instead.*/
		for(mapnum = 0; mapnum != NUM_CSS_MAPS; mapnum++) {
			i = (uchar *)((unsigned)MAPBASE | (mapnum << 28));
			slotadd = (*i & 0x0f);	/* get the data. */
			slotnum = (*i >> 4) & 0x0f;
			printf("Map: %d  Slot: %X  addr: %X:\n", 
				mapnum, slotnum, slotadd);
		}
		printf("\n");
		return(0);
	}
	if(arg_cnt != 3) {	/* else, it's bad. */
		printf("Error: argument count should be 3.\n");
		return(1);
	}
	printf ("Set SPM map register.\n");
	mapnum = (uint)atox(comm_args[1]);
	slotnum = (uchar)atox(comm_args[2]);
	slotadd = (uchar)atox(comm_args[3]);
	i = (uchar *)((unsigned)MAPBASE | (mapnum << 28));
	*i = (slotnum << 4) | slotadd;
	return(0);
}


jsr(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	register int (*mem)();
 
	(int *)mem = (int *)(atox(comm_args[1]));
	if(arg_cnt == 1)
		(*mem)();
	if(arg_cnt == 2)
		(*mem)(atox(comm_args[2]));
	if(arg_cnt == 3)
		(*mem)(atox(comm_args[2]),atox(comm_args[3]));
	if(arg_cnt == 4)
		(*mem)(atox(comm_args[2]),atox(comm_args[3]),atox(comm_args[4]));
	if(arg_cnt == 5)
		(*mem)(atox(comm_args[2]),atox(comm_args[3]),atox(comm_args[4]),atox(comm_args[5]));
	if(arg_cnt == 6)
		(*mem)(atox(comm_args[2]),atox(comm_args[3]),atox(comm_args[4]),atox(comm_args[5]),atox(comm_args[6]));
	if(arg_cnt == 7)
		(*mem)(atox(comm_args[2]),atox(comm_args[3]),atox(comm_args[4]),atox(comm_args[5]),atox(comm_args[6]),atox(comm_args[7]));
}


extern int mtinit();
exemem(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uint	*MemBase;
	uchar	slot;
	int	laps_to_do, siz;
	int	MemBoardSize;	/* total MM size */
	int	MemTestSize;	/* tested size */
	uint	megs;
	uint	longs;

	/* process the arguments */
	slot = (uchar)atoi(comm_args[1]);	/* slots are decimal */

	if (arg_cnt == 2)
		MemTestSize = atox(comm_args[2]);
	else
		MemTestSize = 0;

	if ((slot >= Sbus_Num_Slot) || ((bdhere[slot]&BDTYPEMASK) != MEMTYPE)) {
		printf("No MM in slot %x (%d)\n", slot, slot);
		return(1);
	}

	MemBoardSize = get_mem_size(bdidsl(slot), 1);	/* MM size in Meg */

	if ((MemTestSize <= 0) || (MemTestSize > MemBoardSize))
		MemTestSize = MemBoardSize;

	HndlBerr = 1;
	MemBase = (uint *)0x80000000;
	cssmap (0, slot, 0);
	/* Init the memory to avoid edac errors */
	for (megs = 0; megs != MemTestSize; megs++) 
		for (longs = 0; longs != 0x100000; longs += 4) 
			*MemBase++ = 0;
			
	BusFault = 0; 			/* Insure BusFault hasn't been set */
	IntToMon = 0;
	Verbose = 1;
	IntToMon = 1;
	HndlBerr = 0;
	/* how many 32bytes are in this board? */
	/* # * megabyte/32 - 1 */
	mtinit((MemTestSize * 0x8000) - 1);

	Verbose = 1;
	IntToMon = 1;
	HndlBerr = 0;
}

reset_berr_loop()
{
	HndlBerr = 0x00;
	emulate = 0x00;
	printf ("Reset bus error loop\n");
}

set_berr_loop()
{
	HndlBerr = 0x01;
	emulate = 0x00;
	printf ("Set bus error loop\n");
}

berr_em()
{
	emulate = 0xff;
	printf ("Bus error emulation ON\n");
}

berr_em_n()
{
	emulate = 0x00;
	printf ("Bus error emulation OFF\n");
}

/*
 *	mov(comm_str, arg_cnt)
 *	char *comm_str;
 *	int arg_cnt;
 *
 *	mov<.[bwl]> src dest count  -  Move memory
 */

move_mem(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register unsigned int dest, src;
	register int i;
	int	loop = 0;
	int	incr;
	char	emulate_save = emulate;

	src = (unsigned int)atox(comm_args[1]);
	dest = (unsigned int)atox(comm_args[2]);
	count = (unsigned int)atox(comm_args[3]);

	if(*comm_args[0] == 'l') { 	/* loop. */
		printf("Press ESC to exit.\n");
		loop = 1;
	}
	else {
		printf ("Move memory.\n");
	}

	switch(mref_siz) {
		case 0  :		/* no mref_siz specified by user */
		case 'l':
			incr = 4;
			break;
		case 'w':
			incr = 2;
			break;
		case 'b':
			incr = 1;
			break;
	}

	do { 
		for(i = 0; i < count; i+=incr ) {
			emulate = 0xff;

			switch(mref_siz) {
			case 0  :
			case 'l':
				*(unsigned int *)dest = *(unsigned int *)src; 
				break;
			case 'w':
				*(unsigned short *)dest = *(unsigned short *)src; 
				break;
			case 'b':
				*(unsigned char *)dest = *(unsigned char *)src; 
				break;
			}

			emulate = emulate_save;
			dest += incr;
			src += incr;
		}

		if (loop)
			printf (".");

		if(ifesc())
			break;
		
	} while (loop);

	printf ("\n");
	return(0);
}


/*------------------------------------------------------------------------------
	compare_mem(comm_str, arg_cnt)
 	cmp<.[bwl]> dest src count  -  Compare memory
------------------------------------------------------------------------------*/
compare_mem(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{ 
	uint	start1, start2;
	register uint dest, src, i;
	uint looping = 0;
	uint stop_on_error = 0;
	uint error = 0;
	char	emulate_save = emulate;

	switch (*comm_str) {
 		case 'l':
			looping = 1;
			break;
		case 'e':
			looping = 1;
			stop_on_error = 1;
			break;
	}

	start1 = (uint)atox(comm_args[1]);
	start2 = (uint)atox(comm_args[2]);
	count = (uint)atox(comm_args[3]);

	if(!count)
		count = 1;		/* Default to at least one. */

	do {
		dest = start1;
		src = start2;
		emulate = 0xff;			/* Ignore bus errors. */
			switch(mref_siz) {
			case 'b':
				for (i=0; i < count; i += 1) {
					if(*(uchar *)dest != *(uchar *)src) {
						printf("Compare error -\n");
						printf("addr 0x%02x = %x, addr 0x%02x = 0x%x\n",
						dest,*(uchar *)dest,src,*(uchar *)src);
				          	if(stop_on_error) {
				   		  	error = 1;		
							break;
						}
					}
				        dest += 1;
				        src += 1;
				}
				break;
			case 'w':
				for (i=0; i < count; i += 2) {
					if(*(ushort *)dest != *(ushort *)src) {
						printf("Compare error -\n");
						printf("addr 0x%04x = %x, addr 0x%04x = 0x%x\n",dest,
						*(ushort *)dest,src,*(ushort *)src);

				       		if (stop_on_error) {
					 	 	error = 1;		
							break;
						}
					}
				dest += 2;
				src += 2;
				}
				break;
			case 0  :
			case 'l':
				for (i=0; i < count; i += 4) {
					if(*(uint *)dest != *(uint *)src) {
						printf("Compare error -\n");
						printf("addr 0x%08x = 0x%x, addr 0x%08x = 0x%x\n",
						dest,*(uint *)dest,src,*(uint *)src);
						if (stop_on_error) {
							error = 1;
							break;
						} 
					}  
					dest += 4;
					src += 4;
				} /* default case */
				break;
			} /* switch */
			
			if (ifesc()) {
				looping = 0;
				break;
			}

			if (error) {
				error = 0;
				printf("ESC to return to monitor, any key to continue.\n");
				if (getchar () == ESCAPE) {
					looping = 0;
					break;
				}
			}
 	emulate = emulate_save;

	} while (looping);
}

/*------------------------------------------------------------------------------
	search_mem : search for a memory pattern as follows.
------------------------------------------------------------------------------*/
search_mem(comm_str, arg_cnt)	/* search for a memory pattern as follows.. */
char *comm_str;
int arg_cnt;
{
	register unsigned int pattern, leng;
	register unsigned char *start;
	register uint i, found=0, i0;
	char 	 chr;				
	char	emulate_save = emulate;

	pattern = (unsigned int)atox(comm_args[1]);
	if(arg_cnt > 1)	/* if there is a two.. */
		start = (unsigned char *)atox(comm_args[2]);
	else	
		start = (unsigned char *)0; /* start it at zero.. */

	if(arg_cnt > 2)	/* if there is a two.. */
		leng = (unsigned int)atox(comm_args[3]);
	else
		leng = 0x20000;  /* only check length of whatever. */

	for(i = 0; i < leng; i++)	/* while memory to test... */
	{
		emulate = 0xff;
		got_berr = 0; /* make sure it's cleared. */
		switch(mref_siz)
		{
		case 0  :		/* no mref_siz specified by user */
		case 'l':
			i0 = *(unsigned int *)start;	/* read in value. */
			break;
		case 'w':
			i0 = *(unsigned short *)start;	/* read in value. */
			break;
		case 'b':
			i0 = *(unsigned char *)start;	/* read in value. */
			break;
		}
		emulate = emulate_save;	/* disallow bus errors. */
		if(got_berr) /* not usable memory.. */
			continue;	/* skip results, since it's bad. */
		switch(mref_siz)
		{
		case 0  :		/* no mref_siz specified by user */
		case 'l':
			if((unsigned int)pattern == (unsigned int)i0)
				found =1;	/* do it. */
			break;
		case 'w':
			if((unsigned short)pattern == (unsigned short)i0)
				found =1;	/* do it. */
			break;
		case 'b':
			if((unsigned char)pattern == (unsigned char)i0)
				found =1;	/* do it. */
			break;
		}
		if(found)	/* if we finally found it.. */	
		{
			printf("Found: Address %8X\n", start); /* print it. */
			printf("Continue? (Y/N): ");
			chr = Getchar();
			if (! (Yes (chr)) ) { 
				printf("\n");
				return;	/* exit. */
			}
			found = 0;	/* skip it. */
			printf("\n");
		}
		if(ifesc()) 
		{
			printf("Search halted. (%8X)\n", start); /*loop until told not to*/
			return;
		}
		start++;	/* searches by chars.. */
	}
	printf("Pattern not found.\n");
	return;	/* it's ok anyway. */
}

int
iopm_tst (comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	uint	block;
	uint	blk_cnt;
	uint	stat;
	uint	iom_num;
	
	printf ("Read IOPM device.\n");
	css_slot = (uchar)atoi(comm_args[1]);
	phys_dev = (uchar)atoi(comm_args[2]);
	block = atoi(comm_args[3]);
	blk_cnt = atoi(comm_args[4]);
	if (arg_cnt == 5) {
		sub_slot = (uchar)atoi(comm_args[5]);
		iom_num = logiom(css_slot);
		if (ios_conf[iom_num][sub_slot].slot_id != IOPHERE) {
			printf ("No IOPM in %x/%x (%d/%d).\n",
				css_slot, sub_slot, css_slot, sub_slot);
			return(1);
		}
	}
	else {
		sub_slot = (uchar)NO_SUB_SLOT;
		if (bdhere[css_slot] != IOPHERE) {
			printf ("No IOPM in %x (%d).\n",
				css_slot, css_slot);
			return(1);
		}
	}

	if (*comm_str == 'w')
		stat = xfer_iopm_dev (block, blk_cnt, 0);  /* reset rd flg wr */
	else
		stat = xfer_iopm_dev (block, blk_cnt, 1);  /* set rd flg - rd */

	if (stat) 
		show_iopm_error(stat);

	return(stat);
}

int
iopm_blk_num (comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	uint	blk_num;
	
	printf ("Get IOPM block number.\n");
	
	css_slot = (uchar)atoi(comm_args[1]);
	phys_dev = (uchar)atoi(comm_args[2]);
	log_dev = (uchar)atoi(comm_args[3]);

	if (arg_cnt == 4)
		sub_slot = atoi(comm_args[4]);
	else
		sub_slot = 0xFF;

	if ((blk_num = get_iopm_blk_num(0)) == (uint)0xFFFFFFFF) 
		return(1);

	if (arg_cnt == 4)
		printf ("Slice %d/%dd%ds%d starting block 0x%08X (%d)\n",
			css_slot, sub_slot, phys_dev, log_dev, blk_num, blk_num);
	else
		printf ("Slice %dd%ds%d starting block 0x%08X (%d)\n",
			css_slot, phys_dev, log_dev, blk_num, blk_num);
	
	return(0);
}
	

int
iopm_dev_type (comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	css_slot = (uchar)atoi(comm_args[1]);
	phys_dev = (uchar)atoi(comm_args[2]);
	if (arg_cnt == 3) 
		sub_slot = (uchar)atoi(comm_args[3]);
	else
		sub_slot = NO_SUB_SLOT;

	if (check_for_iopm (css_slot, sub_slot))
		return(1);
	
	return(get_iopm_dev_type());
}

int
check_for_iopm (c_slot, s_slot)
uint	c_slot;
uint	s_slot;
{
	uint iom_num;
	
	if (s_slot != NO_SUB_SLOT) {
		if ((iom_num = logiom(c_slot)) == NOBOARD) {
			printf ("No IOM is slot %x (%d).\n", c_slot, c_slot);
			return(1);
		}

		if (ios_conf[iom_num][s_slot].slot_id != IOPHERE) {
			printf ("No IOPM in %x/%x (%d/%d).\n",
				c_slot, s_slot, c_slot, s_slot);
			return(1);
		}
		return(0);
	}

	if (bdhere[c_slot] != IOPHERE) {
		printf ("No IOPM in %x (%d).\n", c_slot, c_slot);
		return(1);
	}
	return(0);
}

