/*
 * util.c
 *
 *	Monitor utilities
 *
 */

#include "types.h"
#include "global.h"
#include "routines.h"
#include "rtc.h"
#include "spm.h"

#define 	CTL_Q 		0x11
#define 	CTL_S 		0x13
#define 	ESCAPE 		0x1b

#define 	LINE_LIMIT 	80	/*  80 column output */
#define 	HALF_LINE 	40	/*  40 column output */

#define		JUNK

extern char escflag;

/*------------------------------------------------------------------------------
	checkloop () :
	Say 'loop?' to user, return response.
	If looping is disabled, just return a 0.
------------------------------------------------------------------------------*/
checkloop ()
{
	char ibuf [17];

	if(!loopstat) return (0);
	printf("Loop? (y/n) ");
	if((gets (ibuf, 16) == (char *)-1)) return (-1);
	if(Yes (*ibuf) || !*ibuf) return (1);
	return (0);
}

/*------------------------------------------------------------------------------
		dummy : for faking up menu entries. 
------------------------------------------------------------------------------*/
dummy() 
{ }

/*------------------------------------------------------------------------------
		dte : print the date this was compiled.
------------------------------------------------------------------------------*/
dte() 
{ 
	printf("%s",date);
}

/*------------------------------------------------------------------------------
	clokpr(prnt) :	 print the message when prnt = 0. saves text space.
			 return 1 if on local clock
			 return 0 if on css clock
------------------------------------------------------------------------------*/
clokpr(prnt)
int prnt;
{
	if((!powmain()) || which_clk()) { /* if main power off or local clk */
		if(prnt) {
			printf("Sorry, you are running on the local clock.\n");
			printf("CSS clock required for this function\n");
		}
		return(1);	/* on local clock */
	}
	return(0);	/* on css clock */
}

/*******************[ STRING MANIPULATION UTILITIES ]**************************/

/*------------------------------------------------------------------------------
	strcmp : return <0 if p1 < p2, 0 if ==, >0 if p1 > p2.
------------------------------------------------------------------------------*/
strcmp (p1, p2)
register uchar *p1, *p2;
{
	for (; *p1 == *p2; p1++, p2++)
		if (*p1 == '\0') return (0);

	return ((int)(*p1 - *p2));
}

/*------------------------------------------------------------------------------
	strcpy : Copy string s2 to s1.  s1 must be large enough. return s1.
------------------------------------------------------------------------------*/
char *
strcpy(s1, s2)
register char *s1, *s2;
{
	register char *os1;

	os1 = s1;
	while(*s1++ = *s2++);
	return(os1);
}

/*------------------------------------------------------------------------------
	strncpy : Copy string s2 to s1, for size of cnt.
	s1 must be large enough. return s1.
------------------------------------------------------------------------------*/
char *
strncpy(s1, s2, cnt)
register char *s1, *s2;
register cnt;	/* number of chars to copy. */
{
	register char *os1;
	register i;

	os1 = s1;				/* save it */
	for(i=0; i < cnt; i++)			/* copy n chars */
	  *s1++ = *s2++;			/* copy chars across */
	return(os1);
}

/*------------------------------------------------------------------------------
	atox : convert input string to hex
------------------------------------------------------------------------------*/
unsigned
atox (ibuf)
char *ibuf;
{
	register char *ib;
	register int i;
	register int tc;

	ib = ibuf;
	i = 0;

	while (*ib) {
		tc = *ib++;
		i = i << 4;
		if (tc >= '0') tc -= '0';
		if (tc <= 9) {
			i += tc;
		}
		else {
			if (tc >= ('A' - '0')) tc -= ('A' - '0');
			if (tc >= ('a' - 'A')) tc -= ('a' - 'A');
			if (tc <= 5) {
				i += tc + 10;
			}
		}
	}
	return (i);
}

/*------------------------------------------------------------------------------
  gets.c : Get a line of input from console and echo characters.
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
 Change carriage return to line feed and echo carriage return line feed.
 Keep getting character and store them into buf until a line feed 
 or carriage return is received.
 take care character delete \b
 take care line delete character  @
 user will never get carriage return in the buf 
------------------------------------------------------------------------------*/
char *
gets(buf, limit)
char *buf;
{
	register char *lp;
	register c, count;

	lp = buf;
	count = 0;
	while(count < limit)
	{
		c = Getchar() & 0177;	/* only take 7 bits, or macro char */
		putchar (c);		/* echo char */

		switch(c)
		{
		case '\n':
		case '\r':
			if (c == '\r') putchar ('\n');
			*lp = '\0';
			return (buf);
		case '\b':
			if (lp <= buf)
				lp = buf;
			else
			{
				count--;
				lp--;
			}
			continue;
		case '@':
			count = 0;
			lp = buf;
			putchar ('\n');
			continue;
		case '\033':
			lp = buf;
			*lp = '\0';
			putchar ('\n');
			return ((char *)-1);
		case '\032':		/* ^Z */
			if((Getchar() & 0177) == '\032') /* has to be two zz */
				promstart ();
			break;
		default:
			count++;
			*lp++ = c;
		}
	}
	buf [limit - 1] = '\0';
	return (buf);
}

/*------------------------------------------------------------------------------
	getchar : Get a character from the console.
		  This is standalone, no macro buffer input allowed routine.
-----------------------------------------------------------------------------*/
getchar()
{
	register char c;

	while((c = con_in ()) == -1);  /* wait until return char */
	return(c);
}

/*------------------------------------------------------------------------------
	putchar : Puts a character in the console.
-----------------------------------------------------------------------------*/
putchar(c)
register char c;
{
	char in1;
	if ((in1=con_in ()) == CTL_S)
		getchar ();	/* don't check for macro chars */
	if(in1==ESCAPE)		/* if escaped */
		escflag=1;	/* show it was pressed */
	if (c != ESCAPE)
	{
		con_out (c);
		if (c == '\n') con_out ('\r');
	}
	return;
}

/*------------------------------------------------------------------------------
		atoi() = ASCII to (decimal) integer conversion.
-----------------------------------------------------------------------------*/
atoi(s)
char s[];
{
	int i,n;

	for(i=n=0;(s[i] >= '0') && (s[i] <= '9'); i++)
		n = 10 * n + s[i] - '0';
	return(n);
}


/*------------------------------------------------------------------------------
	strlen : Standard string length counter.
-----------------------------------------------------------------------------*/
strlen(s)
char s[];
{
	char len;
	for(len=0;s[len] != '\0'; ++len);
	return(len);
}

/*------------------------------------------------------------------------------
	clscrn : Print specified number of lines.
------------------------------------------------------------------------------*/
clscrn(i) 
register  uint i; 
{ 
	register  int x;
	char	buffer[LINE_LIMIT + 1];

	if (i > LINE_LIMIT)
		i = LINE_LIMIT;
	
	for(x = 0; x != i; x++)
		buffer[x] = '\n';

	buffer[i] = '\0';
	printf("%s", buffer);
}

/*------------------------------------------------------------------------------
	anykey : Pressing any key will return to caller.
------------------------------------------------------------------------------*/
anykey() 
{ 
	printf("\n\rAny key to continue...");
	return(getchar());
}

/*------------------------------------------------------------------------------
	pline : Prints a pattern specified by 'c', 'n' number of times.
------------------------------------------------------------------------------*/
pline(c,n) 
char c;
uint n; 
{ 
	register uint i;
	char	buffer[LINE_LIMIT + 1];

	if (n > LINE_LIMIT)
		n = LINE_LIMIT;

	for(i=0; i != n; i++)
		buffer[i] = (char)c;

	buffer[n] = '\0';
	printf("%s",buffer);
}

/*------------------------------------------------------------------------------
	cline : Prints a string pointed to by *s, in the center of the line,
	and fills both ends of the line with a pattern specified by 'p'.
------------------------------------------------------------------------------*/
cline(p,s) 
char p;
char *s; 
{ 
	register int len=0;
	printf("\n");

	if((len=strlen(s)) % 2) 
		pline(p,(HALF_LINE - 1 - (len/2)));
	else 
		pline(p,(HALF_LINE - (len/2)));

	printf("%s",s);

	pline(p,(HALF_LINE - (len/2)));
}

/*------------------------------------------------------------------------------
	invld : Invalid Entry prompt. 
------------------------------------------------------------------------------*/
invld() 
{ 
	printf("\nInvalid Entry.\n");
}
/*-----------------------------------------------------------------------------
	 Yes : check for Y or y answer
-----------------------------------------------------------------------------*/
int
Yes (chr)
	char chr;
{

	if ((chr == 'Y') || (chr == 'y')) {
		return (1);
	}
	else 
		return (0);
}
/*-----------------------------------------------------------------------------
	 No : check for N or n answer
-----------------------------------------------------------------------------*/
int
No (chr)
	char chr;
{

	if ((chr == 'N') || (chr == 'n')) {
		return (1);
	}
	else 
		return (0);
}

/*-----------------------------------------------------------------------------
	 wait_cnt : wait loop-count amount of time
-----------------------------------------------------------------------------*/
wait_cnt(count)
uint count;
{
	count = count * 275;
	for (; count--;) ;
	return(0);
}
/******************************************************************************/

#ifdef	REMOVE
/* Tims star */
char	star_tab[] = {
	'|',
	'/',
	'-',
	'\\',
	};

int
star (comm_str, arg_cnt)
{

	int  pos = 0;
	int  count = atoi(comm_args[1]);

	while (!ifesc()) {

		putchar ('\r');
		putchar (' ');
		putchar (star_tab[(pos++ & 0x03)]);
		wait_cnt(count);

	}
	return(0);
}
#endif


char *days[]= {
	"Monday",
	"Tuesday",
	"Wednesday",
	"Thursday",
	"Friday",
	"Saturday",
	"Sunday",
};


/*
time of day
*/
int
tod(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	register struct rtc *rtcptr = RTC;

	if(arg_cnt == 0) {
		printf("\nTime: %02x:%02x:%02x", rtcptr->hour, rtcptr->min, 
			rtcptr->sec);
		printf("\nDate: %s %02x/%02x/%02x\n", 
			days[((rtcptr->day - 1) % 7)], rtcptr->month, 
			rtcptr->date, rtcptr->year);
	}
	else {
		/* stop the oscillator */
		rtcptr->control = C_WR;		/* set write bit */
		rtcptr->sec = 0x00;		/* stop bit = 0 */
		rtcptr->hour = H_KSTART;	/* kick start = 1 */
		rtcptr->control = 0x00;		/* write bit = 0 */
		wait_cnt(80);			/* wait 2 secs */	
		rtcptr->control = C_WR;		/* set write bit */
		rtcptr->hour = 0x00;		/* kick start = 0 */
		
		/* set time and date */
		
		if(arg_cnt >= 1)
			rtcptr->hour = (atox(comm_args[1]) & RTC_HOUR_MASK);
		if(arg_cnt >= 2)
			rtcptr->min =  (atox(comm_args[2]) & RTC_MIN_MASK);
		if(arg_cnt >= 3)
			rtcptr->sec = (atox(comm_args[3]) & RTC_SEC_MASK);
		if(arg_cnt >= 4)
			rtcptr->day = (atox(comm_args[4]) & RTC_DAY_MASK);
		if(arg_cnt >= 5)
			rtcptr->month = (atox(comm_args[5]) & RTC_MONTH_MASK);
		if(arg_cnt >= 6)
			rtcptr->date = (atox(comm_args[6]) & RTC_DATE_MASK);
		if(arg_cnt == 7)
			rtcptr->year = (atox(comm_args[7]) & RTC_YEAR_MASK);
		rtcptr->control = 0x00;		/* write bit = 0 */
	}

}



