/*
**	FILE NAME:	rwiad.c
**	TYPE:		C Source Code File
*/ 
#include "types.h"
#include "spm.h"
#include "rwi.h"
#include "rwiad.h"
#include "novram.h"
#include "rwicio.h"

#define ADC_START 0x0100		/* AD start command */
#define TIMEOUT 10000			/* AD timeout value */

extern int adloopflag;
extern uint ps_num;			/* Power supply pointer. */
extern struct novram *novram;

uint temp_conv();
char *volt_conv();
char *curr_conv();

char svolt[15];
char scurr[15];

/*
**	FUNCTION NAME:	get_sensor_data
**	SYNTAX:		unsigned char
**			get_sensor_data(s_ptr)
**			SENSOR *s_ptr;
**	ARGUMENT:
**			s_ptr: A pointer to a structure of type SENSOR
**
**	RETURN VALUE:	the data sampled by the sensor lower byte
**
**	OVERVIEW:	with information about pointer contained 
**			in the SENSOR descriptor pointed by the argument 
**			a read cycle is initiated and and data is read 
**			and returned to the caller
*/ 
uchar
get_sensor_data(s_ptr)
SENSOR *s_ptr;
{
	unsigned short read_data = 0;
	int timer = 0;
	unsigned char data;
	unsigned short *mux_chan;
	unsigned short *adc_ctrl;
	unsigned short *adc_data;
	
	mux_chan = MUX_LE;
	adc_ctrl = ADC_CTRL_PORT;	
	adc_data = ADC_DATA_PORT;
	
	/* 
	 * select multiplexer
	 */
	*mux_chan = s_ptr->muxsel;
	*adc_ctrl = s_ptr->adch | ADC_START;	/* start AD conversion */

	/* wait for it to finish or timeout */
	while(!(read_data & 0x8000) && (timer++ < TIMEOUT))
		read_data = *adc_data;			/* read the data */

	/* 
	 * Save the AD conversion number
	 */
	data =(unsigned char)( read_data & 0x00ff);
	return (data);
}

/*
**	FUNCTION NAME:	ad_test
**	SYNTAX:		ad_test(chan_num,adc_ctrl,adc_data,mux,type)
**			int chan_num;
**			unsigned short *adc_ctrl;
**			unsigned short *adc_data;
**			int type;
**			int mux;
**
**	ARGUMENT:
**			chan_num
**				AD channel number
**			adc_ctrl
**				AD control port address 
**			adc_data
**				AD data port address 
**			type
**				type of conversion
**			mux
**				mux selection 
*/ 
ad_test(chan_num,adc_ctrl,adc_data,mux,type)
int chan_num;				/* AD channel number */ 
ushort *adc_ctrl;		/* AD control port address */ 
ushort *adc_data;		/* AD data port address */ 
int type;				/* type of conversion */ 
int mux;				/* mux selection */	
{ 
	ushort read_data = 0; 
	int timer = TIMEOUT; 
	uint data;
	ushort *mux_chan;

	if (ACRW_HERE && (adc_ctrl != (ushort *)ADC_CNTL)) {
		printf ("ACRW present, no RWI ADC, invalid ADC selected\n");
		return(1);
	}

	mux_chan = MUX_LE;			/* Select the Channel. */
	*mux_chan = mux;			/* Select multiplexer. */
	*adc_ctrl = chan_num | ADC_START;	/* Start AD conversion. */

	while(!(read_data & ADC_EOC) && --timer)
		read_data = *adc_data;			/* Read the data */

	if(timer == 0) {
		printf("Timeout: ADC_PORT = 0x%08x CHAN = 0x%02x DATA = 0x%04x",
		adc_ctrl,chan_num,read_data);
		return (1);
	}

	printf("Chan = 0x%02x  Data = 0x%04x ",chan_num,read_data);
	data = read_data & 0x00ff;	/* Save the AD conversion number */

	switch (type) {
		case TEMP:
			temp_conv(data,0);
			break;
		case VOLTS:
			volt_conv(data,chan_num,mux,0);
			break;
		case CURRENT:
			if(chan_num == PS_MUX_0)
				(mux==MUX_VAL_4) ? (ps_num=0) : (ps_num=1);
			if(chan_num == PS_MUX_1)
				(mux==MUX_VAL_4) ? (ps_num=2) : (ps_num=3);
			if(chan_num == PS_MUX_2)
				ps_num = 4;
		
			curr_conv(data,0);
			break;
		default:
			printf ("Unrecognized A/D conversion requested\n");
			break;
	}

	if(adloopflag != 0)	/* if we are not looping send a CR */
		putchar('\r');			

	return(0);
}

/*
convert AD value to temperature. The conversion numbers
can be found in the RWI spec.
*/

uint
temp_conv(ad_val,flg)	
int  ad_val;
int  flg;				/* direct conversion or in string */
{
	ulong temp;
	ulong volts;
	ulong int_part;
 
	if(ad_val <= 0)
		int_part = 0;
	else
	{							
		ad_val = ad_val * 10000;	/* we don't have FPU */
		volts = ad_val * 176;		/* so we scale it */
		temp = ((volts - 300) / 703);
		int_part = temp/10000;
	}

	if (!flg) {
		printf("Temp = %02d C",int_part);
	}
	return(int_part);
}

char *
volt_conv(ad_val,chan,mx,flg)
uint ad_val;
int chan;
int mx;
int flg;
{
	uint int_part, flt_tens, flt_ones;
	ulong volts = 0;

	strcpy(svolt,"");
	switch(chan) {
		case REF:
		case PS_MUX_0:
		case PS_MUX_1:
		case PS_MUX_2:
			break;
		default:
			if(!flg)
			printf("No voltage on this channel\r");
			return(0);
	}
	switch(mx) {
		/* 5 volt supply */
		case MUX_VAL_1:
			volts = ad_val * 352;
			break;
		/* +12 volt supply */
		case MUX_VAL_2:
			volts = ad_val * 850;
			break;
		/* -12 volt supply */
		/* this is kind of strange because we're going in a negative */
		/* dir. A count of 0 out of the AD will equal -25.92 Volts */
		/* Change rate = -118.8 volts/bit */
		/* Any value greater than 0xda from the AD is out of range */
		/* therfore	0x00 = -25.92   */
		/*		0x75 = -12.00   */
		/*		0xda =   0.00   */

		/* -12 volt supply */
		case MUX_VAL_3:
				if(ad_val >= 0xda)
					volts = 0;
				else
					volts = 259200 - (1188 * ad_val);
			break;
		/* 5 volt supply */
		case MUX_VAL_5:
			volts = ad_val * 352;
			break;
		/* +12 volt supply */
		case MUX_VAL_6:
			volts = ad_val * 850;
			break;
		/* -12 volt supply */
		/* this is kind of strange because we're going in a negative */
		/* dir. A count of 0 out of the AD will equal -25.92 Volts */
		/* Change rate = -118.8 volts/bit */
		/* Any value greater than 0xda from the AD is out of range */
		/* therfore	0x00 = -25.92   */
		/*		0x75 = -12.00   */
		/*		0xda =   0.00   */

		case MUX_VAL_7:
				if(ad_val >= 0xda)
					volts = 0;
				else
					volts = 259200 - (1188 * ad_val);
			break;
	}

	/* we have to do all this garbage because the diagnostic */
	/* does not support floating point and printf */
	/* doesn't support padding on the left (ex %-02d) */
	int_part = volts / 10000;
	flt_tens = volts % 10000;
	flt_ones = flt_tens % 1000;
	flt_tens = flt_tens / 1000;
	flt_ones = flt_ones / 100;

	switch(mx) {
		/* 5 volt supply */
		case MUX_VAL_1:
		if(!flg) {
			printf("Volts = +%02d.%d%d DC",
				int_part,flt_tens,flt_ones); 
		} else {
			sprintf(svolt,"+%02d.%d%d V",
				int_part,flt_tens,flt_ones); 
		}
		break;
		/* +12 volt supply */
		case MUX_VAL_2:
		if(!flg) {
			printf("Volts = +%02d.%d%d  DC",
				int_part,flt_tens,flt_ones); 
		} else {
			sprintf(svolt,"+%02d.%d%d V",
				int_part,flt_tens,flt_ones); 
		}
		break;
		/* -12 volt supply */
		case MUX_VAL_3:
		if(!flg) {
			printf("Volts = -%02d.%d%d  DC"
				,int_part,flt_tens,flt_ones); 
		} else {
			sprintf(svolt,"-%02d.%d%d V"
				,int_part,flt_tens,flt_ones); 
		}
		
		break;
		/* 5 volt supply */
		case MUX_VAL_5:
		if(!flg) {
			printf("Volts = +%02d.%d%d  DC", int_part,flt_tens,flt_ones); 
		} else {
			sprintf(svolt,"+%02d.%d%d V", int_part,flt_tens,flt_ones); 
		}
		break;
		/* +12 volt supply */
		case MUX_VAL_6:
		if(!flg) {
			printf("Volts = +%02d.%d%d  DC", int_part,flt_tens,flt_ones); 
		} else {
			sprintf(svolt,"+%02d.%d%d V", int_part,flt_tens,flt_ones); 
		}
		break;
		/* -12 volt supply */
		case MUX_VAL_7:
		if(!flg) {
			printf("Volts = -%02d.%d%d  DC" ,int_part,flt_tens,flt_ones); 
		} else {
			sprintf(svolt,"-%02d.%d%d V" ,int_part,flt_tens,flt_ones); 
		}
		break;
	}
	return(&svolt[0]);
}

char *
curr_conv(data,flg)
uint data;
int flg;
{
	uint int_part=0, flt_tens=0, flt_ones=0;

	if(novram->main_pwr[ps_num].ps_id == 0xff) {
		if(!flg) { printf("  No Supply"); return(0); }
	}
	else conv_adc(MAIN_FIVE_VOLT_CURRENT,data,&int_part,&flt_tens);

	if(!flg)
		printf("Curr = %02d.%d%d  AMPs" ,int_part,flt_tens,flt_ones); 
	else
		sprintf(scurr," %02d.%d%d A", int_part,flt_tens,flt_ones); 

	return(&scurr[0]);
}

