/*
******************************************************************************
**	DESCRIPTION:	This file contain the Real World Interface Tests.
**		The Code relative to this sub system is to be found in rwi.c
**      	rwicio.c rwiad.c and rwiscc.c. Each module contains the routines
**      	required for exercising the related portion of the board
**		(the rwicio's, the ad converter and the scc's). More details of 
**      	the architecture of this board can be found in FS-02580-00 for 
**		rev 1A and in FS-02580-00 for rev 2B.  The Board REV2B implement
**		new features and modify some of the old ones.
******************************************************************************
*/ 

#include "types.h"
#include "globl.h"
#include "global.h"
#include "spm.h"		/* Include Service Module Definitions	*/
#include "novram.h"
#include "menu.h"
#include "loc_routines.h"
#include "routines.h"
#include "rwi.h"		/* Include Definitions for RWI Board	*/
#include "rwiad.h"		/* Include Definitions for RWI Board	*/
#include "rwicio.h"		/* Include cio definitions for REV checking */

#define NB_TIMER	3	/* Number of timers in the CIO */
#define LINES_PER_SCREEN 22


#define NFND	(SENSOR *)PNULL

extern rwi_test;
extern uint ps_num;			/* Power supply pointer. */
extern int adloopflag;
extern struct novram *novram;		/* needed to consult nbr of cabinet */

static struct cio *cio_1_ptr = CIO_1;

/*
 * This array of SENSORS defines the order of the test as well 
*/

static SENSOR adc[] = {
	{"CAB_0_TEMP_1 ", MUX_VAL_1, TEMP_MUX_0, TEMP, 0},
	{"CAB_0_TEMP_2 ", MUX_VAL_2, TEMP_MUX_0, TEMP, 0},
	{"CAB_1_TEMP_1 ", MUX_VAL_3, TEMP_MUX_0, TEMP, 1},
	{"CAB_1_TEMP_2 ", MUX_VAL_4, TEMP_MUX_0, TEMP, 1},
	{"CAB_2_TEMP_1 ", MUX_VAL_5, TEMP_MUX_0, TEMP, 2},
	{"CAB_2_TEMP_2 ", MUX_VAL_6, TEMP_MUX_0, TEMP, 2},
	{"CAB_3_TEMP_1 ", MUX_VAL_7, TEMP_MUX_0, TEMP, 3},
	{"CAB_3_TEMP_2 ", MUX_VAL_8, TEMP_MUX_0, TEMP, 3},
	{"CAB_4_TEMP_1 ", MUX_VAL_1, TEMP_MUX_1, TEMP, 4},
	{"CAB_4_TEMP_2 ", MUX_VAL_2, TEMP_MUX_1, TEMP, 4},
	{"CAB_5_TEMP_1 ", MUX_VAL_3, TEMP_MUX_1, TEMP, 5},
	{"CAB_5_TEMP_2 ", MUX_VAL_4, TEMP_MUX_1, TEMP, 5},
	{"CAB_6_TEMP_1 ", MUX_VAL_5, TEMP_MUX_1, TEMP, 6},
	{"CAB_6_TEMP_2 ", MUX_VAL_6, TEMP_MUX_1, TEMP, 6},
	{"PS_0_TEMP    ", MUX_VAL_1, TEMP_MUX_2, TEMP, 0},
	{"PS_1_TEMP    ", MUX_VAL_2, TEMP_MUX_2, TEMP, 1},
	{"PS_2_TEMP    ", MUX_VAL_3, TEMP_MUX_2, TEMP, 2},
	{"PS_3_TEMP    ", MUX_VAL_4, TEMP_MUX_2, TEMP, 3},
	{"PS_4_TEMP    ", MUX_VAL_5, TEMP_MUX_2, TEMP, 4},
	{"PS_0_PLUS_5  ", MUX_VAL_1, PS_MUX_0, VOLTS, 0},
	{"PS_0_PLUS_12 ", MUX_VAL_2, PS_MUX_0, VOLTS, 0},
	{"PS_0_MINUS_12", MUX_VAL_3, PS_MUX_0, VOLTS, 0},
	{"PS_0_I_MON   ", MUX_VAL_4, PS_MUX_0, CURRENT, 0},
	{"PS_1_PLUS_5  ", MUX_VAL_5, PS_MUX_0, VOLTS, 1},
	{"PS_1_PLUS_12 ", MUX_VAL_6, PS_MUX_0, VOLTS, 1},
	{"PS_1_MINUS_12", MUX_VAL_7, PS_MUX_0, VOLTS, 1},
	{"PS_1_I_MON   ", MUX_VAL_8, PS_MUX_0, CURRENT, 1},
	{"PS_2_PLUS_5  ", MUX_VAL_1, PS_MUX_1, VOLTS, 2},
	{"PS_2_PLUS_12 ", MUX_VAL_2, PS_MUX_1, VOLTS, 2},
	{"PS_2_MINUS_12", MUX_VAL_3, PS_MUX_1, VOLTS, 2},
	{"PS_2_I_MON   ", MUX_VAL_4, PS_MUX_1, CURRENT, 2},
	{"PS_3_PLUS_5  ", MUX_VAL_5, PS_MUX_1, VOLTS, 3},
	{"PS_3_PLUS_12 ", MUX_VAL_6, PS_MUX_1, VOLTS, 3},
	{"PS_3_MINUS_12", MUX_VAL_7, PS_MUX_1, VOLTS, 3},
	{"PS_3_I_MON   ", MUX_VAL_8, PS_MUX_1, CURRENT, 3},
	{"PS_4_PLUS_5  ", MUX_VAL_1, PS_MUX_2, VOLTS, 4},
	{"PS_4_PLUS_12 ", MUX_VAL_2, PS_MUX_2, VOLTS, 4},
	{"PS_4_MINUS_12", MUX_VAL_3, PS_MUX_2, VOLTS, 4},
	{"PS_4_I_MON   ", MUX_VAL_4, PS_MUX_2, CURRENT, 4}
};

SENSOR *get_sensor_info();
extern uchar get_sensor_data();
extern uint temp_conv();
extern char *volt_conv();
extern char *curr_conv();

extern	int	rwi_cio_init();
int	rwicioinit();
extern	int	margin();
extern	int	relay();
extern	int	display_cios_status();

/****************************************************************************
 *
 * 	R W I   T E S T  M E N U 
 * 
 ****************************************************************************/


struct init_menu rwi_tests [] = {
#ifdef	REMOVE
	"scci","", 
	"Scc internal loopback tests", 'c', 0, 0, testsccs, (char *)0,
	"scce1","", 
	"Scc external loopback tests J2 & J3",'c',0,0,scc_ext_a,(char *)0,
	"scce2","", 
	"Scc external loopback tests J3 & J4",'c',0,0,scc_ext_b,(char *)0,
	"cio","", 
	"CIO register & interrupt tests",'c', 0, 0,cio_tst, (char *)0,
	"ciot","", 
	"CIO timer tests", 'c', 0, 0,cio_time, (char *)0,
#endif
	
	"margin","ps_num  <l|h|n>", 
	"Power supply margin control",'c',2,2,margin,(char *)0,
	
	"relay","ps_num  <on|off>", 
	"Power supply relay control",'c',2,2,relay,(char *)0,
	
	"ciod","", 
	"Display CIO status", 'c', 0, 0,display_cios_status, (char *)0,
	
	"cior","", 
	"Read CIO ports", 'c', 0, 0,cio_in, (char *)0,
	
	"cioi","", 
	"Initialize the RWI CIO's", 'c', 0, 0, rwicioinit, (char *)0,
	
	"ad","", 
	"RWI A/D tests", 'c', 0, 0,ad_tst, (char *)0,
	
	"adl(oop)", "arg_cnt:\n0 - show table   1 - offset   3 - mux chan type",
	"A/D loop", 'c', 0, 3, ad_loop, (char *)0,
	
	"ads(ys)","", 
	"A/D Synoptical display", 'c', 0, 0, ad_sys, (char *)0,
	
	"rwi_ver","", 
	"RWI Rev Level", 'c', 0, 0, rwi_chk_ver, (char *)0,
	
	"", "", "", '\0', 0, 0, 0, (char *)0,
};

int
rwicioinit ()
{
	char	chr;	
	
	if (!clokpr(0)) { 
		printf ("Executing this command will reset the\n");
		printf ("CIO on the RWI/ACRW. System power will be lost.\n");
		printf ("Continue (y/n)?:  ");
		chr = getchar ();
		putchar('\n');
		if (No (chr))
			return(0);
	}		
	rwi_cio_init ();
	return(0);
}

#ifdef	REMOVE
/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	scc_ext_a		
**	SYNTAX:		scc_ext_a()		
**	CALLED BY:	menu system 
**	DESCRIPTION:	external loopback via J2 & J3
**	SYNOPSYS:	This function initializes both channels the on SCC,
**			invokes the external loopback routine display on
**			any error occuring, and initializes the SCC's. 
******************************************************************************
*/ 
scc_ext_a()		/* external loopback via J2 & J3 */
{
	int err;
	/* 
	 * Program the chips
	 */
	progscc(1, 1, 0);		/* init scc channel # 1 */
	progscc(2, 1, 0);		/* init scc channel # 2 */
	/* 
	 *   Do External Loopback If error, display return code
	 */
	if((err = sccenoint(1,2)) != 0 ) { 
		printf("Error = %d\n",err);
	}
	/*
	 *  initialize SCC's like we found them 
	 */
	initscc();
}
#endif

#ifdef	REMOVE
/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	scc_ext_b
**	SYNTAX:		scc_ext_b()
**	CALLED BY:	menu system	
**	DESCRIPTION:	external loopback via J3 & J4
******************************************************************************
*/ 
scc_ext_b()		/* external loopback via J3 & J4 */
{
	int err;

	if (ACRW_HERE) {
		printf ("ACRW present, no SCC1, scc_ext_b is an invalid test\n");
		return;
	}

	progscc(2, 1, 0);		/* init scc channel # 2 */
	progscc(3, 1, 0);		/* init scc channel # 3 */
	if((err = sccenoint(2,3)) != 0) {		/* Do external Loopback. */
		printf("Error = %d\n",err);
	}
	initscc();		/* Initialize SCC's like we found them. */
}
#endif

#ifdef	REMOVE
/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	cio_tst	
**	SYNTAX:		cio_tst()	
**	OVERVIEW:	verify that we can read or write CIO's and
**			generate int's.	
**	SYNOPSYS:	tell ISR that we are in diag mode
**			test the cio's local cio if compile option
**			TEST_SPM is defined
**		        reprogram  CIO if we are in TEST_SPM 
******************************************************************************
*/ 
cio_tst()	
{
	rwi_test = 1;		/* Tell ISR's we are doing diagnostics */
	test_rwi_cio(CIO_0);	/* Test CIO's */
	test_rwi_cio(CIO_1);	/* Test CIO's */
	rwi_test = 0;		/* Tell ISR's we are done with diagnostics */
}
#endif

#ifdef	REMOVE
/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	cio_time	
**	SYNTAX:		cio_time()	
**	OVERVIEW:	Verify all the timers on each 
**			CIO operates properly
******************************************************************************
*/ 
cio_time()	
{
	int tmr_idx;		/* a timer number  */

	rwi_test = 1;		/* Tell ISR's we are doing diagnostics */

/* Test Local Timers */
	for(tmr_idx = 0; tmr_idx < NB_TIMER; tmr_idx++) {	
		printf("Testing local timer %d\n", tmr_idx+1);  
		timer_tst(LOCCIO,tmr_idx);
	}

/* Test CIO.0 Timers */
	for(tmr_idx = 0; tmr_idx < NB_TIMER; tmr_idx++) {	
		printf("Testing CIO.0 timer %d\n", tmr_idx+1);  
		timer_tst(CIO_0,tmr_idx);
	}

/* Test CIO.1 Timers */
	for(tmr_idx = 0; tmr_idx < NB_TIMER; tmr_idx++) {	
		printf("Testing CIO.1 timer %d\n", tmr_idx+1);  
		timer_tst(CIO_1,tmr_idx);
	}

	initcio();		/* re-program cio. */
	rwi_test = 0;		/* tell ISR's we are doing diagnostics */
}
#endif

/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	ad_tst	
**	SYNTAX:		ad_tst()	
**	OVERVIEW:	do an AD conversion for each channel
**			and display results 
******************************************************************************
*/ 
ad_tst()
{
	unsigned short *adc_ctrl;
	unsigned short *adc_data;
	unsigned short mux_num; 
	unsigned short chan_num;
	int type;
	int i;
	int lines_printed = 0;
	char ch;

	if (ACRW_HERE) {
		printf ("ACRW present, no auxilliary ADC: invalid test\n");
		return;
	}

	adloopflag = 0;

	adc_ctrl = ADC_CTRL_PORT; 
	adc_data = ADC_DATA_PORT;
	
	/*
	 * Do test for all available channels
	 */
	for(i = 0; i < (sizeof(adc)/sizeof(SENSOR)); i++) {
		if (!(novram->cab_present[adc[i].cabinet].present))
			continue;	/* this cabinet doesn't exist */

		if (ifesc()) 
			break;

		mux_num = adc[i].muxsel;    /* Get appropriate Information */
		chan_num = adc[i].adch;
		type =	adc[i].chtyp; 
/* 
 * Perform the test for this SENSOR
 */
		printf("%s ",adc[i].name);
		if (ad_test(chan_num,adc_ctrl,adc_data,mux_num,type)) 
			break;

		putchar('\n');
		if (++lines_printed == LINES_PER_SCREEN) {
			lines_printed = 0;
			printf("Anykey to continue, ESC to quit");
			ch = Getchar(); /* get input from macro buffer*/
			printf("\r                               \r");
			if (ch == 0x1b)
				break;
		}
	}
	putchar('\n');
}



/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	ad_loop
**	SYNTAX:		ad_loop()	
**	CALLED BY:	Menu System	
**	OVERVIEW:	do an AD conversion for each channel and display results 
******************************************************************************
*/ 
ad_loop(comm_str, arg_cnt)		
char *comm_str;
int arg_cnt;
{
	unsigned short *adc_ctrl;
	unsigned short *adc_data;
	unsigned short chan_num;
	unsigned short mux_num; 
	unsigned short adc_offset; 
	int mux_val;
	int type;

	if (ACRW_HERE) {
		printf ("ACRW present, no auxilliary ADC: invalid test\n");
		return(0);
	}

	switch (arg_cnt) {
		case 0:
			show_adc_table();
			return(0);
		case 1:
			adc_offset = (unsigned short)atoi(comm_args[1]);
			if (adc_offset >= (sizeof(adc)/sizeof(SENSOR))) {
				printf ("adc offset %d out of range (0-%d).\n",
				adc_offset, ((sizeof(adc)/sizeof(SENSOR)) - 1));
				return(1);
			} 
			mux_num = adc[adc_offset].muxsel;
			chan_num = adc[adc_offset].adch;
			type = adc[adc_offset].chtyp;
			break;
		case 3:
			mux_val = atox(comm_args[1]);
			if ((mux_val < 1) || (mux_val > 8)) {
				printf("Invalid MUX Selection\n");
				return(1);
			}	
			mux_num = ((mux_val - 1) << 4);
			chan_num = atox(comm_args[2]);
			type = atox(comm_args[3]);
			break;
		default:
			printf ("Illegal number of arguments:\n");
			printf ("        0 - display adc table,\n");
			printf ("        1 - offset in adc table,\n");
			printf ("        3 - explicit sensor selection.\n");
			return(1);
	}

	adc_ctrl = ADC_CTRL_PORT;
	adc_data = ADC_DATA_PORT;

	switch(chan_num) {
	case TEMP_MUX_0:
		break;
	case 0x01:
		printf("Reference Voltage\n");
		break;
	case TEMP_MUX_1:
		if(mux_num == MUX_VAL_7 || 
		   mux_num == MUX_VAL_8) {
			printf("Not connected\n");
			return(1);
		}
		break;
	case TEMP_MUX_2:
		if(mux_num == MUX_VAL_6 ||
		   mux_num == MUX_VAL_7 ||
		   mux_num == MUX_VAL_8) {
			printf("Not connected\n");
			return(1);
		}
		break;
	case PS_MUX_0:
		break;
	case PS_MUX_1:
		break;
	case PS_MUX_2:
		if (mux_num == MUX_VAL_5 ||
		   mux_num == MUX_VAL_6 || 
		   mux_num == MUX_VAL_7 ||
		   mux_num == MUX_VAL_8) {
			printf("Not connected\n");
			return(1);
		}
		break;
	default:
		printf("Invalid ADC channel\n");
		return(1);
	}

	switch(type)
	{
	case NONE:
		printf("No Conversion \n");
		return(1);
	case TEMP:
		printf("Temperature Conversion \n");
		break;
	case VOLTS:
		printf("Voltage Conversion \n");
		break;
	case CURRENT:
		printf("Current Conversion \n");
		break;
	default:
		printf("Invalid Conversion type\n");
		return(1);
	}

	if (adloopflag = checkloop())     /* 1 is looping  */
		printf("Press <ESC> to terminate loop\n");

	do {
/* 
 * Do Sensor conversion
 */
		ad_test(chan_num,adc_ctrl,adc_data,mux_num,type);

		if(ifesc())
			break;
	} while(adloopflag);
	printf("\n");	/* next line, please.  cleaner display. */
}


/*
*************************************************************************fun**
**			(c) Copyright ARIX SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	ad_sys
**	SYNTAX:		ad_sys()	
**	OVERVIEW:	The sensors are displayed in a synoptical fashion
**			for a rapid verification of proper operation
******************************************************************************
*/ 
ad_sys()
{
	int i;
	int j;

	SENSOR *s_ptr;
	char str[15];
	unsigned char wrk[7];	/* work area to store sensor data */
	char vstr[7][15];	/* strings to contain voltages display info */
	char cstr[7][15];	/* strings to contain current display info */

	if (ACRW_HERE) {
		printf ("ACRW present, no auxilliary ADC: invalid test\n");
		return;
	}

	printf("\f\t\tA/D  Conversion Synopsis\n\n");
	printf("  CABINET   ");
	for(i=0; i<CAB_PER_SYS; i++) 
		printf("    %d    ",i);

	printf("\n+");
	pline ('-', 73);
	printf("+\n");

	printf("| TEMP.1.D ");
	for (j=0; j<CAB_PER_SYS; j++) {
		sprintf(str,"CAB_%1d_TEMP_1 ",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
		}
	}
	printf("|\n");

	printf("| TEMP.1.V ");
	for (j=0; j<CAB_PER_SYS; j++) {
		printf("|   %02d C ", temp_conv(wrk[j],1)); 
	}
	printf("|\n");

	printf("|          ");
	for (j=0; j<CAB_PER_SYS; j++) 
		printf("|        ");
	printf("|\n");

	printf("| TEMP.2.D ");
	for (j=0; j<CAB_PER_SYS; j++) {
		sprintf(str,"CAB_%1d_TEMP_2 ",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
		}
	}
	printf("|\n");

	printf("| TEMP.2.V ");
	for (j=0; j<CAB_PER_SYS; j++) {
		printf("|   %02d C ", temp_conv(wrk[j],1)); 
	}
	printf("|\n");

	printf("|          ");
	for (j=0; j<PS_PER_SYS; j++) 
		printf("|        ");
	putchar('+');
	pline ('-', 17);
	printf("+\n");

	printf("| PS.TEMP.D");
	for (j=0; j<PS_PER_SYS; j++) {
		sprintf(str,"PS_%1d_TEMP    ",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
		}
	}
	printf("|\n");

	printf("| PS.TEMP.V");
	for (j=0; j<PS_PER_SYS; j++) {
		printf("|   %02d C ", temp_conv(wrk[j],1)); 
	}
	printf("|\n");

	printf("|          ");
	for (j=0; j<PS_PER_SYS; j++) 
		printf("|        ");
	printf("|\n");

	printf("| PS.+5V.D ");
	for (j=0; j<PS_PER_SYS; j++) {
		sprintf(str,"PS_%1d_PLUS_5  ",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
			strcpy(vstr[j], volt_conv(wrk[j],
					 s_ptr->adch, s_ptr->muxsel, 1));
		}
	}
	printf("|\n");

	printf("| PS.+5V.V ");
	for (j=0; j<PS_PER_SYS; j++) {
		printf("|%s", vstr[j]); 
	}
	printf("|\n");

	printf("|          ");
	for (j=0; j<PS_PER_SYS; j++) 
		printf("|        ");
	printf("|\n");

	printf("| PS.+12V.D");
	for (j=0; j<PS_PER_SYS; j++) {
		sprintf(str,"PS_%1d_PLUS_12 ",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
			strcpy(vstr[j], volt_conv(wrk[j],
					 s_ptr->adch, s_ptr->muxsel, 1));
		}
	}
	printf("|\n");

	printf("| PS.+12V.V");
	for (j=0; j<PS_PER_SYS; j++) {
		printf("|%s", vstr[j]); 
	}
	printf("|\n");

	printf("|          ");
	for (j=0; j<PS_PER_SYS; j++) 
		printf("%s","|        ");
	printf("|\n");

	printf("| PS.-12V.D");
	for (j=0; j<PS_PER_SYS; j++) {
		sprintf(str,"PS_%1d_MINUS_12",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
			strcpy(vstr[j], volt_conv(wrk[j],
					 s_ptr->adch, s_ptr->muxsel, 1));
		}
	}
	printf("|\n");

	printf("| PS.-12V.V");
	for (j=0; j<PS_PER_SYS; j++) {
		printf("|%s", vstr[j]); 
	}
	printf("|\n");

	printf("|          ");
	for (j=0; j<PS_PER_SYS; j++) 
		printf("|        ");
	printf("|\n");

	printf("| PS.IMON.D");
	for(j=0; j<PS_PER_SYS; j++) {
		ps_num=j;
		sprintf(str,"PS_%1d_I_MON   ",j);
		if((s_ptr = get_sensor_info(str)) != NFND) {
			printf("|   %2x   ", (wrk[j]=get_sensor_data(s_ptr))); 
			strcpy(cstr[j], curr_conv(wrk[j],1));

		}
	}
	printf("|\n");

	printf("| PS.IMON.V");
	for (j=0; j<PS_PER_SYS; j++) {
		printf("|%s", cstr[j]); 
	}
	printf("|\n");

	putchar('+');
	pline ('-', 55);
	printf("+\n");
}


/*
******************************************************************************
**			(c) Copyright ARIX SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	get_sensor_info	
**	SYNTAX:		
**			SENSOR *
**			get_sensor_info(sensor_name)
**			char *sensor_name;	
**	ARGUMENT:	sensor_name
**				A string of character i.e CAB_0_TEMP_1
**	RETURN VALUE:	a pointer to a SENSOR descriptor contained in 
**			the adc array of SENSOR descriptor	
**	OVERVIEW:	A search is made in adc array to find a matching
**			name of sensor a NULL pointer is returned 
**			if name not found otherwise the address of the 
**			matching sensor descriptor is returned
******************************************************************************
*/ 
SENSOR *
get_sensor_info(sensor_name)
{
	SENSOR *sptr = NFND;
	register int i;
	
	for (i=0; i < sizeof(adc)/sizeof(SENSOR); i++) {
		if( strcmp(sensor_name,adc[i].name) == 0 ) {
			sptr = &adc[i];
			break;
		}
	}	
	if(adc[i].chtyp == CURRENT) {
		if(adc[i].adch == PS_MUX_0)
			(adc[i].muxsel == MUX_VAL_4) ? (ps_num = 0) : (ps_num = 1);
		if(adc[i].adch == PS_MUX_1)
			(adc[i].muxsel == MUX_VAL_4) ? (ps_num = 2) : (ps_num = 3);
		if(adc[i].adch == PS_MUX_2)
			ps_num = 4;
	}
	return (sptr);
}


/*
*************************************************************************fun**
**			(c) Copyright ARETE SYSTEMS CORPORATION 1988
**			    All Rights Reserved.
******************************************************************************
**	FUNCTION NAME:	rwi_chk_ver  
**	SYNTAX:		rwi_chk_ver	
**	ARGUMENT:	rwi_chk_ver
**	RETURN VALUE:	REV2B for new Board
**			REV1A for old Board
**	CALLED BY:	Temporary decision marking routine for 
**			the purpose of keeping both code for new and old 
**			board in the same prom.
**	OVERVIEW:	the RWI type bits (FS-02580-00 Rev.2B 2.2.5 page 3)
**			are accessed by reading CIO.1, Port A bit 2-0 
**			those three bits indicate the revion number as follows
**				0x00 = Revision 1A.
**				0x01 = Revision 2B.
**
******************************************************************************
*/ 
rwi_chk_ver()
{ 
	int rev;

	/* 
	 * Read Port A of CIO.1 Chip (z8036)
	 */
	rev = cio_1_ptr->portdata[0].reg;  
	/*
	 * Mask off unwanted bits
	 */ 
	rev &= 0x07;
	switch(rev)
	{
	case REV1A:
		printf("RWI is at Rev. 1A\n");
		break;
	case REV2B:
		printf("RWI is at Rev. 2B\n");
		break;
	case ACRW:
		printf("RWI is an ACRW. \n");
		break;
	default:
		printf("Currently not supported rev level/n");
		break;
	}
}


show_adc_table()
{
	
	char	ch;
	uint	i;
	uint	lines_printed = 0;
	
	for (i = 0; i != (sizeof(adc)/sizeof(SENSOR)); i++) {
		printf ("  %2d:  %s  is MUX:  %2x, CHAN:  %2x, CONV:  %2x\n",
			i, adc[i].name, ((adc[i].muxsel >> 4) + 1), 
			adc[i].adch, adc[i].chtyp);
	
		if (++lines_printed == LINES_PER_SCREEN) {
			lines_printed = 0;
			printf("Anykey to continue, ESC to quit");
			ch = Getchar(); /* get input from macro buffer*/
			printf("\r                               \r");
			if (ch == 0x1b)
				break;
		}
	}
}
