 /*
 * parse.c
 *
 *	parse (c_ptr) - parse command string
 *	char *c_ptr
 *
 *	This is the main command line interpreter.  It checks for special
 *	command functions, then searches the current menu for a match for
 *	a match.  If no match is found, an error message is printed.
 *
 *	On a command match with the current menu, one of two courses of
 *	action is taken, based upon the command type.  A 'c' signifies that
 *	this is a specific command, to be executed.  A 'm' means to traverse
 *	to this sub-menu.
 */

#include "types.h"
#include "menu.h"
#include "global.h"

extern char comm_args [MAXARGS][MAXARGSIZE];
extern char tmp_size;

parse (c_ptr, add)
char *c_ptr;
int	add;	/* if set, don't add to global buffer if needed. */
{
	register i, n_arg;
	struct menu *menup;

	menup = menu_stack[menu_ptr].m_ptr;

	/* Check for command match */
	while(*menup->comm_text != '\0') {
		/* Check for trivial input */
		if ((n_arg = argsplit (c_ptr)) == -1) return (0);
		if(comm_cmp (comm_args[0], menup->comm_text) == 0) {
			/* Check for help request '?' */
			if ((n_arg == 1) && (!strcmp ("?", comm_args[1]))) {
				show_help(menup);
				return(1);
			}
			/* Check for correct number of arguments */
			if ((n_arg < menup->lo_limit) || (n_arg > menup->hi_limit)) {
				argerr (menup, n_arg);
				return (-1);
			}
			/* Update memory reference size */
			mref_siz = tmp_size;  /* save this, if filled in. */
			/* Check for command */
			if (menup->comm_type == 'c')
			{	/* if valid command.. */

				i = (*menup->command.procedure)(c_ptr, n_arg);
				/* Check for menu traverse */
				return(i);	 /* and return the value. */
			} else if (menup->comm_type == 'm')
			{
				/* Traverse to new menu */
				menu_ptr++;
				menu_stack[menu_ptr].m_ptr = menup->command.menu_p;
				menu_stack[menu_ptr].name = menup->comm_desc;
				return (0);
				/* Anything else is an error */
			} else {
				printf ("Error: illegal command type.\n");
				return (-1);
			}
		}
		menup++;
	}
	/* At this point, the menu's been exhausted */
	if (!global) {
		if ((n_arg == 1) && (!strcmp ("?", comm_args[1]))) {
			if((i=com_search(c_ptr,n_arg, 1, add)) == -1 || !i)  {
				return (0);
			}
			else {
				printf ("Command not found\n");
			}
		}
		else {
			if((i=com_search(c_ptr,n_arg, 0, add)) == -1 || !i) {
				return (0);
			}
			else {
				printf ("Command not found\n");
			}
		}
	}
	return(-1);
}

argerr(menup, n_args)
int n_args;
struct menu *menup;
{
	printf ("Illegal number of arguments: ");
	if (menup->hi_limit == 0) printf ("None allowed.\n");
	else {
		printf ("was %d, should be ", n_args);
		if (menup->lo_limit != menup->hi_limit)
			printf ("between %d and ", menup->lo_limit);
		printf ("%d.\n", menup->hi_limit);
	}
}

/*
 * Comm_cmp
 *
 *	This puppy compares the comm_in with comm_ref, returning a 0 on
 *	equality, and -1 on inequality.
 *
 *	Comm_in is the string to be checked.
 *
 *	Comm_ref is the reference string, and follows the following rules:
 *		'(' begins the optional part of the command identifier string.
 *			The following characters need not be included in the
 *			input string.
 *		')' ends the optional part of the command identifier.
 *		'<' begins the optional postfix for the command.
 *		'<' ends the optional postfix for the command.
 *		'[' begins a list of legal postfix characters.  Only one
 *			character per brace set is allowed.
 *		']' ends a list of legal postfix characters.
 *
 *		Example:  for the compare string "mm(emory)<.[bwl]>", legal
 *		input strings are "mm", "mm.b", "mmemory" and so forth.
 *		An illegal input would be "mmem.b" or "mm.bw".
 *		Note that when the postfix is used with this command, the
 *		'.' is mandatory.
 *
 *		Currently, only one of each type of field is supported.
 */

comm_cmp (comm_in, comm_ref)
char *comm_in, *comm_ref;
{
	register flag, option;

	option = 0;

	/* First keyletters */
	while (*comm_in && *comm_ref && (*comm_in == *comm_ref))
	{
		comm_in++;
		comm_ref++;
	}
	if (!*comm_in && !*comm_ref)
		return (0);

	/* Optional keyletters begin with open paren */
	if (*comm_ref == '(')
	{
		comm_ref++;
		while (*comm_in && *comm_ref && (*comm_in == *comm_ref))
		{
			comm_in++;
			comm_ref++;
		}
		if (!*comm_in && !*comm_ref)
			return (0);
		/* Space past the close paren */
		while (*comm_ref && (*comm_ref++ != ')'));
	}

	/* Postfix =  String followed by optional character */
	if (*comm_ref == '<')
	{
		comm_ref++;
		option++;
		if (!*comm_in)
			return (0);
	}
	/* Check string */
	while (*comm_in && *comm_ref && (*comm_in == *comm_ref))
	{
		comm_in++;
		comm_ref++;
	}
	if (*comm_ref == '>')
		comm_ref++;
	while (*comm_ref == ' ')
		comm_ref++;
	/* Check for equality */
	if (!*comm_in && !*comm_ref)
		return (0);
	if (!*comm_in && !option)
		return (-1);
	/* clear old postfix saved in tmp_size */
	tmp_size = 0;
	/* Compare postfix character table */
	if ((*comm_ref == '[') && (*comm_in || !option))
	{
		flag = 0;
		comm_ref++;
		while (*comm_ref != ']')
		{
			if (*comm_in == *comm_ref)
			{
				flag++;
				tmp_size = *comm_in;	/* Store size in holding register */
			}
			comm_ref++;
		}
		comm_ref++;
		if (!flag)
			return (-1);
		comm_in++;
	}
	/* (if we put in multiples...) if (*comm_ref == '>') comm_ref++; etc... */
	/* Check for extraneous stuff */
	if (*comm_in)
			return (-1);
	return (0);
}

/*
 * Argsplit takes an input string and divides it into separate arguments,
 *	stored in the array comm_args.  The number of arguments - 1 is
 *	returned.
 *
 *	White space is spaces and tabs.
 *
 *	If there are more than MAXARGS arguments, MAXARGS - 1 is returned.
 *
 *	If an argument exceeds MAXARGSIZE size, it is truncated.
 *	If no arguments are detected -1 is returned.
 */

argsplit (c_ptr)
char *c_ptr;
{
	register char *cp;
	register argcnt, argsiz;

	argcnt = 0;
	while (1)
	{
		argsiz = 1;
		cp = comm_args[argcnt];
								/* Find an argument */
								/* Spaces and tabs are white space */
		while (*c_ptr == ' ' || *c_ptr == '\t') c_ptr++;
								/* Copy the argument */
		while (((*cp++ = *c_ptr) != ' ') && (*c_ptr != '\t') &&
										(argsiz++ < MAXARGSIZE) && *c_ptr)
			c_ptr++;
		cp--;
		*cp = '\0'; /* Terminate arg string */
		if (*c_ptr != ' ' && *c_ptr != '\t')		/* Truncate too long args */
			while (*c_ptr != ' ' && *c_ptr != '\t' && *c_ptr) c_ptr++;
		/* Check for end of input string */
		if (!*c_ptr)
		{		/* If there was white space at the end, decrement arg count */
			if (cp == comm_args[argcnt])
				argcnt--;
			return
				(argcnt);
		}
		if (argcnt == (MAXARGS - 1))
			return (argcnt);
		argcnt++;
	}
}

/* created 880316 m.tassano
   if parse() doesn't find a match for the command entered,
com_search() is called, sets up some globals for recursion, then
search() starts at the top of the menu structures and does a 
pre-order search for a match.  It will go down each directory
until either the menus are exhausted or a match is found.
	Return: TRUE (matched&executed) or FALSE (not found), (-1) for ERROR
*/
/* algorithm: Simple recursive search function
	search(menu)
		put menu into list
		check all 'c' type entries
			if a match is found,
				execute it
				return (1)
		if no match, search for sub-menus
		if an 'm' entry is found that isn't in the list
			x = search(menu)
			if x == match or error,
				return x
		else
			return FALSE    no further menus, no matching command here. 

*/

#define MAXMENUS 40

struct menu *mlist[MAXMENUS];
int mlistindx,n_arglobal;
char *mlistcmd;

com_search(curcmd,argcnt, helpset, add)
char *curcmd;
int argcnt;
int helpset;	/* 1 if this is a help parse. */
int add;	/* set if we are in add mode. */
{
int j;

	/* init list of menus checked */
	for(j=0;j<MAXMENUS; j++)
		mlist[j] = (struct menu *)(0);
	mlistindx = 0;		/* zero the index into the list of menus checked */
	mlistcmd = curcmd;  /* put command into global for search() to see */
	n_arglobal = argcnt; /* save the n_args from parse() */

	j = search( menu_stack[0].m_ptr, helpset, add); /* start at the top menu */

	return j;
}

int search(cmenu, helpset, add)
struct menu *cmenu;
int helpset;	/* set if this is a help parse. */
int add; /* set if we are in add mode. */
{
	int j;
	int found;

	/* place this menu on the list, increment the counter/index */
	mlist[mlistindx++] = cmenu;

	/* check all 'c' type entries for a match */
	while( *cmenu->comm_text != '\0' ) {
		if( cmenu->comm_type == 'c' )
			if(comm_cmp (comm_args[0], cmenu->comm_text) == 0) {
				if(!helpset)	/* checking limits.. */
				{
					/* Check for correct number of arguments */
					if ((n_arglobal < cmenu->lo_limit) || (n_arglobal > cmenu->hi_limit)) {
						argerr (cmenu, n_arglobal);
						return (-1);
					}
					/* Update memory reference size */
					mref_siz = tmp_size;  /* save this, if filled in. */
					printf("%s\n", cmenu->comm_text);
					j = (*cmenu->command.procedure)(mlistcmd,n_arglobal);
				}
				else
				{
					show_help (cmenu);
				}
				return (0); /* done with search.  Recurse back out */
			}
			
		cmenu++;
	}
	/* no command match found.  Look through the menu entries. */
	cmenu = mlist[mlistindx - 1];    /* start at the beginning again. */
	while( *cmenu->comm_text != '\0') {   /* search to the end if need be */
		if( cmenu->comm_type == 'm' ) {   /* is this a command? */
			found = 0;
			for(j=0; j<mlistindx; j++)       /* is this one in the list? */
				if( mlist[j] == cmenu->command.menu_p ) {
					found = (1);
					break;
				}
			if(!found) {  /* an unexplored menu level.  Follow it down */
				j = search(cmenu->command.menu_p, helpset, add); /* recurse.  This'll put cmenu in list */
				if(j != 1) /* match found and executed below somewhere or error */
					return j; /* so higher levels know we're done */
			}
		}
		cmenu++;  /* wasn't an 'm' type, look at next entry */
	}
	return (1);  /* didn't find it here, no 'm' types left. */
}

int
show_help (cmenu)
struct menu *cmenu;
{
	printf("%25s",cmenu->comm_text); /* show it. */
	printf("%50s\n",cmenu->comm_desc); /* show it. */
	printf("%60s\n",cmenu->comm_opt); /* show it. */
	if(cmenu->help)	/* if pointer is there. */
		printf("%s\n",cmenu->help); /* show it. */
	
	return(0);
}
