
/*------------------------------------------------------------------------------
	nov_util.c
------------------------------------------------------------------------------*/
#include "spm.h"
#include "types.h"
#include "novram.h"
#include "disp.h"
#include "ioa0.h"
#include "icb.h"
#include "icb_config.h"
#include "rwi.h"
#include "globl.h"
#include "global.h"
#include "rwi.h"
#include "rwicio.h"
#include "rwiad.h"
#include "dev.h"
#include "misc.h"

/*
**	Updated to allow structure changes for iosba
**	2 or less 1.8.7 and no IOSBA; 3 or greater 2.x IOSBA supported
*/

#define		ESCAPE		0x1b
#define		NOVRAM_VER	3
#define		OLD_NOVRAM_VER	2

#define		MEM_TYPE_MASK	7

extern char *MEM_TYPE [];
extern ushort crctab[];
extern uchar bdhere[];
extern char comm_args[][MAXARGSIZE];
extern uchar *buff;
extern char validmap;

extern uint get_device_info ();

extern struct novram *novram;		/* Points to Base of NVRAM. */
extern struct new_novram *new_novram;	/* Points to Base of NVRAM. */

extern uint boot_type;

#define	MAX_CRC_TYPE	2
/*------------------------------------------------------------------------------
	up_nov_chksum() : Update the novram checksum.
------------------------------------------------------------------------------*/
up_nov_chksum(comm_str, arg_cnt)	
char *comm_str;
int arg_cnt;
{
	uint	crc_type = 0;

	if (arg_cnt)
		crc_type = atoi(comm_args[1]);
	
	if (crc_type > MAX_CRC_TYPE) {
		printf ("crc:  crc_type %d is out of range (0-%d).\n",
			crc_type, MAX_CRC_TYPE);
		return(1);
	}

	switch (crc_type) {
		case 0:
			novram->prom_crc = nov_chk_sum(crc_type);
			break;
		case 1:
			novram->rwi_val_crc = nov_chk_sum(crc_type);
			break;
		case 2:
			novram->shared_crc = nov_chk_sum(crc_type);
			break;
		default:
			printf ("crc:  crc_type %d is out of range (0-%d).\n",
				crc_type, MAX_CRC_TYPE);
			return(1);
	}
	return(0);
}

/*------------------------------------------------------------------------------
	nov_chk_sum() : Checksum the novram, and return value calculated.
					Starting at location "st".
			flg == 1: warning low and high levels
			flg == 0: boot paths, configuration
------------------------------------------------------------------------------*/
nov_chk_sum(flg)
register unsigned flg;
{
	register unsigned char *nv;
	register crc=0, i, end;			/* temp variables. */

	switch (flg) {
		case 0:
			nv = (unsigned char *)NOVRAM;
			end = ((uint)(&novram->diag_fly_by));
			break;
		case 1:
			nv = ((unchar *)(&novram->inlet_air_low_warning));
			end = ((uint)(&novram->console_baud_rate));
			break;
		case 2:
			nv = ((unchar *)(&novram->inlet_air_low_warning));
			end = ((uint)(&novram->shared_crc));
			break;
		default:
			return(0);
	}

#ifdef DEBUG
printf("1.) nov_chk_sum: prom_crc=%x nv=%x end=%x\n",novram->prom_crc,nv,end);
#endif

	for(i=(uint)nv; i < end; i++, nv++)
		crc = (crc<<8) ^
		      crctab[((((crc>>(8))&0x00ff) ^ (*nv & 0x00ff))&0x00ff)];

#ifdef DEBUG
printf("2.) nov_chk_sum: prom_crc=%x nv=%x end=%x\n",novram->prom_crc,nv,end);
#endif
	return(crc & 0x0000ffff);	/* and return to calling program. */
}

int
display_ps_info (ps_number)
uint	ps_number;
{

	char *pwr="for Power Supply #";
	char *dis="DIS", *ena="EN", *abl="ABLED.";
	char *cur="Sensor is";
	register struct novram *nv = NOVRAM;
 
	clscrn(24);			/* Clear the screen. */
	cline('=',"[ Power Supply Sensor Enable/Disable ]");

	printf("\n  1.) I.D. %s%x......................0x%xh.",
		pwr, ps_number, nv->main_pwr[ps_number].ps_id);
	printf("\n  2.) Scaling factor %s%x............0x%xh.",
		pwr, ps_number, nv->main_pwr[ps_number].scale_factor);
	printf("\n  3.) Positive Five Volt %s%x %s.....%s%s",
		pwr, ps_number, cur,
		(nv->main_pwr[ps_number].five_volt)?ena:dis,abl);
	printf("\n  4.) Positive Twelve Volt %s%x %s...%s%s",
		pwr, ps_number, cur,
		(nv->main_pwr[ps_number].twelve_volt)?ena:dis,abl);
	printf("\n  5.) Negative Twelve Volt %s%x %s...%s%s",
		pwr, ps_number, cur,
		(nv->main_pwr[ps_number].neg_twelve_volt)?ena:dis,abl);
	printf("\n  6.) Five Volt Current %s%x %s......%s%s",
		pwr, ps_number, cur,
		(nv->main_pwr[ps_number].five_volt_current)?ena:dis,abl);
	printf("\n  7.) Power Supply Temperature %s%x %s...%s%s",
		pwr, ps_number, cur,(nv->main_pwr[ps_number].temp)?ena:dis,abl);
	
	return(0);
}


int
edit_ps_info(ps_number)
uint	ps_number;
{

	char	b[8];
	uint	x;
	register struct novram *nv = NOVRAM;

	while (1) {
		printf("\nEnter the Parameter to set, ESC to quit edit:");
		switch(getchar()) {
			case '1':
				printf("\nEnter the PS I.D. (hex):");
				gets(b,3);
				nv->main_pwr[ps_number].ps_id = atox(b);
				break;
			case '2':
				printf("\nEnter the Scale Factor (hex):");
				gets(b,3);
				x = atox(b);
				if(x == 0) { 
					invld(); 
					break; 
				}
				nv->main_pwr[ps_number].scale_factor = x;
				break;
			case '3':
				nv->main_pwr[ps_number].five_volt = 
				(nv->main_pwr[ps_number].five_volt) ? 0 : 1;
				break;
			case '4':
				nv->main_pwr[ps_number].twelve_volt =
				(nv->main_pwr[ps_number].twelve_volt) ? 0 : 1;
				break;
			case '5':
				nv->main_pwr[ps_number].neg_twelve_volt =
				(nv->main_pwr[ps_number].neg_twelve_volt)?0:1;
				break;
			case '6':
				nv->main_pwr[ps_number].five_volt_current =
				(nv->main_pwr[ps_number].five_volt_current)?0:1;
				break;
			case '7':
				nv->main_pwr[ps_number].temp =
				(nv->main_pwr[ps_number].temp) ? 0 : 1;
				break;
			case ESCAPE:
				return(0);
			default :
				invld();
				break;
		}
		display_ps_info(ps_number);
		clscrn(8);		
	}
}

int
display_cab_info(cab_num)
uint	cab_num;
{
	char *cur="Sensor is";
	char *dis="DIS", *ena="EN", *abl="ABLED.";
	register struct novram *nv = NOVRAM;

	clscrn(24);		/* Clear the screen. */
	cline('=',"[ Cabinet Sensor Enable/Disable ]");

	printf("\n\n  1.) Cabinet %d %s......... %s%s", cab_num,cur,
		(nv->cab_present[cab_num].present) ? ena:dis, abl);
	printf("\n  2.) Cabinet %d Inlet Air %s %s%s", cab_num,cur,
		(nv->cab_present[cab_num].inlet_air)?ena:dis, abl);
	printf("\n  3.) Cabinet %d Exhaust Air %s .... %s%s", cab_num,cur,
		(nv->cab_present[cab_num].exhaust_air) ? ena :dis, abl);
	if(cab_num == 0 && ACRW_HERE){
		printf("\n  4.) Fan Failure Alarm is ...%s%s",
			(nv->fan_stat == ENABLE_FAN_ALRM)?ena:dis,abl);
	}	
	
	return(0);
}

int
edit_cab_info(cab_num)
uint	cab_num;
{

	register struct novram *nv = NOVRAM;

	while (1) {
		printf("\nChoose the sensor to toggle; ESC to quit edit.");

		switch(getchar()) {
			case '1':
				nv->cab_present[cab_num].present =
				(nv->cab_present[cab_num].present) ? 0 : 1;
				break;
			case '2':
				nv->cab_present[cab_num].inlet_air =
				(nv->cab_present[cab_num].inlet_air) ? 0 : 1;
				break;
			case '3':
				nv->cab_present[cab_num].exhaust_air =
				(nv->cab_present[cab_num].exhaust_air) ? 0 : 1;
				break;
			case '4':
				if(cab_num == 0 && ACRW_HERE){ 
					nv->fan_stat = 
					(nv->fan_stat & ENABLE_FAN_ALRM)?
					DISABLE_FAN_ALRM : ENABLE_FAN_ALRM;
				}
				break;
			case ESCAPE:
				return(0);
			default :
				invld();
				break;
		}
		display_cab_info(cab_num);
		clscrn(8);		
	}
}


/*------------------------------------------------------------------------------
	nvedit: Nvram edit and display.
------------------------------------------------------------------------------*/
nvedit()
{
	register uint   cnt0;
	register struct novram *nv = NOVRAM;
	char quit_loop;
	char next_loop;
	char edit_flag;
	char number_ps;
	char num_cabs;

	if (ACRW_HERE) {
		number_ps = 2;
		num_cabs = 2;
	}
	else {
		number_ps = PS_PER_SYS;
		num_cabs = CAB_PER_SYS;
	}

	nov_cpy (nv, buff);	/* save original novram, for recovery */

start_over:
	quit_loop = 0;
	next_loop = 0;
	edit_flag = 0;
	while(!quit_loop && !next_loop) {
		for(cnt0 = 0; cnt0 != number_ps; cnt0++) {

			display_ps_info (cnt0);

			printf("\n\n  <Esc> = Quit with no changes.");
			printf("\n    <e>   = Edit this screen.");
			printf("\n    <q>   = Quit prompt and save changes.");
			printf("\n    <c>   = Cabinet sensor menu.");
			printf("\n    <CR>  = Next power supply.");
			printf("\n    <s>   = Start over.\n\n");
			pline('-',80);
			printf("\n");

			switch (getchar()) {
				case ESCAPE:
					nov_cpy (buff, nv);
					return (0);
				case 'e':
				case 'E':
					edit_flag = 1;
					break;
				case 'q':
				case 'Q':
					quit_loop = 1;
					break;
				case 'c':
				case 'C':
					next_loop = 1;
					break;
				case 's':
				case 'S':
					nov_cpy (buff, nv);
					goto start_over;
					break;
				default:
					break;
			}
			
			if (quit_loop || next_loop)
				break;
			
			if (!edit_flag) {
				continue;
			}
			else {
				edit_flag = 0;
				edit_ps_info (cnt0);
			}
		}
	}

	next_loop = 0;
	while(!quit_loop) { /* Cabinet Sensors */

		for(cnt0 = 0; cnt0 != num_cabs; cnt0++) {

			display_cab_info(cnt0);

			printf("\n\n  <Esc> = Quit with no changes.");
			printf("\n    <e>   = Edit this screen.");
			printf("\n    <q>   = Quit prompt and save changes.");
			printf("\n    <CR>  = Next cabinet.");
			printf("\n    <s>   = Start over.\n\n");
			pline('-',80);
			printf("\n");

			switch (getchar()) {
				case ESCAPE:
					nov_cpy (buff, nv);
					return (0);
				case 'e':
				case 'E':
					edit_flag = 1;
					break;
				case 'q':
				case 'Q':
					quit_loop = 1;
					break;
				case 's':
				case 'S':
					nov_cpy (buff, nv);
					goto start_over;
					break;
				default:
					break;
			}
			
			if (quit_loop)
				break;
			
			if (!edit_flag) {
				continue;
			}
			else {
				edit_flag = 0;
				edit_cab_info(cnt0);
			}
		}
	}

	if(nov_chk_sum(0) != nv->prom_crc) {
		clscrn(24);
		printf ("Save Changes? (y/n):");
		if (Yes (getchar()))
			up_nov_chksum();
		else 
			nov_cpy (buff, nv);
	}
	putchar('\n');
	return(0);
}

/*------------------------------------------------------------------------------
	nov_boot_path() : Modifies the Default boot path in the NVRAM
------------------------------------------------------------------------------*/
nov_boot_path(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	char b[PATH_LENGTH],x;

	switch (*(comm_str)) {
		case 'c':
			printf("Clear the boot paths? (y/n)\n");
			x = getchar ();
			if ( ! Yes (x) )
				return(0);

			for(x=0; x != PATH_LENGTH; x++) {
				novram->spm_boot_disk[x] = (char)0;
				novram->diag_boot[x] = (char)0;
				novram->prom_crc = nov_chk_sum (0);
			}
			return(0);
		case 'd':
			printf("Enter new config default path; ESC to quit\n");
			if(gets(b,31) == -1) 
				return(0);
			strcpy(novram->diag_boot,b);
			novram->prom_crc = nov_chk_sum (0);
			break;
		case 'r':
			printf("Enter new SPM default path; ESC to quit\n");
			if(gets(b,31) == -1) 
				return(0);
			strcpy(novram->spm_boot_disk,b);
			novram->prom_crc = nov_chk_sum (0);
			break;
		case 'v':
			break;
	}
	printf("Runtime code default boot path = %s\n", novram->spm_boot_disk);
	printf("Configuration default boot path = %s\n", novram->diag_boot);
	return(0);
}


/*------------------------------------------------------------------------------
	c_dfb: Change the Diagnostic Fly By flag.
------------------------------------------------------------------------------*/
c_dfb()
{
	novram->diag_fly_by = (novram->diag_fly_by) ? 0 : 1;
	printf("\nDiagnostic Fly By flag is currently %s.\n",
		(novram->diag_fly_by) ? "SET" : "NOT SET");
	up_nov_chksum(0);		/* Update the novram checksum. */
}

/*------------------------------------------------------------------------------
	c_atb: Change the autoboot flag.
------------------------------------------------------------------------------*/
c_atb()
{
	novram->autoboot = (novram->autoboot) ? 0 : 1;
	printf("\nAutoboot is currently %sABLED\n",(novram->autoboot)?"EN":"DIS");
	up_nov_chksum(0);		/* Update the novram checksum. */
}

/*---[tgm]----------------------------------------------------------------------
nov_def_path : Verifies that the path in the novram is valid.
	       If not default to 1st IOM, 1st controller, drive 0.
------------------------------------------------------------------------------*/
nov_def_path()
{
	char *pathptr;
	char  path_error=0;

	pathptr=(char *)&novram->spm_boot_disk[0];	/* Get default. */

	get_device_info (pathptr);

	if (boot_type == DV_NO_DEV) {
		printf("Illegal syntax in default SPM path %s\n", pathptr);
		path_error += 1;
	}

	pathptr=(char *)&novram->diag_boot[0];		/* Get default. */

	get_device_info (pathptr);
	
	if (boot_type == DV_NO_DEV) {
		printf("Illegal syntax in default config path %s\n", pathptr);
		path_error += 2;
	}

	if (path_error) 
		update_boot_path (path_error);
	else
		return (0);

	return (1);
}


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
show_config()	/* show novram configuration. */
{
	register unsigned num;
	register unsigned sbus_slot;
	register unsigned iom_num, slot;
	char	opt_string[16];
	int	arg;
	int	mem_type;

	if (!validmap) {
		printf ("Configuration Table not valid, execute cl css.\n");
		return;
	}

	printf("\n");
	for(num=0; num < PS_PER_SYS; num++) {
		printf("Power supply %d type is ",num);
		switch(novram->main_pwr[num].ps_id ) {
			case SWITCHING_POWER:
				printf("Switching Power.\n");
				break;
			case HC_POWER:
				printf("HC Power.\n");
				break;
			case NO_POWER_SUPPLY:
				printf("No P/S Detected.\n");
				break;
			default:
				printf("Unknown type.\n");
				break;
		}
	}

	printf("Runtime code default = %s\n",novram->spm_boot_disk);
	printf("Configuration default = %s\n",novram->diag_boot);
	printf("Default SPM port baud rates are:\n");
	printf("    Console port : %d\n", novram->console_baud_rate);
	printf("      Modem port : %d\n", novram->modem_baud_rate);
	printf("    Printer port : %d\n", novram->printer_baud_rate);
	printf("        UPS port : %d\n", novram->ups_baud_rate);
	printf("Diagnostic AREA NV RAM checksum is : %4X\n\n",
	nov_chk_sum(0));

	printf("\nPress ESC to quit, or");
	if ((anykey()) == ESCAPE) {
		printf("\n");
		return(0);
	}
	clscrn (25);

	for (sbus_slot = 0; sbus_slot != Sbus_Num_Slot; sbus_slot++) {
		opt_string[0] = '\0';
		switch(novram->css_slot[sbus_slot] & BDTYPEMASK) {
			case SPMTYPE: printf("SPM    "); break;
			case MEMTYPE: 
				printf("MM     ");
				arg = bdidsl(sbus_slot); 
				mem_type = 
					(get_mem_size(arg, 0) & MEM_TYPE_MASK);
				strcpy (opt_string, MEM_TYPE[mem_type]);
				break;
			case CPUTYPE: printf("PM     "); break;
			case DPMTYPE: printf("DPM    "); break;
			case IOMTYPE: printf("IOM    "); break;
			case IOPTYPE: printf("IOPM   "); break;
			case VAMTYPE: printf("VAM    "); break;
			case NOBOARDTYPE: continue; break;
			default: printf("UNKNOWN"); break;
		}
		printf(" module in slot 0x%x (%d)  %s\n", 
				sbus_slot, sbus_slot, opt_string);
	}

	printf("\nPress ESC to quit, or");
	if ((anykey()) == ESCAPE) {
		printf("\n");
		return(0);
	}
	clscrn (25);

	switch (novram->novram_ver) {
		case OLD_NOVRAM_VER:
			show_config_ioa();
			break;
		case NOVRAM_VER:
			show_config_iosba();
			break;
		default:
			break;
	}
	return(0);
}

show_config_iosba()
{
	uint	iom_num;
	uint	slot;

	for(iom_num = 0; iom_num < IOM_PER_SYS; iom_num++) {
		if (new_novram->new_iom_slot[iom_num].ext_bd_type == IOSBA_HERE) {
			for (slot = 0; slot != MAX_CSS_SLOT; slot++) {
				switch(new_novram->new_iom_slot[iom_num].io_bds.ios_slot[slot]) {
					case IOPHERE:
						printf ("IOPM     ");
						break;
					case VAMHERE:
						printf ("VAM      ");
						break;
					case NOBOARD: continue; break;
					default:
						printf ("UNKNOWN  ");
						break;
				}
				printf(" module, IOM 0x%x CSS slot 0x%x (%d) IO slot 0x%x (%d)\n",
				iom_num,new_novram->new_iom_slot[iom_num].iom_slot, 
				new_novram->new_iom_slot[iom_num].iom_slot, slot, slot);
			}
		}
		else {
			for(slot = 0; slot < ICB_DEVS_PER_IOA; slot++) {
				switch(new_novram->new_iom_slot[iom_num].io_bds.icb_slot[slot]) {
					case EDT:  printf("EDT    "); break;
					case SCSI: printf("SCSI   "); break;
					case GCP:
					case EGCP: printf("GC     "); break;
					case MAC:  printf("MAC    "); break;
					case ICB_NO_BOARD: continue; break;
					default:   printf("UNKNOWN"); break;
				}
				printf(" module, IOM 0x%x CSS slot 0x%x (%d) ICB slot 0x%x (%d)\n",
				iom_num,new_novram->new_iom_slot[iom_num].iom_slot, 
				new_novram->new_iom_slot[iom_num].iom_slot, slot, slot);
			}
		}
	}
}


show_config_ioa()
{
	uint	iom_num;
	uint	slot;

	for(iom_num = 0; iom_num < IOM_PER_SYS; iom_num++) {
		for(slot = 0; slot < ICB_DEVS_PER_IOA; slot++) {
			switch(novram->iom_slot[iom_num].icb_slot[slot]) {
				case EDT:  printf("EDT    "); break;
				case SCSI: printf("SCSI   "); break;
				case GCP:
				case EGCP: printf("GC     "); break;
				case MAC:  printf("MAC    "); break;
				case ICB_NO_BOARD: continue; break;
				default:   printf("UNKNOWN"); break;
			}
			printf(" module, IOM 0x%x CSS slot 0x%x (%d) ICB slot 0x%x (%d)\n",
			iom_num,novram->iom_slot[iom_num].iom_slot, 
				novram->iom_slot[iom_num].iom_slot, slot, slot);
		}
	}
}


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/

update_boot_path (flag)
uint	flag;
{
	char	b[PATH_LENGTH];

	switch (flag) {
		case 1:
			printf("Current SPM boot path:  <%s>\n", 
				novram->spm_boot_disk);
			printf("Enter new SPM default path; ESC to quit\n");
			if(gets(b,31) == -1) 
				return(0);
			strcpy(novram->spm_boot_disk,b);
			novram->prom_crc = nov_chk_sum (0);
			break;
		case 2:
			printf("Current CONFIG boot path:  <%s>\n", 
				novram->diag_boot);
			printf("Enter new config default path; ESC to quit\n");
			if(gets(b,31) == -1) 
				return(0);
			strcpy(novram->diag_boot,b);
			novram->prom_crc = nov_chk_sum (0);
			break;
		case 3:
			printf("Current SPM boot path:  <%s>\n", 
				novram->spm_boot_disk);
			printf("Enter new SPM default path; ESC to quit\n");
			if(gets(b,31) != -1) {
				strcpy(novram->spm_boot_disk,b);
				novram->prom_crc = nov_chk_sum (0);
			}

			printf("Current CONFIG boot path:  <%s>\n", 
				novram->diag_boot);
			printf("Enter new config default path; ESC to quit\n");
			if(gets(b,31) == -1) 
				return(0);
			strcpy(novram->diag_boot,b);
			novram->prom_crc = nov_chk_sum (0);
			break;
	}
	return(0);
}


extern  struct baud baudtab[];

extern  uchar	modem_baud_num;
extern  uchar	console_baud_num;
extern  uchar	printer_baud_num;
extern  uchar	ups_baud_num;

int
chg_baud(comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	struct  scc  *sccptr;	/*  Pointer to SCC on RWI  */
	char	port[16];	/*  String for port in printf */
	uchar	*baud_num_ptr;	/*  pointer to global baud number */
	uchar	c;

	switch (*comm_str) {
		case 'm':
			sccptr = SCC0B;
			baud_num_ptr = &modem_baud_num;
			strcpy (port, "Modem");
			break;
		case 'c':
			sccptr = SCC0A;
			baud_num_ptr = &console_baud_num;
			strcpy (port, "Console");
			break;
		case 'p':
			sccptr = SCC1A;
			baud_num_ptr = &printer_baud_num;
			strcpy (port, "Printer");
			break;
		case 'u':
			sccptr = SCC1B;
			baud_num_ptr = &ups_baud_num;
			strcpy (port, "UPS");
			break;
	}

	clscrn (24);
	printf ("\n\n%s Baud Rate Currently:    %d\n", port, 
			baudtab[*baud_num_ptr].bdrate);

	printf ("\nEnter the desired baud rate ID:\n");
	printf ("\n    0.  38400");
	printf ("\n    1.  19200");
	printf ("\n    2.  9600");
	printf ("\n    3.  4800");
	printf ("\n    4.  2400");
	printf ("\n    5.  1200");
	printf ("\n    6.  300");
	printf ("\n    7.  No Change\n\n");

	if (((c = getchar()) < '0') || (c > '6'))
		return(0);
	
	*baud_num_ptr = (c - '0');
	
	printf ("\nNew baud rate:  %d.\n", baudtab[*baud_num_ptr].bdrate);
	return(change_baud (sccptr, *baud_num_ptr));
}

