/*----------------------------------------------------------------------------
	memtools.c

	common to SPM_PROM and SPM_CONFIG
-----------------------------------------------------------------------------*/

#include "types.h"
#include "spm.h"
#include "disp.h"
#include "novram.h"
#include "globl.h"

/*
	update the following define if the size of tst_pat changes
*/
#define		TST_PAT_SIZ	32
#define		MAX_MEM_ERR	16

/* mem related global variables */

uint tst_pat[] = {
	0xaaaaaaaa, 0x55555555, 
	0xffffffff, 0x00000000,
	0x00000000, 0x00000000, 
	0x55555555, 0x55555555,
	0x66666666, 0x66666666, 
	0x78787878, 0x78787878,
	0x7F807F80, 0x7F807F80, 
	0x7FFF8000, 0x7FFF8000,
	0x7FFFFFFF, 0x80000000, 
	0x80000000, 0x00000000,
	0x00000000, 0x12488421, 
	0x84211248, 0x137FEC80,
	0x12488421, 0x84211248, 
	0x137FEC80, 0x08CEF731,
	0xFFFFFFFF, 0xEDB77BDE, 
	0x74568781, 0x17897921,
};

void	fill32 ();
int	check32 ();


/* note: the following is setup as map 1 referenced only!!! */
uint	*ERR_INFO_REG= 	(uint *)0x9fffffec;
uint	*ERR_ADDR_REG= 	(uint *)0x9ffffff4;
uint	*INT_INFO_REG= 	(uint *)0x9fffffc0;
uint	*STAT_REG =	(uint *)0x9ffffffc;
uchar	*CTL_REG =	(uchar *)0x9fffffe4;

#define MM_INIT 0x78	/* green LED on, EDAC and error interrupt bits */

extern uchar bdhere[];

/*------------------------------------------------------------------------------
	mm_init : intialize a memory module in a given slot.
		  Enable EDAC, parity checking, interrupt generation.
		  Turn off Red LED, and turn on Green LED.
------------------------------------------------------------------------------*/
mm_init(x,y)
char x,y;
{
	char i;
	uint int_info;

	int_info = 0x80800fa0 | ((*STATUSREG & STAT_SLOTMASK) >> 12);

	if(y) {
		cssmap(MAP01,x,(char)0x0f);	/* don't forget addr! */
		*INT_INFO_REG = int_info;
		*(char *)CTL_REG = MM_INIT;
	}
	else {
		for(i=0; i != Sbus_Num_Slot; i++) {
			if(bdhere[i]==MEMHERE) {
				cssmap(MAP01,i,(char)0x0f);	/* setup map */
				*INT_INFO_REG = int_info;	/* int info */
				*(char *)CTL_REG = MM_INIT;
			}
		}
	}
}

/*
 * mm_interrupt:
 *
 *	handle a memory module error interrupt.
 *
 *	read error information and error address which will
 *	re-enable interrupts.
 *
 *	To run on SPM:
 *
 *		The SPM should read the two registers, then log the error
 *		for itself, and then pass the error along to a processor
 *		module so that the kernel can log the error in its unix
 *		error logging facility.
 *
 */

mm_interrupt(mm_slot)
register uint	mm_slot;
{
	register uint		mm_err_info, mm_err_addr;

	/*
	 * This register must be read first, as the err info register
	 * resets the error capturing.
	 */

	/* read memory error address register */

	mm_err_addr = *ERR_ADDR_REG;

	/* read memory error information register */

	mm_err_info = *ERR_INFO_REG;

	show_mm_err(mm_slot, mm_err_addr, mm_err_info );
}


show_mm_err(mm_slot, mm_err_addr, mm_err_info )
register uint	mm_slot;
register uint	mm_err_addr;
register uint	mm_err_info;
{
	printf("\nMemory board error in slot 0x%x, location 0x%x, error reg 0x%x\n",
	  mm_slot, mm_err_addr & 0xfffffff, mm_err_info);

	/*
	 * If a multiple bit error is seen, then single
	 * bit errors should not be checked for.
	 */

	if (!(mm_err_info & 0x00800000)) {
		printf("Array 0 uncorrectable error\n");
		disp_mm_err((mm_err_info & 0x007f0000) >> 16);
		disp_mm_synr((mm_err_info & 0x7f000000) >> 24);
	}
	else if (!(mm_err_info & 0x00000080)) {
		printf("Array 1 uncorrectable error\n");
		disp_mm_err(mm_err_info & 0x0000007f);
		disp_mm_synr((mm_err_info & 0x00007f00) >> 8);
	}
	else if (!(mm_err_info & 0x80000000)) {
		printf("Array 0 single bit corrected error\n");
		disp_mm_err((mm_err_info & 0x007f0000) >> 16);
		disp_mm_synr((mm_err_info & 0x7f000000) >> 24);
	}
	else if (!(mm_err_info & 0x00008000)) {
		printf("Array 1 single bit corrected error\n");
		disp_mm_err(mm_err_info & 0x0000007f);
		disp_mm_synr((mm_err_info & 0x00007f00) >> 8);
	}
}


disp_mm_err(err_code)
register uint	err_code;
{
	printf("Source board is in CSS slot %x\n", err_code & 0x0f);

	if (err_code & 0x40) {
		printf("Partial write error\n");
		return;
	}

	/* otherwise it was a read error */
	if (!(err_code & 0x10))
		printf("Read 32 error\n");
	else if (!(err_code & 0x20))
		printf("Read 16 error\n");
	else
		printf("Partial read error\n");
}


/*
   Using syndrome information to get a byte from syndrome table as below.
   The byte from syndrome table decode as following:

   _________________________________________________________________
   | Bit 7 | Bit 6 | Bit 5 | Bit 4 | Bit 3 | Bit 2 | Bit 1 | Bit 0 |
   -----------------------------------------------------------------
      1       1   Uncorrectable multi-bit error
      1       0   Double bit error
      0       1   error in check bit (Bit0 - Bit5 represent bit location)
      0       0   error in data bit  (Bit0 - Bit5 represent bit location)
*/

#define UNCORRECT	0xc0
#define DOUBLE		0x80
#define CHKBIT		0x40
#define DATABIT		0x00

uchar syntab[128] =	{
	0xc0, 0x80, 0x80, 0xc0, 0x80, 0xc0, 0xc0, 0x80, /* 0 - 7 */
	0x80, 0xc0, 0x1f, 0x80, 0xc0, 0x80, 0x80, 0x1e, /* 8 - F */
	0x80, 0xc0, 0x1d, 0x80, 0x1c, 0x80, 0x80, 0x1b, /* 10 - 17 */
	0x1a, 0x80, 0x80, 0x19, 0x80, 0x18, 0xc0, 0x80, /* 18 - 1F */
	0x80, 0xc0, 0x07, 0x80, 0x06, 0x80, 0x80, 0x05, /* 20 - 27 */
	0x04, 0x80, 0x80, 0x03, 0x80, 0x02, 0xc0, 0x80, /* 28 - 2F */
	0x00, 0x80, 0x80, 0xc0, 0x80, 0x01, 0xc0, 0x80, /* 30 - 37 */
	0x80, 0xc0, 0xc0, 0x80, 0xc0, 0x80, 0x80, 0x46, /* 38 - 3F */

	0x80, 0xc0, 0xc0, 0x80, 0xc0, 0x80, 0x80, 0xc0, /* 40 - 47 */
	0xc0, 0x80, 0x80, 0x0f, 0x80, 0xc0, 0x0e, 0x80, /* 48 - 4F */
	0xc0, 0x80, 0x80, 0x0d, 0x80, 0x0c, 0x0b, 0x80, /* 50 - 57 */
	0x80, 0x0a, 0x09, 0x80, 0x08, 0x80, 0x80, 0x45, /* 58 - 5F */
	0xc0, 0x80, 0x80, 0x17, 0x80, 0x16, 0x15, 0x80, /* 60 - 67 */
	0x80, 0x14, 0x13, 0x80, 0x12, 0x80, 0x80, 0x44, /* 68 - 6F */
	0x80, 0x10, 0xc0, 0x80, 0x11, 0x80, 0x80, 0x43, /* 70 - 77 */
	0xc0, 0x80, 0x80, 0x42, 0x80, 0x41, 0x40, 0xff, /* 78 - 7F */
};


/* Decode memory syndrome information from syndrome table above */
disp_mm_synr(err_code)
register uint	err_code;
{
	uchar	syncode;
	
	syncode = syntab[err_code];
	if ((syncode & 0xc0) == UNCORRECT)
		printf("Uncorrectable multi-bit error\n");
	else
	if ((syncode & 0xc0) == DOUBLE)
		printf("Double-bit error\n");
	else
	if ((syncode & 0xc0) == CHKBIT)
		printf("Error in check bit = %d\n", syncode & 0x3f);
	else
	if ((syncode & 0xc0) == DATABIT)
		printf("Error in data bit = %d\n", syncode & 0x3f);
}

void
fill8 (ptr_start, ptr_limit, tseed, incr)

register  uchar *ptr_start;
register  uchar *ptr_limit;
register  uchar tseed;
register  uchar incr;
{
	if (incr) {
		do {
			*ptr_start = tseed;
			tseed += incr;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	else {
		do {
			*ptr_start = tseed;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
}

int
check8 (ptr_start, ptr_limit, tseed, incr)

register uchar *ptr_start;
register uchar *ptr_limit;
register uchar tseed;
register uchar incr;
{
register uchar rdata;
register uint err = 0;

	if (incr) {
		do {
			if ((rdata = *ptr_start) != tseed) {
				print_addr(ptr_start);
				printf ("EXP: %02X;  1ST: %02X;  2ND:  %02X\n", 
						tseed, rdata, *ptr_start);
				if (err++ == MAX_MEM_ERR)
					return (err);
				}
			tseed += incr;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	else {
		do {
			if ((rdata = *ptr_start) != tseed) {
				print_addr(ptr_start);
				printf ("EXP: %02X;  1ST: %02X;  2ND:  %02X\n", 
						tseed, rdata, *ptr_start);
				if (err++ == MAX_MEM_ERR)
					return (err);
				}
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	return (err);
}

void
fill16 (ptr_start, ptr_limit, tseed, incr)
register  ushort *ptr_start;
register  ushort *ptr_limit;
register  ushort tseed;
register  ushort incr;
{
	if (incr) {
		do {
			*ptr_start = tseed;
			tseed += incr;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	else {
		do {
			*ptr_start = tseed;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
}

int
check16 (ptr_start, ptr_limit, tseed, incr)
register ushort *ptr_start;
register ushort *ptr_limit;
register ushort tseed;
register ushort incr;
{
register ushort rdata;
register uint err = 0;

	if (incr) {
		do {
			if ((rdata = *ptr_start) != tseed) {
				print_addr(ptr_start);
				printf ("EXP: %04X;  1ST: %04X;  2ND:  %04X\n", 
						tseed, rdata, *ptr_start);
				if (err++ == MAX_MEM_ERR)
					return (err);
				}
			tseed += incr;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	else {
		do {
			if ((rdata = *ptr_start) != tseed) {
				print_addr(ptr_start);
				printf ("EXP: %04X;  1ST: %04X;  2ND:  %04X\n", 
						tseed, rdata, *ptr_start);
				if (err++ == MAX_MEM_ERR)
					return (err);
				}
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	return (err);
}

void
fill32 (ptr_start, ptr_limit, tseed, incr)
register  uint *ptr_start;
register  uint *ptr_limit;
register  uint tseed;
register  uint incr;
{
	if (incr) {
		do {
			*ptr_start = tseed;
			tseed += incr;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	else {
		do {
			*ptr_start = tseed;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
}

int
check32 (ptr_start, ptr_limit, tseed, incr)
register uint *ptr_start;
register uint *ptr_limit;
register uint tseed;
register uint incr;
{
register uint rdata;
register uint err = 0;

	if (incr) {
		do {
			if ((rdata = *ptr_start) != tseed) {
				print_addr(ptr_start);
				printf ("EXP: %08X;  1ST: %08X;  2ND:  %08X\n", 
						tseed, rdata, *ptr_start);
				if (err++ == MAX_MEM_ERR)
					return (err);
				}
			tseed += incr;
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	else {
		do {
			if ((rdata = *ptr_start) != tseed) {
				print_addr(ptr_start);
				printf ("EXP: %08X;  1ST: %08X;  2ND:  %08X\n", 
						tseed, rdata, *ptr_start);
				if (err++ == MAX_MEM_ERR)
					return (err);
				}
			ptr_start++;
			} while (ptr_start < ptr_limit);
	}
	return (err);
}

#define	MAP_SIZE	0x10000000

print_addr(addr)
uint	addr;
{
	uchar	slot;
	uint	map_offset;
	uchar	map_data;
	
	if (!(addr & 0x80000000)) {    /* Not CSS bus address */
		printf("SPM OFFSET:  0x%08X;  ", addr);
		return(0);
	}

	map_offset = (addr & 0x70000000);
	map_data = *(uchar *)(MAPBASE + map_offset);
	slot = ((map_data & 0xf0) >> 4);
	if (((bdhere[slot] & BDTYPEMASK) != IOMTYPE) || 
					((map_data & 0x0f) == 0x0f)) {
		printf("SLOT:  %X;  OFFSET:  0x%08X; ", slot, 
			((addr % MAP_SIZE) | ((map_data & 0x0F) * MAP_SIZE)));
	}
	else {
		printf("SLOT:  %X/%X;  OFFSET:  0x%08X; ", slot, 
				(map_data & 0x0F), (addr | 0xF0000000));
	}
	return(0);
}

