/******************************************************************************
                   (C)  COPYRIGHT ARIX CORPORATION 1989.
         ALL RIGHTS RESERVED.  NO PART OF THIS PROGRAM MAY BE PHOTOCOPIED,
          REPRODUCED, OR TRANSLATED TO ANOTHER LANGUAGE WITHOUT THE PRIOR
                    WRITTEN CONSENT OF ARIX CORPORATION.
*******************************************************************************
*
*  PGNAME:  iosba.c
*
*  PURPOSE:
*	The module will handle the miscellaneous functions required to use
*	the IOSBA. 
*
*  METHOD:	N/A
*
*  INPUT: 	N/A
*
*  OUTPUT:	N/A 
*
*  REVISION:	A
*
*  LAST MODIFIED:
*    11/08/89(grw)    Initial creation
*
******************************************************************************/

#include "types.h"
#include "iosba_addr.h"
#include "afifo.h"
#include "iom.h"
#include "iopm_comm.h"
#include "disp.h"
#include "ioa0.h"
#include "icb.h"
#include "icb_config.h"
#include "spm.h"

extern  char	got_berr;
extern  char	emulate;
extern  struct  iosb_config ios_conf[][MAX_CSS_SLOT];
extern  struct  icb_config io_conf[][MAXSLOTS];
extern	uint	iopm_present;

#define	IOPM_WRITES	8
#define	IOPM_SETTLE	0x50000

#define	FAIL		1
#define	PASS		0

#define	TST_PAT_SIZ	32
#define	TEST_INCR	0x04030201
#define	IOPM_START_ADDR	0xA8000000
#define	IOPM_MEM_SIZE	0x00020000

#define IOPM_CMD_TIMEOUT	0xFFFF
#define	IOPM_DISK_TIMEOUT	5000
#define	IOPM_TAPE_TIMEOUT	50000
#define	WASTE_LIMIT		0x1000

#define BOARD_ID_ADDR	0x8fffffff

#define IOSBA_WAIT	2
#define BIG_NUMBER      100

struct IOSBA_INIT{
	uint reg_ptr;		/* board address of the fifo register */
	uchar init_value;	/* initialization value		      */
};

/************ initialization pointer/value structure for the AFIFO *****/

struct IOSBA_INIT iosba_init[] = {
	/******** clear errors ************************************************/
	{ IOSBA_CTL_ERROR, 0},					/* 0xffffe800 */

	/******** downlink fifo initialization ********************************/
	{ IOSBA_DL_CMD_AP, FIFO_READ_WRITE },			/* 0xffffeb20 */
	{ IOSBA_DL_CMD_AR, IN_PAR_GEN_DIS + READYAB_SYNC},	/* 0xffffeb00 */
	{ IOSBA_DL_CMD_BP, FIFO_READ_WRITE },			/* 0xffffeb24 */
	{ IOSBA_DL_CMD_BR, IN_PAR_GEN_DIS + READYAB_SYNC},	/* 0xffffeb04 */
	{ IOSBA_DL_RSP_AP, FIFO_READ_WRITE },			/* 0xffffeb28 */
	{ IOSBA_DL_RSP_AR, IN_PAR_GEN_DIS + READYAB_SYNC},	/* 0xffffeb08 */
	{ IOSBA_DL_RSP_BP, FIFO_READ_WRITE },			/* 0xffffeb2c */
	{ IOSBA_DL_RSP_BR, IN_PAR_GEN_DIS + READYAB_SYNC},	/* 0xffffeb0c */

	/********* uplink fifo initialization *********************************/
	{ IOSBA_UL_CMD_AP, FIFO_READ_WRITE },			/* 0xffffeb30 */
	{ IOSBA_UL_CMD_AR, IN_PAR_GEN_DIS  },			/* 0xffffeb10 */
	{ IOSBA_UL_CMD_BP, FIFO_READ_WRITE },			/* 0xffffeb34 */
	{ IOSBA_UL_CMD_BR, IN_PAR_GEN_DIS  },			/* 0xffffeb14 */
	{ IOSBA_UL_RSP_AP, FIFO_READ_WRITE },			/* 0xffffeb38 */
	{ IOSBA_UL_RSP_AR, IN_PAR_GEN_DIS  },			/* 0xffffeb18 */
	{ IOSBA_UL_RSP_BP, FIFO_READ_WRITE },			/* 0xffffeb3c */
	{ IOSBA_UL_RSP_BR, IN_PAR_GEN_DIS  },			/* 0xffffeb1c */

	/***** enable transmit and receive ***********************************/
	{ IOSBA_TST_MODE,  TST_MODE_RXEN + TST_MODE_TXEN },	/* 0xffffe901 */

	/***** turn on the led's *********************************************/
	{ IOSBA_RST_CTRL,  RST_CTRL_FLT + RST_CTRL_RDY }
};

/****************************************************************************
	test_for_adapter(slot_number_of_IOM)

	this routine will read the identification register in the IOM memory
space to see if either an IOA, IOSBA, or NO_ADPTER is attached. The ID of the 
attached adapter is returned, 0x00 - IOA; 0x07 - IOSBA; 0xFF - NO_ADAPTER.
****************************************************************************/

test_for_adapter(slot)
int slot;
{
	uchar id_byte, *id_ptr;
	char emulate_save = emulate;
	char got_berr_save = got_berr;

	emulate = 1;
	got_berr = 0;

	id_ptr = (uchar *)IOSBA_ST_ID;
	id_ptr = (uchar *)CSSMAP7((uint)id_ptr);
	cssmap(MAP07,slot,IOSBA);
	id_byte = *id_ptr;
	if (got_berr) {
		emulate = emulate_save;
		got_berr = got_berr_save;
		return(NO_ADAPTER);
	}

	emulate = emulate_save;
	got_berr = got_berr_save;

	if ((id_byte & CARD_ID_VALUE) == CARD_ID_VALUE)
		return(TYPE_IOSBA);
	
	return(TYPE_IOA);
}


/****************************************************************************
	init_iosb_regs(slot_number_of_IOM)

	this routine will initialize the registers in the iosba using the values
and pointers from the array "iosba_init". It assumes that Map 7 is set to point 
to the IOSBA before entering this routine.
****************************************************************************/

init_iosb_regs(slot)
int slot;
{
	register uchar *fifo_ptr;
	register uchar *iom_reset_ptr;
	register int x;
	unsigned int *iom_ctl_ptr;

	cssmap(MAP07,slot,IOSBA); 	/* the iosba is always here */

	iom_ctl_ptr = (uint *)CSSMAP7(IOMCTL);
	*iom_ctl_ptr = 0x80000000;	/* ooh! magic!!!!	    */
	
	/* reset IOM path error pointer */
	iom_reset_ptr = (uchar *)CSSMAP7(PATHRST);

	for ( x = 0 ; x < (sizeof(iosba_init)/sizeof(iosba_init[1])); x++) {
	
		/****** reset IOM path errors  ***********/
		*iom_reset_ptr = 0;

		/****** get pointer to the register ******/
		fifo_ptr = (uchar *)iosba_init[x].reg_ptr;

		/****** tweek pointer to use Map7 ******/
		fifo_ptr = (uchar *)CSSMAP7((unsigned)fifo_ptr);

		/****** load initialization value ******/
		*fifo_ptr = iosba_init[x].init_value;

		/****** test to see if the register was loaded ******/
		if ( *fifo_ptr != iosba_init[x].init_value)
		 	printf("IOSBA init error, slot = %x, value = %x/@%x\n",
		 	  slot,iosba_init[x].init_value,iosba_init[x].reg_ptr);
	}
}


/****************************************************************************
	init_iosba(slot_number_of_IOM)

	this routine will handle iosba initialization.
	if flg = 0  do module enable and interface enable
	if flg = 1  do module enable
	if flg = 2  do interface enable

****************************************************************************/
init_iosba(slot)
int slot;
{
	register uint reg_ptr;
	register int iosba_slot_address;
	
	cssmap(MAP07,slot,IOSBA);

	reg_ptr = IOSBA_ST_CARD;
	reg_ptr = (unsigned  int )CSSMAP7(reg_ptr);
	iosba_slot_address = *(int *)reg_ptr;		/* get iosba slot     */
	iosba_slot_address = iosba_slot_address >> 8;	/* move to low byte   */
	iosba_slot_address &= 0x0f;		/* mask junk bits     */
	
	reg_ptr = (int)CSSMAP7(IOMCTL);
	*(int *)reg_ptr = 0x80000000;		/* enable 4 gbyte add */

}

reset_iosba_error(slot)
int slot;
{
	int iosba_ptr;
	int save_map;

	save_map = cssmap_save(MAP07);
	cssmap(MAP07,slot,IOSBA);

	iosba_ptr = (int)CSSMAP7(IOSBA_CTL_ERROR); /* 0xffffe800 */

	/* clear all bits in the error control registers */
	*(int *)iosba_ptr = 0;
	cssmap_restore(MAP07, save_map);
}

get_iosba_status(slot)
int slot;
{
int iosba_ptr;
int status_byte;
int save_map;

	save_map = cssmap_save(MAP07);
	cssmap(MAP07,slot,IOSBA);

	iosba_ptr = (int)CSSMAP7(IOSBA_CTL_ERROR); /* 0xffffe800 */

	status_byte = *(int *)iosba_ptr;
	status_byte &= 0xff000000; /* only want the top byte */
	cssmap_restore(MAP07, save_map);
	return (status_byte); /* 0 when ok */
}

int
ios_poll(flg, c_slot, iom_num)
uint	flg;
uint	c_slot;
uint	iom_num;
{
	uint	s_slot;
	uint	iosba_slot_address;
	uint	reg_ptr;

	io_conf[iom_num][0].iomslot = NOBOARD;

	cssmap(MAP07,c_slot,IOSBA); 	/* the iosba is always here */
	reg_ptr = IOSBA_ST_CARD;
	reg_ptr = (uint )CSSMAP7(reg_ptr);
	iosba_slot_address = *(int *)reg_ptr;		/* get iosba slot     */
	iosba_slot_address = iosba_slot_address >> 8;	/* move to low byte   */
	iosba_slot_address &= 0x0f;		/* mask junk bits     */

	got_berr = 0;			 /* clear bus error */
        /* loop through each slot in the second card cage */
	switch (flg) {
	case 0:
		for ( s_slot = 0; s_slot < MAX_CSS_SLOT; s_slot++){
			ios_conf[iom_num][s_slot].iomslot = c_slot;
	
			/* enable the interfaces on each sub-slot */
			/* except the one that has the iosba in it */
			if ( s_slot == iosba_slot_address )
				continue;

			if (io_control(MOD_DISABLE, c_slot, s_slot))
				continue;

			if (io_control(INTERFACE_DISABLE, c_slot, s_slot))
				continue;

			if (io_control(INTERFACE_ENABLE, c_slot, s_slot))
				continue;

			iopm_no_css_reset(c_slot, s_slot);
		}

		for ( s_slot = 0; s_slot < MAX_CSS_SLOT; s_slot++){
			if (io_control(INTERFACE_ENABLE, c_slot, s_slot))
				continue;

			/* the board exists, so get the board id */
			cssmap(MAP00,c_slot,s_slot);
			reg_ptr = (int)CSSMAP(BOARD_ID_ADDR);
			ios_conf[iom_num][s_slot].slot_id = *(uchar *)reg_ptr;

			if (ios_conf[iom_num][s_slot].slot_id == IOPHERE) {
				if (check_iopm_console(c_slot, s_slot))
					ios_conf[iom_num][s_slot].slot_id =
						IOPM_CONSOLE;
			}
		} 
		break;
	case 1:
		for ( s_slot = 0; s_slot < MAX_CSS_SLOT; s_slot++){
	
			switch (ios_conf[iom_num][s_slot].slot_id) {
			case IOPHERE:
				if (iopm_init(c_slot, s_slot)) {
					io_control(INTERFACE_DISABLE, 
						c_slot, s_slot);
					ios_conf[iom_num][s_slot].slot_id=0xFF;
					break;
				}	
					
				io_control(MOD_ENABLE, c_slot, s_slot);
				ios_conf[iom_num][s_slot].slot_id = IOPHERE;
				iopm_present++;
				break;
			case IOPM_CONSOLE:
				break;
			case VAMHERE:
				break;

			}
		} 
		break;
	}
}


int
io_control (control, c_slot, s_slot)
uchar	control;
uint	c_slot;
uint	s_slot;
{
	register uint	err = 0;
	uint	reg_ptr;
	char	emulate_save = emulate;

	emulate = 1;
	cssmap (MAP07, (uchar)c_slot, IOSBA);

	reg_ptr = (uint )IOSBA_WR_CONTROL | s_slot;
	reg_ptr = (uint )CSSMAP7(reg_ptr);

	*(char *)reg_ptr = control;
	wait_cnt(900);
	if (get_iosba_status(c_slot)){/* true if error */
		reset_iosba_error(c_slot); /* clear it */
		err = 1;
	}
			
	if (got_berr) {
		err = 2;
	}

	got_berr = 0;
	emulate = emulate_save;
	return(err);
}


