

/*
 * int_disp.c   --	Routines associated with the interrupt dispatcher.
 *
 */

#include "types.h"
#include "menu.h"
#include "routines.h"
#include "globl.h"
#include "spm.h"
#include "global.h"
#include "rtc.h"
#include "disp.h"
#include "dev.h"
#include "novram.h"
#include "misc.h"
#include "ioa0.h"
#include "icb_config.h"

extern uchar myslot;
extern uint  int_data;

				
init_dispatcher(comm_str,arg_cnt)
{
	register uint mapnum;
	register uchar slotnum,slotadd;
	register uchar *i;
	register uint *j;

	printf ("Initialize interrupt dispatcher.\n");
	if(clokpr(1))	 /* and if it returns bad, so do we. */
		return;
	init_css();
	rready();
	iready();
	mapnum = 0;
	slotnum = 0x0b;
	slotadd = 0x0f;
	i = (uchar *)((uint)MAPBASE | (mapnum << 28));
	*i = (slotnum << 4) | slotadd;
	*WRCNTL1 &= ~WR1_ID_RST;
	*WRCNTL1 |= WR1_ID_RST;
	if (arg_cnt)
		/* setup for cpu access to dispatcher rams */
		*WRCNTL1 &= ~(WR1_FINTREC | WR1_DREQ);
	else
		*WRCNTL1 &= ~WR1_DREQ; 
	/* reset interrupt dispatcher */
	/* source slot */
	*WRCNTL0 = (*WRCNTL0 & 0xff00ffff) | 0x004b0000;
	/* slot id */
	*WRCNTL0 = (*WRCNTL0 & 0xffff00ff) | 0x0000bf00;
	/* initialize the queue ram */
	for(j=QUEUE_RAM; j<QUEUE_RAM + QUEUE_LWORDS;j++) 
		*j = ((unsigned)j << 16) | ((unsigned)j & 0x0000ffff);
	/* initialize the pointer rams */
	for(j=POINTER_RAM; j<POINTER_RAM + POINTER_LWORDS;j++)
		*j= 0x7f7f0000;
	/* initialize the miscellaneous rams */
	for(j=MISC_RAM; j<MISC_RAM + MISC_LWORDS; j++)
		*j= 0x1f0003ff;
	if(arg_cnt)
		*WRCNTL1 = (*WRCNTL1 & 0x00ffffff) | WR1_FRC_RST_ACK | WR1_IRDY
			| WR1_RRDY | WR1_FMOD | WR1_DATYPE | WR1_DREQ | WR1_CSSRST; 
	else
		*WRCNTL1 = (*WRCNTL1 & 0x00ffffff) | WR1_FRC_RST_ACK | WR1_IRDY
			| WR1_RRDY | WR1_FMOD | WR1_DATYPE | WR1_FINTREC
			| WR1_DREQ | WR1_CSSRST; 
	*WRCNTL1 &= ~WR1_ID_RST; /* reset again.. sigh. */
	*WRCNTL1 |= WR1_ID_RST;
}



dram()
{
	dram_access(0);

	if( dram_access(0) ) {
		printf("\r\nUnable to access Dispatcher RAMs.");
		printf("\r\nDispatcher RAM tests skipped.\r\n");
		return;
	}

	/* cpu has access to dispatcher rams now */
	queue_ram_dripple();
	queue_ram_aripple();
	queue_ram_content();
	counter_ram_dripple();
	counter_ram_aripple();
	counter_ram_content();
	service_ram_aripple();
	service_ram_content();
	config_ram_dripple();
	config_ram_aripple();
	config_ram_content();
	pointer_ram_dripple();
	pointer_ram_aripple();
	pointer_ram_content();
		
	/* turn off the request now */
	dram_access(1);
}

queue_ram_dripple()
{
	register unsigned tstpat,readback;
	char *msg;

	msg = "Queue Ram Data Ripple";
	/* ripple a one bit in first location */
	for(tstpat=0x00000001; tstpat != 0; tstpat = tstpat << 1) {
		*QUEUE_RAM = tstpat;
		readback = *QUEUE_RAM;
		if(readback != tstpat) {
			print_err(msg,(unsigned)QUEUE_RAM,tstpat,readback);
			return;
		}
	}
	/* ripple a zero bit through first location */
	for(tstpat = 0xfffffffe; tstpat != (unsigned)0xffffffff; 
					tstpat = (tstpat << 1) | 0x00000001) {
		*QUEUE_RAM = tstpat;
		readback = *QUEUE_RAM;
		if(readback != tstpat) {
			print_err(msg,(unsigned)QUEUE_RAM,tstpat,readback);
			return;
		}
	}
}

queue_ram_aripple()
{
	register unsigned *addr,readback,offadd;
	char *msg;
	unsigned pat1=0x00000000;
	unsigned pat2=0x55555555;
	unsigned pat3=0xaaaaaaaa;

	msg = "Queue Ram Address Ripple";
	for(offadd=0x00000800; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)QUEUE_RAM + (uint)offadd);
		*addr = pat1;
	}
	for(offadd=0x00000800; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)QUEUE_RAM + (uint)offadd);
		readback = *addr;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

queue_ram_content()
{
	register unsigned readback,*addr;
	char *msg;
	register unsigned pat1=0x55555555;
	register unsigned pat2=0xaaaaaaaa;
	register unsigned pat3=0x00000000;

	msg = "Queue Ram Content";
	for(addr=QUEUE_RAM; addr < QUEUE_RAM + QUEUE_LWORDS ; addr++) {
		*addr = pat1;
		readback = *addr;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
	}
	for(addr=QUEUE_RAM; addr < QUEUE_RAM + QUEUE_LWORDS ; addr++) {
		readback = *addr;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
	}
	for(addr=QUEUE_RAM; addr < QUEUE_RAM + QUEUE_LWORDS ; addr++) {
		readback = *addr;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

counter_ram_dripple()
{
	register unsigned tstpat,readback;
	char *msg;

	msg = "Counter Ram Data Ripple";
	/* ripple a one bit in first location */
	for(tstpat=0x00000001; tstpat<0x00000400; tstpat = tstpat << 1) {
		*MISC_RAM = tstpat;
		readback = *MISC_RAM & CNTR_MASK;
		if(readback != tstpat) {
			print_err(msg,(unsigned)MISC_RAM,tstpat,readback);
			return;
		}
	}
	/* ripple a zero bit through first location */
	for(tstpat=0x000003fe; tstpat != 0x000003ff;
				tstpat=((tstpat<<1) | 0x00000001) & CNTR_MASK) {
		*MISC_RAM = tstpat;
		readback = *MISC_RAM & CNTR_MASK;
		if(readback != tstpat) {
			print_err(msg,(unsigned)MISC_RAM,tstpat,readback);
			return;
		}
	}
}

counter_ram_aripple()
{
	register unsigned *addr,readback,offadd;
	char *msg;
	unsigned pat1=0x00000000;
	unsigned pat2=0x00000155;
	unsigned pat3=0x000002aa;

	msg = "Counter Ram Address Ripple";
	for(offadd=0x00000010; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)MISC_RAM + (uint)offadd);
		*addr = pat1;
	}
	for(offadd=0x00000010; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)MISC_RAM + (uint)offadd);
		readback = *addr & CNTR_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & CNTR_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr & CNTR_MASK;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

counter_ram_content()
{
	register unsigned readback,*addr;
	char *msg;
	register unsigned pat1=0x00000155;
	register unsigned pat2=0x000002aa;
	register unsigned pat3=0x00000000;

	msg = "Counter Ram Content";
	for(addr=MISC_RAM; addr < MISC_RAM + CNTR_LWORDS ; addr++) {
		*addr = pat1;
		readback = *addr & CNTR_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
	}
	for(addr=MISC_RAM; addr < MISC_RAM + CNTR_LWORDS ; addr++) {
		readback = *addr & CNTR_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & CNTR_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
	}
	for(addr=MISC_RAM; addr < MISC_RAM + CNTR_LWORDS ; addr++) {
		readback = *addr & CNTR_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr & CNTR_MASK;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

service_ram_aripple()
{
	register unsigned *addr,readback,offadd;
	char *msg;
	unsigned pat1=0x00000000;
	unsigned pat2=0x00008000;

	msg = "Service Pending Ram Address Ripple";
	for(offadd=0x00000100; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)MISC_RAM + (uint)offadd);
		*addr = pat1;
	}
	for(offadd=0x00000100; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)MISC_RAM + (uint)offadd);
		readback = *addr & SERV_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & SERV_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
	}
}

service_ram_content()
{
	register unsigned readback,*addr;
	char *msg;
	register unsigned pat1=0x00008000;
	register unsigned pat2=0x00000000;

	msg = "Service Pending Ram Content";
	for(addr=MISC_RAM; addr < MISC_RAM + SERV_LWORDS ; addr++) {
		*addr = pat1;
		readback = *addr & SERV_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
	}
	for(addr=MISC_RAM; addr < MISC_RAM + SERV_LWORDS ; addr++) {
		readback = *addr & SERV_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & SERV_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
	}
}

config_ram_dripple()
{
	register unsigned tstpat,readback;
	char *msg;

	msg = "Configuration Ram Data Ripple";
	/* ripple a one bit in first location */
	for(tstpat=0x01000000; tstpat<0x10000000; tstpat = tstpat << 1) {
		*MISC_RAM = tstpat;
		readback = *MISC_RAM & CONFIG_MASK;
		if(readback != tstpat) {
			print_err(msg,(unsigned)MISC_RAM,tstpat,readback);
			return;
		}
	}
	/* ripple a zero bit through first location */
	for(tstpat=0x0e000000; tstpat != 0x0f000000;
			tstpat=((tstpat<<1) | 0x01000000) & CONFIG_MASK) {
		*MISC_RAM = tstpat;
		readback = *MISC_RAM & CONFIG_MASK;
		if(readback != tstpat) {
			print_err(msg,(unsigned)MISC_RAM,tstpat,readback);
			return;
		}
	}
}

config_ram_aripple()
{
	register unsigned *addr,readback,offadd;
	char *msg;
	unsigned pat1=0x00000000;
	unsigned pat2=0x05000000;
	unsigned pat3=0x0a000000;

	msg = "Configuration Ram Address Ripple";
	for(offadd=0x00000010; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)MISC_RAM + (uint)offadd);
		*addr = pat1;
	}
	for(offadd=0x00000010; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)MISC_RAM + (uint)offadd);
		readback = *addr & CONFIG_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & CONFIG_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr & CONFIG_MASK;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

pointer_ram_dripple()
{
	register unsigned tstpat,readback;
	char *msg;

	msg = "Pointer Ram Data Ripple";
	/* ripple a one bit in first location */
	for(tstpat=0x00010000; tstpat < (unsigned)0x80000000; tstpat <<= 1) {
		if(tstpat == 0x00800000)
			tstpat=tstpat << 1;
		*POINTER_RAM = tstpat;
		readback = *POINTER_RAM & POINTER_MASK;
		if(readback != tstpat) {
			print_err(msg,(unsigned)POINTER_RAM,tstpat,readback);
			return;
		}
	}
	/* ripple a zero bit through first location */
	for(tstpat=0xfffe0000; tstpat != (unsigned)0xffff0000;
					tstpat=(tstpat<<1) | 0x00010000) {
		*POINTER_RAM = tstpat;
		readback = *POINTER_RAM & POINTER_MASK;
		if(readback != (tstpat & POINTER_MASK)) {
			print_err(msg,(unsigned)POINTER_RAM,
					(tstpat & POINTER_MASK),readback);
			return;
		}
	}
}

pointer_ram_aripple()
{
	register unsigned *addr,readback,offadd;
	char *msg;
	unsigned pat1=0x00000000;
	unsigned pat2=0x55550000;
	unsigned pat3=0x2a2a0000;

	msg = "Pointer Ram Address Ripple";
	for(offadd=0x00000080; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)POINTER_RAM + (uint)offadd);
		*addr = pat1;
	}
	for(offadd=0x00000080; offadd >= 0x00000004 ; offadd = offadd >> 1) {
		addr = (unsigned *)((uint)POINTER_RAM + (uint)offadd);
		readback = *addr & POINTER_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & POINTER_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr & POINTER_MASK;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

pointer_ram_content()
{
	register unsigned readback,*addr;
	char *msg;
	register unsigned pat1=0x55550000;
	register unsigned pat2=0x2a2a0000;
	register unsigned pat3=0x00000000;

	msg = "Pointer Ram Content";
	for(addr=POINTER_RAM; addr < POINTER_RAM + POINTER_LWORDS ; addr++) {
		*addr = pat1;
		readback = *addr & POINTER_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
	}
	for(addr=POINTER_RAM; addr < POINTER_RAM + POINTER_LWORDS ; addr++) {
		readback = *addr & POINTER_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & POINTER_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
	}
	for(addr=POINTER_RAM; addr < POINTER_RAM + POINTER_LWORDS ; addr++) {
		readback = *addr & POINTER_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr & POINTER_MASK;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}

config_ram_content()
{
	register unsigned readback,*addr;
	char *msg;
	register unsigned pat1=0x05000000;
	register unsigned pat2=0x0a000000;
	register unsigned pat3=0x00000000;

	msg = "Configuration Ram Content";
	for(addr=MISC_RAM; addr < MISC_RAM + CONFIG_LWORDS ; addr++) {
		*addr = pat1;
		readback = *addr & CONFIG_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
	}
	for(addr=MISC_RAM; addr < MISC_RAM + CONFIG_LWORDS ; addr++) {
		readback = *addr & CONFIG_MASK;
		if(readback != pat1) {
			print_err(msg,addr,pat1,readback);
			return;
		}
		*addr = pat2;
		readback = *addr & CONFIG_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
	}
	for(addr=MISC_RAM; addr < MISC_RAM + CONFIG_LWORDS ; addr++) {
		readback = *addr & CONFIG_MASK;
		if(readback != pat2) {
			print_err(msg,addr,pat2,readback);
			return;
		}
		*addr = pat3;
		readback = *addr & CONFIG_MASK;
		if(readback != pat3) {
			print_err(msg,addr,pat3,readback);
			return;
		}
	}
}
	
print_err(msg,addr,expvalue,readvalue)
unsigned *msg,addr,expvalue,readvalue;
{
	printf("\r\n%s Test failed.",msg);
	printf("\r\nAddress = %8x, Expected: %8x, Read: %8x\r\n",addr,expvalue,
		readvalue);
}

dram_access(flag)
register unsigned char flag;
{
	register unsigned i;

	*WRCNTL1 |= WR1_ID_RST;
	if(flag)	
		/* turn the request off */
		*WRCNTL1 |= WR1_DREQ;
	else {	
		*WRCNTL1 &= ~WR1_DREQ;		/* cpu wants access to rams */
		for(i=0; i<0x20000; i++) {	/* wait for cpu signal */
			if(!(*STATUSREG & STAT_REG_IDRAM))
			break;
		}
		if(i < 0x20000)
			return(0);
		else
			return(1);
	}
	return(0);
}	


/* 
	sp_int_set(): To set service module to dispatch CSS interrupt
                      back to itself and cause a css command interrupt
		      to service module.
*/

sp_int_set()
{
	register unsigned 	*k, m;

	*WRCNTL1 &= ~WR1_ID_RST;
	*WRCNTL1 |= WR1_ID_RST;

	/* setup for cpu access to dispatcher rams */
	/* *WRCNTL1 &= ~WR1_FINTREC & ~WR1_DREQ; */
	*WRCNTL1 &= ~WR1_DREQ;

	/* initialize the queue ram */
	for(k=QUEUE_RAM; k<QUEUE_RAM + QUEUE_LWORDS;k++) 
		*k = ((unsigned)k << 16) | ((unsigned)k & 0x0000ffff);

	/* initialize the pointer rams */
	for(k=POINTER_RAM; k<POINTER_RAM + POINTER_LWORDS;k++)
		*k= 0x7f7f0000;

	/* initialize the miscellaneous rams */
	for(k=MISC_RAM; k<MISC_RAM + MISC_LWORDS; k++)
		*k= 0x0f0003ff;

	*WRCNTL1 = (*WRCNTL1 & 0x00fbffff) | WR1_FRC_RST_ACK | WR1_IRDY
			| WR1_FDEST | WR1_FINTREC | WR1_RRDY | WR1_FMOD 
			| WR1_DATYPE | WR1_DREQ; 
	*WRCNTL1 |= WR1_ID_RST;  /* turn on dispatcher again. */


	/* request the dispatcher rams */
	*WRCNTL1 &= ~WR1_DREQ;
	/* fill cfig ram with requested cpu number (service module number) */
	for(k=MISC_RAM; k<MISC_RAM + CONFIG_LWORDS; k++) {
		m = *k;
		/* check that if the count is foxed then bit 10 is zero,
		otherwise write a one */
		if((m & CNTR_MASK) == 0x3ff)
			m &= ~CNTR_ZERO_BIT;
		else
			m |= CNTR_ZERO_BIT;
		*k = (m & ~CONFIG_MASK) | (myslot << CONFIG_SH);
	}
	/* reset dispatcher hardware */
	*WRCNTL1 &= ~WR1_ID_RST; 
	*WRCNTL1 |= WR1_ID_RST; 
	/* force SPM to pretend to be the requested cpu */
	*WRCNTL0 = (*WRCNTL0 & ~WR0_PRETEND_MASK) | 
			((myslot << WR0_PRETEND_SH) & 0x00ffffff);
	*WRCNTL0 = (*WRCNTL0 & 0xff00ffff) | 
		((~myslot << 20) & 0x00ffffff) | ((myslot << 16) & 0x00ffffff);
	/* *WRCNTL1 &= ~WR1_FINTREC;	force SP receive interrupt */
	*WRCNTL1 |= WR1_DREQ;

}

sp_int_ack2(sp_slot)
unsigned sp_slot;
{
	register unsigned 	ack_offset, *ip;
	unsigned char		addr;

	ack_offset = (unsigned)0xffffff2c;
	addr = (unsigned char)CSSADD(ack_offset);
	cssmap(MAP03, (unsigned char)sp_slot, addr);
	ip  = (unsigned *)0xbfffff2c;

	/*
	4 bytes css read to service module itself to 
	acknowledge interrupt to itself
	*/
	int_data = *ip; 	/* css 4 bytes read */
}

