
/*
 * file: flp_test.c
 *		Test routines accessed with the i_flop_test menu are 
 *		in this file.  Essentially this will provide a test
 *		user interface to the floppy device driver.
 *
 *  890819  Redo floppy driver 
 *
 *	flp.c		--  User interface
 *	flp_exec.c	--  Floppy command start
 *	flp_result.c	--  Process floppy command
 *	flp_check.c	--  Analyze floppy response
 *	flp_test.c	--  Floppy tests -- from i_flop_test menu
 *	head/flp.h	--  General defines used by all
 *	head/flp_chk.h  --  Floppy result checking include
 */

#include "types.h"
#include "global.h"
#include "menu.h"
#include "flp.h"

#define BUFPOINT	0x0103e000

extern  uint    tst_pat[];
#define	TST_PAT_SIZ	32

uchar	*flp_buf;			/* buffer used read and write */

extern	uchar	flp_status;		/* Status returned by a sense drive
					   status command		*/

extern	uint flp_errors[];		/* error array */

/*
	Floppy commands in flp_exec.c
*/
extern	void 	mon();
extern	void	moff();
extern	int	seek();
extern	int	recal();
extern	int	rdsec();
extern	int	rdtrack();
extern	int	wrsec();
#ifdef	REMOVE
extern	int	format_track();
#endif

int	lseek();
int	lrds();
int	lwrs();
int	lwrrd();
int	rdall();
int	cmp_mem();
void	fil_buff();

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
lseek(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	seek1;
	uchar	seek2;

	seek1 = (uchar)atoi(comm_args[1]);
	seek2 = (uchar)atoi(comm_args[2]);

	if (seek1 > MAX_CYL_NUM) {
		printf ("\nInvalid cylinder (%d); Max cylinder (%d).\n",
			seek1, MAX_CYL_NUM);
		return(1);
	}

	if (seek2 > MAX_CYL_NUM) {
		printf ("\nInvalid cylinder (%d); Max cylinder (%d).\n",
			seek2, MAX_CYL_NUM);
		return(1);
	}

	mon ();

	if (recal ())
		return(1);

	printf ("Looping seek between cylinders %x (%d) and %x (%d).\n",
		seek1, seek1, seek2, seek2);
	printf ("Depress ESC to Quit\n");

	while (1) {

		if (seek(seek1) != seek1)
			return(1);

		if (seek(seek2) != seek2)
			return(1);

		if (ifesc())
			break;
	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
lrds(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	cyl;
	uchar	head;
	uchar	sect;
	uint	sect_cnt;
	uchar	*buf_save;

	cyl = (uchar)atoi(comm_args[1]);
	head = (uchar)atoi(comm_args[2]);
	sect= (uchar)atoi(comm_args[3]);
	sect_cnt = (uint)atoi(comm_args[4]);

	if (arg_cnt == 5)
		buf_save = (uchar *)atox(comm_args[5]);
	else
		buf_save = (uchar *)BUFPOINT;

	printf ("Looping read sector, cyl: %d, head: %d, sect: %d, cnt: %d\n",
		cyl, head, sect, sect_cnt);
	printf ("Depress ESC to Quit.\n");

	while (1) {
		flp_buf = buf_save;

		if (floppy_req (cyl, head, sect, sect_cnt, FLP_READ))
			return(1);

		if (ifesc ())
			break;
	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
lwrs(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	cyl;
	uchar	head;
	uchar	sect;
	uint	sect_cnt;
	uchar	*buf_save;

	cyl = (uchar)atoi(comm_args[1]);
	head = (uchar)atoi(comm_args[2]);
	sect= (uchar)atoi(comm_args[3]);
	sect_cnt = (uint)atoi(comm_args[4]);

	if (arg_cnt == 5)
		buf_save = (uchar *)atox(comm_args[5]);
	else
		buf_save = (uchar *)BUFPOINT;

	printf ("Looping write sector, cyl: %d, head: %d, sect: %d, cnt: %d\n",
		cyl, head, sect, sect_cnt);
	printf ("Depress ESC to Quit.\n");

	while (1) {
		flp_buf = buf_save;

		if (floppy_req (cyl, head, sect, sect_cnt, FLP_WRITE))
			return(1);

		if (ifesc ())
			break;
	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
lwrrd(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	cyl;
	uchar	head;
	uchar	sect;
	uint	sect_cnt;
	uint	buff_size;
	uint	data;
	uint	pass = 0;

	uchar	*buff_rd;
	uchar	*buff_wr;

	cyl = (uchar)atoi(comm_args[1]);
	head = (uchar)atoi(comm_args[2]);
	sect= (uchar)atoi(comm_args[3]);
	sect_cnt = (uint)atoi(comm_args[4]);
	buff_wr = (uchar *)atox(comm_args[5]);
	buff_rd = (uchar *)atox(comm_args[6]);
	buff_size = SECT_SZ * sect_cnt;

	printf ("Looping wr/rd/cmp sector, cyl: %d, hd: %d, sct: %d, cnt: %d\n",
		cyl, head, sect, sect_cnt);
	printf ("Depress ESC to Quit.\n");

	while (1) {

		data = tst_pat[pass++ % TST_PAT_SIZ];
		fil_buff (buff_wr, buff_size, data);

		flp_buf = buff_wr;
		if (floppy_req (cyl, head, sect, sect_cnt, FLP_WRITE))
			return(1);

		if (ifesc ())
			break;

		flp_buf = buff_rd;
		if (floppy_req (cyl, head, sect, sect_cnt, FLP_READ))
			return(1);

		if (cmp_mem (buff_wr, buff_rd, (SECT_SZ * sect_cnt))) {
			CPERR++;
			return(1);
		}

		if (ifesc ())
			break;

	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
rdall(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	cyl;
	uchar	head;
	uchar	sect;

	uint	data;
	uint	mask;
	uint	pass = 0;

	uchar	quit = 0;

	uchar	*buff_rd;
	uchar	*buff_wr;

	buff_wr = (uchar *)atox(comm_args[1]);
	buff_rd = (uchar *)atox(comm_args[2]);

	printf ("Looping write/read single sector whole disk.\n");
	printf ("Depress ESC to Quit.\n");

	while (!quit) {

		for (cyl = 0; cyl != NUMBER_CYLINDERS && !quit; cyl++) {
			for (head = 0; head != NUMBER_HEADS && !quit; head++) {
				mask = ((cyl << 16) | (head << 8));
				printf ("\rcyl:  %2d,  head:  %d  ", cyl, head);
				for (sect = 1; sect <= EOT && !quit; sect++) {

					data = tst_pat[pass % TST_PAT_SIZ];
					data ^= (mask | sect);
					fil_buff (buff_wr, SECT_SZ, data);

					flp_buf = buff_wr;
					if (floppy_req (cyl, head, sect, 1, FLP_WRITE))
						return(1);

					if (ifesc ()) 
						quit = 1;

					flp_buf = buff_rd;
					if (floppy_req (cyl, head, sect, 1, FLP_READ))
						return(1);

					if (cmp_mem(buff_wr, buff_rd, SECT_SZ)){
						CPERR++;
						return(1);
					}

					if (ifesc ())
						quit = 1;
				}
			}
		}
		printf ("\r Pass %d completed.\n", ++pass);
	}
	return(0);
}

int
cmp_mem (buff_wr, buff_rd, size)
register uchar	*buff_wr;
register uchar	*buff_rd;
register uint	size;
{
	register uchar	read;
	register uchar	write;
	register uint	cnt;
	register uchar	errors = 0;
	
	for (cnt = 0; cnt != size; cnt++) {
		
		if ((write = *buff_wr) != (read = *buff_rd)) {
			printf ("Mismatch at byte %d of %d.\n", cnt, size);
			printf ("Expected:  %02x  Received:  %02x.\n",
				write, read);
			if (errors++ == MAX_ERRORS)
				return(errors);
		}
		else {
			buff_wr++;
			buff_rd++;
		}
	}
	return(errors);
}

void
fil_buff (buf_ptr, buf_size, pattern)
uint	*buf_ptr;
uint	buf_size;
uint	pattern;
{

	uint	cnt;

	for (cnt = 0; cnt != buf_size; cnt += 4)
		*buf_ptr++ = pattern;
	
	return;
}
