
/*
 * file: flp_check.c
 *		This file contains the routines which analyze the 
 *		result structure, printing error messages as 
 *		detected.
 *
 *  890819  Redo floppy driver 
 *
 *	flp.c		--  User interface
 *	flp_exec.c	--  Floppy command start
 *	flp_result.c	--  Process floppy command
 *	flp_check.c	--  Analyze floppy response
 *	flp_test.c	--  Floppy tests -- from i_flop_test menu
 *	head/flp.h	--  General defines used by all
 *	head/flp_chk.h  --  Floppy result checking include
 */

#include "types.h"
#include "flp.h"			/* definition of floppy command block */
#include "flp_check.h"			/* definition of floppy command block */

extern	struct	flp_cmd	 flp_cmd;
extern	union	flp_res	 flp_res;
extern	uint flp_errors[];		/* error array */
extern	uchar	*flp_buf;		/* address used by read and write
					   commands.			*/
extern	uchar	flp_status;		/* Status returned by a sense drive
					   status command		*/

int	check_result();
int	check_st0();
int	check_st1();
int	check_st2();
void	check_all();
int	check_flp_result();


/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
check_result()
{
	int	error = 0;

	switch (flp_cmd.cmd1) {
		case  FD_RDSEC:
			error += check_flp_result("RDSEC");
			check_all("RDSEC");
			break;
		case  FD_RDTRACK:
			error += check_flp_result("RDTRACK");
			check_all("RDTRACK");
			break;
		case  FD_WRSEC:
			error += check_flp_result("WRSEC");
			check_all("WRSEC");
			break;
		case  FD_RDID:
			error += check_flp_result("RDID");
			break;
		case  FD_FMT:
			error += check_flp_result("FMT");
			break;
		case  FD_SEEK:
			error += check_st0(flp_res.seek.st0, "SEEK");

			if (!(flp_res.seek.st0 & SR0SE)) {
				printf ("SEEK:  Seek end not set in ST0.\n");
				error += SEEK_NOT_DONE;
			}

			if (flp_res.seek.cyl != flp_cmd.args.seek.new_cyl) {
				printf ("SEEK:  Cylinder not as expected.\n");
				printf ("SEEK:  Expect:  %d;  Return:  %d.\n",
				flp_cmd.args.seek.new_cyl, flp_res.seek.cyl);
			}
			break;
		case  FD_RECAL:
			error += check_st0(flp_res.recal.st0, "RECAL");

			if (flp_res.recal.st0 & SR0EC) {
				printf ("RECAL:  Equipment check detected.\n");
				error += EQIP_CHK;
			}

			if (flp_res.recal.cyl != (uchar)0) {
				printf ("RECAL:  Unexpected cylinder.\n");
				printf ("RECAL:  Expect:  0;  Return:  %d.\n",
				flp_res.recal.cyl);
			}
			break;
		case  FD_SDS:
			flp_status = flp_res.sds.st3;
			return(0);
		case  FD_SPEC:
			return(0);
		case  FD_SIS:
			return(0);
	}
	return(error);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
check_st0(result_byte, cmd_str)
uchar	result_byte;
char	*cmd_str;
{
	int	error = 0;
	uchar	unit;

	if ((result_byte & SR0MSK) == SR0DNC) {
		printf ("%s:  Command did not complete.\n", cmd_str);
		error += CMD_DNC;
	}

	if ((result_byte & SR0MSK) == SR0INVL) {
		printf ("%s:  Invalid command did not start.\n", cmd_str);
		error += CMD_INVL;
	}

	if ((unit = (result_byte & SR0USX)) != UNIT0) {
		printf ("%s:  Improper unit in st0:  %d.\n", cmd_str, unit);
		error += WRONG_UNIT;
	}

	return(error);
}	

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
check_st1(result_byte, cmd_str)
uchar	result_byte;
char	*cmd_str;
{
	int	error = 0;

	if (result_byte & SR1DE) {
		printf ("%s:  Data error detected.\n", cmd_str);
		error += DATA_ERROR;
	}

	if (result_byte & SR1OR) {
		printf ("%s:  Overrun detected.\n", cmd_str);
		error += OVERRUN;
	}

	if (result_byte & SR1ND) {
		printf ("%s:  No data error detected.\n", cmd_str);
		error += NO_DATA;
	}

	if (result_byte & SR1NW) {
		printf ("%s:  Not writable media.\n", cmd_str);
		error += NOT_WRITE;
	}

	if (result_byte & SR1MA) {
		printf ("%s:  Missing address mark.\n", cmd_str);
		error += MISS_ADDR;
	}

	return(error);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
check_st2(result_byte, cmd_str)
uchar	result_byte;
char	*cmd_str;
{
	int	error = 0;
	
	if (result_byte & SR2CM) {
		printf ("%s:  Control mark detected.\n", cmd_str);
		error += CNTL_MARK;
	}

	if (result_byte & SR2DD) {
		printf ("%s:  CRC data field error.\n", cmd_str);
		error += CRC_ERR;
	}

	if (result_byte & SR2WC) {
		printf ("%s:  Wrong cylinder reported.\n", cmd_str);
		error += WRONG_CYL;
	}

	if (result_byte & SR2BC) {
		printf ("%s:  Bad cylinder reported.\n", cmd_str);
		error += BAD_CYL;
	}

	if (result_byte & SR2MD) {
		printf ("%s:  Missing address mark in data field.\n", cmd_str);
		error += MISS_ADDR;
	}

	return(error);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
check_flp_result(cmd_str)
char	*cmd_str;
{
	int	error = 0;

	error += check_st0(flp_res.other.st0, cmd_str);
	error += check_st1(flp_res.other.st1, cmd_str);
	error += check_st2(flp_res.other.st2, cmd_str);

	return(error);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
check_all(cmd_str)
char	*cmd_str;
{
	
	uchar	exp_cyl;
	uchar	exp_head;
	uchar	exp_sect;

	exp_head = flp_cmd.args.rd_wr.head;

	if (flp_cmd.cmd1 == FD_RDTRACK) {
		exp_cyl = flp_cmd.args.rd_wr.cyl + 1;
		exp_sect = 1;
	}
	else {
		if (flp_cmd.args.rd_wr.sect == EOT) {
			exp_cyl = flp_cmd.args.rd_wr.cyl + 1;
			exp_sect = 1;
		}
		else {
			exp_cyl = flp_cmd.args.rd_wr.cyl;
			exp_sect = flp_cmd.args.rd_wr.sect + 1;
		}
	}
	
	if (flp_res.other.cyl != exp_cyl) {
		printf ("%s:  Result cyl mismatch request cyl.\n", cmd_str);
		printf ("%s:  Result:  %d;  Command:  %d.\n", cmd_str,
		flp_res.other.cyl, flp_cmd.args.rd_wr.cyl);
	}

	if (flp_res.other.head != flp_cmd.args.rd_wr.head) {
		printf ("%s:  Result head mismatch request head.\n", cmd_str);
		printf ("%s:  Result:  %d;  Command:  %d.\n", cmd_str,
		flp_res.other.head, flp_cmd.args.rd_wr.head);
	}

	if (flp_res.other.sect != exp_sect) {
		printf ("%s:  Result sect mismatch request sect.\n", cmd_str);
		printf ("%s:  Result:  %d;  Command:  %d.\n", cmd_str,
		flp_res.other.sect, flp_cmd.args.rd_wr.sect);
	}

	if (flp_res.other.nbyte != flp_cmd.args.rd_wr.nbyte) {
		printf ("%s:  Result nbyte mismatch request nbyte.\n", cmd_str);
		printf ("%s:  Result:  %d;  Command:  %d.\n", cmd_str,
		flp_res.other.nbyte, flp_cmd.args.rd_wr.nbyte);
	}

}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
check_int_result()
{
	
	printf ("check_int_result: got here.\n");
	return;
}

