/*
 *  This module contains the interrupt service routines for the Service
 *  Processor board.
 */

#include "globl.h"
#include "misc.h"
#include "spm.h"
#include "ascc.h"
#include "rwicio.h"
#include "bus.h"
#include "iom.h"
#include "types.h"
#include "ring.h"
#include "disp.h"

struct ring ringer[MAXRING];	/* allow lots of them.. */
char int_head;			/* global pointer into ringer */
char int_tail;			/* global pointer into ringer */
char int_flag;			/* global int counter. */

extern char got_berr;		/* set if we got bus error */
extern char emulate;		/* set if we got bus error */
char berr_cnt;		/* set if we got bus error */
extern uchar	dtcintp;
extern char ignore_it;	/* If set, ignore CSS bus errors. */
extern char booting;
extern  unsigned 	int_data;
extern uchar bdhere[];	/* slots for CSS board's... */
extern uchar myslot;

extern uint	long_throw;

#define ctl_S 0x13
#define ctl_Q 0x11
#define ctl_Z 0x1a

int stack_ptr;
union location *sim_addr;
unsigned	sim_data;
extern unsigned char HndlBerr,MemSimFlag;

buserr(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6,internal10,internal14,internal18)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned internal10,internal14,internal18;
unsigned short *PC;
{
	register copy_from, copy_to;
	register unsigned char value;
	register unsigned fault_addr,mapnum;
	register unsigned char slotnum,slotadd,size;
	register unsigned char *i;
	register int	 	errdata;
	unsigned short 		ssw;

	long_throw = *CSSERROR;
	*WRCNTL0 &= ~WR0_FRC_INH7; /* disable it.. eh? */
	switch(HndlBerr)
	{
	case TRUE:
		if(ifesc())
			value=IntToMon;
		else	{
			value= 0;	/* repeat bus error cycle */
		}
		break;
	case FALSE:
		fault_addr = ( (internal14 << 16) | (internal18 >> 16) );
		printf("\nData cycle fault address = %x\n", fault_addr);
		if ( (fault_addr >> 31) == 1 )
		{
			printf("CSS bus time-out error\n");
			mapnum = ( (fault_addr & 0x70000000) >> 28 );
			printf("Map#: %x",mapnum);
			i =(unsigned char *)((unsigned)MAPBASE | (mapnum <<28));
			slotadd = (*i & 0x0f);
			slotnum = (*i >> 4) & 0x0f;
			printf("  Map slot:%x  Map addr:%x\n",slotnum,slotadd);
		}
		else
		{
			printf("\nBus time-out error\n");
		}
		ssw = internal10 >> 16 ;
		if (ssw & 0x0040)
			printf("\nREAD");
		else
			printf("\nWRITE");
		size = ( (ssw & 0x0030) >> 4 );
		switch(size)
		{
			case 0:
				printf(" 4 byte");
				break;
			case 1:
				printf(" 1 byte");
				break;
			case 2:
				printf(" 2 byte");
				break;
			case 3:
				printf(" 3 byte");
				break;
		}
		report(internal6,PC,SR,&d0);
		dobufprint();	/* go do a print routine. */
		value=IntToMon;
		break;
	}
	/* check what type of stack frame */
	if ((int)(internal6 & 0xf0000000) == (int)0xa0000000) 
	{
		/* short stack frame so change stack frame to 4 word */
		asm("mov.l	%sp,stack_ptr");
		internal6 = internal6 & 0x0fffffff;
		copy_from = (int)&internal6;
		copy_to = (int)copy_from + 24;
		while (copy_from >= stack_ptr)  {
			*(short *)copy_to = *(short *)copy_from;
			copy_to -= 2;
			copy_from -= 2;
		}
		asm("mov.l	%sp,%d0");
		asm("add.l	&24,%d0");
		asm("mov.l	%d0,%sp");
		asm("mov.l	%a6,%d0");
		asm("add.l	&24,%d0");
		asm("mov.l	%d0,%a6");
	}
	else if ((int)(internal6 & 0xf0000000) == (int)0xb0000000) 
	{              /* long stack frame */
		asm("mov.l	%sp,stack_ptr");
					/* change stack frame to 4 word */
		internal6 = internal6 & 0x0fffffff;
		copy_from = (int)&internal6;
		copy_to = (int)copy_from + 84;
		while (copy_from >= stack_ptr)  {
			*(short *)copy_to = *(short *)copy_from;
			copy_to -= 2;
			copy_from -= 2;
		}
		asm("mov.l	%sp,%d0");
		asm("add.l	&84,%d0");
		asm("mov.l	%d0,%sp");
		asm("mov.l	%a6,%d0");
		asm("add.l	&84,%d0");
		asm("mov.l	%d0,%a6");
	}
	*WRCNTL0 |= WR0_FRC_INH7; /* enable it.. eh? */
	return(value);
}

badinst(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned short *PC;
{
	rprintf("\nIllegal instruction");
	report(internal6,PC,SR,&d0);
	dobufprint();	/* go do a print routine. */
	return(IntToMon);
}

badexcept(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned short *PC;
{
	rprintf("\nBad exception");
	report(internal6,PC,SR,&d0);
	dobufprint();	/* go do a print routine. */
	return(IntToMon);
}

zerodivide(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned short *PC;
{
	rprintf("\nZero divide");
	report(internal6,PC,SR,&d0);
	dobufprint();	/* go do a print routine. */
	return(IntToMon);
}

priverror(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned short *PC;
{
	rprintf("\nPrivilege violation");
	report(internal6,PC,SR,&d0);
	dobufprint();	/* go do a print routine. */
	return(IntToMon);
}

addresserror(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned short *PC;
{
	rprintf("\nAddress error");
	report(internal6,PC,SR,&d0);
	dobufprint();	/* go do a print routine. */
	return(IntToMon);
}

formaterror(d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,PC,internal6)
unsigned d0,d1,d2,d3,d4,d5,d6,d7,a0,a1,a2,a3,a4,a5,a6,a7,SR,internal6;
unsigned short *PC;
{
	rprintf("\nFormat error");
	report(internal6,PC,SR,&d0);
	dobufprint();	/* go do a print routine. */
	return(IntToMon);
}


/*
 * handle autovectors
 */

autoint(d0,d1,a0,a1,level)
register unsigned level;
{
	register unsigned err,disp_err;
	register unsigned slot;
	register char bd_id;
	switch(level)
	{
	case 1:	/* dispatcher error */
		disp_err = *DISPERROR;
		err = (disp_err >> DISP_ERR_SH) & 0x07;
		switch(err)
		{
		case 1:	/* insane ack error */
			*WRCNTL1 &= ~WR1_DREQ;
			*WRCNTL1 &= ~WR1_FRC_RST_ACK;
			*WRCNTL1 |= WR1_FRC_RST_ACK;
			*WRCNTL1 |= WR1_DREQ;
			if(booting)
			{
				iready(); /* inc ready, this is expected.*/
				break; /* and exit this routine. */
			}
			printf("\nACK (with no purpose)");
			break;
		case 2: /* request with no int. pending */
			printf("\nHardware failure (request with none pending)");
			*WRCNTL1 &= ~WR1_DREQ;
			break;
		case 3: /* undefined error is now receive error (case 7). */
		case 5:	/* timeout on ack response */
			printf("\nACK (bus error on ACK response)");
			*WRCNTL1 &= ~WR1_FRC_RST_ACK;
			*WRCNTL1 |= WR1_FRC_RST_ACK;
			*WRCNTL1 &= ~WR1_DREQ;
			iready();
			break;
		case 6:	/* timeout on request */
			printf("\nRequest error");
			*WRCNTL1 &= ~WR1_DREQ;
			break;
		case 7:	/* receive error */
			printf("\nReceive error (queue full)");
			*WRCNTL1 &= ~WR1_FRC_RST_RCV;			/* */
			*WRCNTL1 |= WR1_FRC_RST_RCV; /* Clear received error */
			*WRCNTL1 &= ~WR1_DREQ;       /* reset the dispatcher, */
			*WRCNTL1 |= WR1_DREQ;        /* and start it up again */
			iready();
			break;
		default:
			printf("\nUndefined dispatcher error");
			*WRCNTL1 &= ~WR1_DREQ;
			break;
		}
		printf("\nDispatcher error register = %8x\n",disp_err);
		break;

	case 2:		/* CSS command register write */
		int_flag++; /* show one more command here.*/
		*WRCNTL0 &= ~WR0_FRC_INH7;	         /* disable it */
		ringer[int_head].intcmd = *CSSCMD;	 /* save command data */
		ringer[int_head].intstatus = *STATUSREG; /* save command data */
		ringer[int_head].interr = *CSSERROR;	 /* save error data */
		int_head++;                            /* leave room for next */
		if(int_head == MAXRING)             /* if it's at the limit */
			int_head = 0;               /* reset to start limit. */
		*WRCNTL0 |= WR0_FRC_INH7;           /* enable it */
		iready();                           /* all done */
		break;

	case 3:
		sp_int_ack2(myslot); /* acknowledge interrupt */
                                     /* Now check interrupt data */
		slot = (int_data & 0x0f00) >> 8;        /* get slot # */
		bd_id = (unsigned char)bdhere[slot];	/* get board id */
		switch(bd_id)	{
			case IOMHERE:
			case IOM2HERE:
				if (int_data & IOD_INT)	/* IO device interrupt*/
					dtcintp++;	/* increment int cnt */
				else	
					iom_err(slot); /* check IOM mod error */
				break;
			case MEMHERE:
				cssmap(MAP01,slot,(char)0x0f); /* addr! */
				mm_interrupt(slot); /* check Memory error */
				break;
			default:
				printf("Unknown board ID = %x in slot %x interrupt\n",bd_id,slot);
				break;
		}
	case 4:
	case 5:	/* peripheral int */
		break;
	case 6:	/* floppy */
		printf("\nAuto vector level %x",level);
		break;
	case 7:	/* CSS bus error */
		if(ignore_it)	/* if we are to ignore this */
		{
			*WRCNTL0 &= ~WR0_FRC_INH7;   /* disable it */
			ringer[int_head].intcmd = *CSSCMD; /* command data */
			ringer[int_head].intstatus = *STATUSREG; /* status */
			ringer[int_head].interr = *CSSERROR; /* error data */
			int_head++;	            /* leave room for next. */
			if(int_head == MAXRING)	    /* if it's at the limit */
				int_head = 0;       /* reset to start limit */
			*WRCNTL0 |= WR0_FRC_INH7;   /* enable it */
			iready();                   /* all done */
			break;                      /* exit the routine */
		}
		handle_berr();
		break;
	default:
		printf("\nAuto vector level out of range (%8d)",level);
		break;
	}
}


char *amsg[] = { /* response message table. */
	"ERROR",
	"ILLEGAL",
	"ERROR DATA 0",
	"ERROR DATA 1",
	"DATA",
	"ILLEGAL",
	"ILLEGAL",
	"ILLEGAL",
};

char *bmsg[] = {	/* read or write message table. */
	"4 byte",
	"1 byte",
	"2 bytes",
	"3 bytes",
	"8 bytes",
	"16 bytes",
	"32 bytes",
	"ILLEGAL",
};

char *cmsg[] = { /* control write message table. */
	"ILLEGAL",
	"module DISABLED",
	"module interface DISABLED",
	"ILLEGAL",
	"ILLEGAL",
	"module ENABLED",
	"module interface ENABLED",
	"ILLEGAL",
};

char *dmsg[] = { /* bus type message table */
	"Response: ",
	"",
	"",
	"Read: ",
	"",
	"Write: ",
	"Control write type:",
	"ILLEGAL BIT VALUE",
};

iom_berr()
{

	/* printf("\nCSS bus error\n"); */
	long_throw = *STATUSREG; /* read status reg first */
	long_throw = *CSSERROR;
}

handle_berr()
{
	register unsigned fault,status,i;

	rprintf("\nCSS bus error\n");
	*WRCNTL0 &= ~WR0_FRC_INH7; /* disable it */
	status = *STATUSREG; /* read status reg first */
	fault = *CSSERROR;
	if(++berr_cnt < 4) /* If number of errors is greater than 3, skip. */
		*WRCNTL0 |= WR0_FRC_INH7; /* enable it */
	rprintf("Status register = %8x\n",status);
	rprintf("CSS error register = %8x\n",fault);
	if(status & 0x00000001) /* bus nack signal. */
		rprintf("*Bus NACK\n");
	if(!(status & 0x00000002)) /* Bus ack signal. */
		rprintf("*No bus ACK\n");
	if(!(status & 0x00000004)) /* Grant error */
		rprintf("Arbiter grant error\n");
	if(status & 0x00000008) /* detected a CSS bus protocol violation */
		rprintf("SPM detected a protocol violation.\n");
	else
		rprintf("No protocol violation\n");
	if(!(status & 0x00000010)) /* Burst present during error */
		rprintf("Burst is active\n");

	if((fault >> DERR) & 0x01)
		rprintf("Destination error\n");
	if((fault >> SERR) & 0x01)
		rprintf("Source error\n");
	rprintf("Destination = %2x\n",(fault >>DEST_SH) & MASK4);
	rprintf("Source = %2x\n",(fault >>SRC_SH) & MASK4);
	i = ((fault >> 11) & 0x07);
		rprintf("Bus type is %s ", dmsg[i]);	/* show what type */
	if(i == 0)
		rprintf("%s\n", amsg[((fault >> 8) & 0x07)]); /* response type*/
	else if(i == 3 || i == 5)
		rprintf("%s\n", bmsg[((fault >> 8) & 0x07)]); /* read or write type. */
	else if(i == 6)
		rprintf("%s\n", cmsg[((fault >> 8) & 0x07)]); /* control write type. */
	rprintf("Bus type parity = %1x\n",(fault >> BPAR_SH) & MASK1);
	dobufprint();	/* go do a print routine. */
	berr_cnt--;	/* remove this one from stack since we printed it. */
	*WRCNTL0 |= WR0_FRC_INH7; /* enable it */
}

LCIO_err()
{
	register struct cio *cioptr = LOCCIO;

	if(cioptr -> ctcs[0].reg & ICR_IP)   /* timer 1 int?*/
		cioptr -> ctcs[0].reg = ICW_CPUS | CT_GCB;
	if(cioptr -> ctcs[1].reg & ICR_IP)   /* timer 2 int?*/
		cioptr -> ctcs[1].reg = ICW_CPUS | CT_GCB;
	if(cioptr -> ctcs[2].reg & ICR_IP)   /* timer 3 int?*/
		cioptr -> ctcs[2].reg = ICW_CPUS | CT_GCB;
}

LCIOt1_int()
{
	register struct cio *cioptr = LOCCIO;

	if(cioptr -> ctcs[0].reg & ICR_IP) {  /* timer 1 int?*/
		cioptr -> ctcs[0].reg = ICW_CPUS | CT_GCB;
		if(++IntCounter == 50) { /*1 sec */
			IntCounter = 0;
			switch((*WRCNTL2 >> WR2_LED01_SH) & 0x03) {
			case 0x00:
				*WRCNTL2 = (*WRCNTL2 | (WR2_LED0 | WR2_LED1));
				break;
			case 0x01:
				*WRCNTL2 = (*WRCNTL2 | (WR2_LED0 | WR2_LED1))
				    & ~WR2_LED1;
				break;
			case 0x02:
				*WRCNTL2 = (*WRCNTL2 | (WR2_LED0 | WR2_LED1))
				    & (~WR2_LED1 & ~WR2_LED0);
				break;
			case 0x03:
				*WRCNTL2 = (*WRCNTL2 & ~WR2_LED0);
				break;
			}
			++Seconds;
		}
		return;
	}
	printf("Spurious local CIO level 4 interrupt.\n");
}

report(internal6,PC,SR,r)
register unsigned *r;
{
	rprintf("\nVector offset = %08x at PC = %08x\nformat = %08x status register = %08x",
		 (internal6>>16) & 0x0fff, PC,(internal6>>28) & 0x0f, (SR & 0xffff));
	rprintf("\nData: %08x %08x %08x %08x ", r[0],r[1],r[2], r[3]);
	rprintf("%08x %08x %08x %08x", r[4],r[5],r[6], r[7]);
	rprintf("\nAddr: %08x %08x %08x %08x ", r[8],r[9],r[10],r[11]);
	rprintf("%08x %08x %08x %08x ", r[12],r[13],r[14],r[15]);
}


cmd_int_poll()
{
	register unsigned level;

	if (int_flag != 0x00)	{	/* if command interrupt happened ? */
		int_flag--;	/* take one command off here..*/
		level = (ringer[int_tail].intcmd >> 24 ) & 0x07;
		printf("\nInterrupt received: CSS level= %02x\n",level);
		int_tail++;	/* process one command */
		if(int_tail == MAXRING) /* if it's at the limit.. */
			int_tail = 0;	/* reset to start of limit. */
	}
}

