
/*------------------------------------------------------------------------------
	boot.c: This program will handle the current simple boot.
------------------------------------------------------------------------------*/
#include "types.h"
#include "icb.h"
#include "iom.h"
#include "spm.h"
#include "globl.h"
#include "global.h"
#include "devcmd.h"
#include "vreg.h"
#include "icb_config.h"
#include "rtb.h"
#include "misc.h"
#include "disp.h"
#include "sa_dir.h"
#include "novram.h"
#include "filehdr.h"
#include "scnhdr.h"
#include "aouthdr.h"
#include "a.out.h"
#include "dev.h"
#include "iopm_comm.h"
#include "boot.h"
#include "iopm_err.h"
#include "sdk_disk.h"

#define	NO_ERROR	0

#define FD_INT_VECT	0x1000078
#define BUFPOINT	0x0103e000
#define	LINES_PER_SCREEN	20
#define	DISK_RETRY      	15
#define	TAPE_RETRY      	45
#define	IOPM_RETRY      	5
#define	NO_ENTRY_FOUND      	0xFF

struct fhdr {
	FILHDR filhdr;
	AOUTHDR saouthdr;
	SCNHDR sctnhdr[6];
};

static struct flp_boot {
	int	st_block;
	int	st_off;
	int	blocks;
	int	addr;
	} flp_blk[6];

extern uchar	*buff;		/* download buffer */
extern char	diskflg;	/* if disk reads,set this. */
extern char	booting;
extern char 	lastblk;	/* set this, if on last blk */

extern	uchar	css_slot;
extern	uchar	sub_slot;
extern	uchar	phys_dev;
extern	uchar	log_dev;
extern	char	filename[];
extern	uint	boot_type;
extern	struct	icb_config	*icb_ptr;

static	int	curr_blk;
static	int	lines_printed;

void	bootme ();
void	print_entry ();
int	print_entries ();
uint	find_entry ();
void	boot ();
void	boot_edt ();
void	boot_iopm ();

#ifdef	SPM_PROM
void	boot_floppy ();
uchar	*mov_flp();
#endif

extern void get_device_info ();
extern int xfer_iopm_dev ();

/*------------------------------------------------------------------------------
	bootme() : 	Boots from the specified boot path in the NVRAM.
			Or if an argument is given, the argument will be used
			as the Path.  If this is invoked with dir on the 
			command line, the directory is printed, and no boot.
------------------------------------------------------------------------------*/
void
bootme(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	char *bootstr;
	char save_path;
	register struct novram *ptr = (struct novram *)NOVRAM;

	if(arg_cnt) {				/* if we have an argument. */
		bootstr = comm_args[1];		/* ok, start here. */
		save_path = 1;
	}
	else {
		save_path = 0;
#ifdef	SPM_PROM
		if (ptr->diag_fly_by)
			bootstr = (char *)&ptr->spm_boot_disk[0]; /* spm boot */
		else
			bootstr = (char *)&ptr->diag_boot[0]; /* diag boot */
#endif

#ifdef	SPM_CONFIG
		bootstr = (char *)&ptr->spm_boot_disk[0]; /* spm boot */
#endif

	}

	if (*(comm_str) == 'd')	{
		lines_printed = 0;
		boot (bootstr, 1, (save_path = 0));    /* This should return....
					 		* when it does, return 
							* to the monitor
							*/
		return;
	}

	if (*(comm_str) == 'b') {
		boot (bootstr, 0, save_path);	/* This should not return.......
					 	 * if it does, return to 
						 * the monitor
						 */
		return;
	}

}

void
boot (bootstr, print_only, save_path)
char	*bootstr;
uint	print_only;
char	save_path;
{
	register struct novram *ptr = (struct novram *)NOVRAM;
	register uint cnt;

	if (bootstr[0] == 'f' && bootstr[1] == 'l' && 
				bootstr[2] == 'p' && bootstr[3] == '/') {
		boot_type = DV_FLP;
		bootstr +=4;		/* skip f-l-p-/ */
		strcpy (filename, bootstr); /* copy filename */
	}
	else {
		if (clokpr(1)) {
			return;
		}
		get_device_info (bootstr);
	}

	if ((boot_type == DV_NO_DEV) || (boot_type == DV_HELP_REQUEST)) {
		display_help_message ();
		return;
	}

	if (save_path) {		/* If booting and boot path provided */
		ptr->boot_dev = boot_type;
		if (strcmp (filename, "config") == 0) {	/* save config path */
			for (cnt = 0; cnt != PATH_LENGTH; cnt++)
				ptr->diag_boot[cnt] = (char)0;

			strcpy (ptr->diag_boot, bootstr);
			ptr->prom_crc = nov_chk_sum (0);
		}

		if (strcmp (filename, "spm") == 0) {	/* save spm path */
			for (cnt = 0; cnt != PATH_LENGTH; cnt++)
				ptr->spm_boot_disk[cnt] = (char)0;

			strcpy (ptr->spm_boot_disk, bootstr);
			ptr->prom_crc = nov_chk_sum (0);
		}
	}

	switch (boot_type) {
		case DV_9T:
		case DV_MT:
		case DV_RV:
			boot_edt(print_only);
			return;
		case DV_IOPM_MT:
		case DV_IOPM_9T:
		case DV_IOPM_DK:
		case DV_IOPM_RV:
			boot_iopm(print_only);
			return;
		case DV_FLP:
#ifdef	SPM_PROM
			boot_floppy(print_only);
			return;
#endif

#ifdef	SPM_CONFIG
			printf ("Floppy boot not supported by config image.\n");
			return;
#endif
		case DV_DK:
			printf ("no EDT slice boot.\n");
			return;
		default:
			printf ("Unrecognized boot_type %x\n", boot_type);
			return;
	}
}


void
boot_edt(print_only) /* edt (smd or scsi) boot */
uint print_only;
{
	struct		dtreq_call 	dtreq_struct;
	struct		dtreq_call 	*dtreq_ptr = &dtreq_struct;
	struct		sa_dir 		*dir= (struct sa_dir *)ICB_RD_ADDR; 
	uchar 		mem_slot;
	uint		i1 = 0;
	uint		entry;
	unsigned 	block, memaddr = 0;
	unsigned 	last_block_read, stat;

	curr_blk = -1;		/* show tape device position unknown */

/*******	get mem_slot - code passed to MM then loaded to SPM *******/
	if((mem_slot = (unsigned char)findmem(1)) == 18) {
		printf("Can't ICB boot without Memory Board.\n");
		return;
	}

/*******	set flags, and maps for using IOM and MM (ie CSS bus) *******/
	diskflg = booting = 1;
	sp_int_set();  /* set SPM to receive int itself */
	cssmap(MAP01,mem_slot,(char)0x00); /* setup mem slot*/
	cssmap(MAP00,css_slot,(char)0x00); /* setup io slot*/
	
/*******	set block to the directory block number    ********/
	if (boot_type == DV_RV)
		block = DIR_BLK_NUM;	/* set for hard disk boots. */
	else
		block = 0;		/* set for tape boots. */

/*******	Setup the dtreq_call structure		********/
	dtreq_ptr->icbp = icb_ptr;
	dtreq_ptr->mem_slot = mem_slot;
	dtreq_ptr->drive = phys_dev;
	dtreq_ptr->memaddr = memaddr;
	dtreq_ptr->block = block;
	dtreq_ptr->count = BLK_SIZE;

/*******	read in directory block set pointer check magic     *******/
	printf("\rRead directory block");
	do {
		if (ifesc())  /* to escape between directories */
			return;

		if (read_edt_dir(dtreq_ptr))
			return;

		if (boot_type == DV_9T)
			wait_cnt(5000);

		if(dir->magic_num != 0x5341) {		
			printf("Invalid magic number\n"); 
			return;
		}

/*******	dirme only prints........this is it      ********/
		if (print_only) {  /* display the directory; return */
			if (print_entries (dir))
				return;
		}
		else {
			entry = find_entry (dir, filename);
		}

		dtreq_ptr->block = block + dir->next_dir_block;

	} while (dir->next_dir_block && 
				(print_only || (entry == NO_ENTRY_FOUND)));

	if (print_only || (entry == NO_ENTRY_FOUND)) {
		if (boot_type != DV_RV) {
			printf("\nRewinding Tape....");
			dtreq_ptr->command = REWIND;
			dtreq_ptr->type = DTTAPE;
			dtreq(dtreq_ptr);
			printf("\n");
		}
		return;
	}

	block = block + dir->entry[entry].start_block_num; /* starting block */
	last_block_read = block + ((dir->entry[entry].byte_count - 1) / dir->block_size);

/*******	if a tape.......skip to first block with dummy reads *******/
	if ((boot_type == DV_MT) || (boot_type == DV_9T))
		seek_tape(dtreq_ptr, block);

/*******	read in the complete file, headers and all, to the MM  ******/
	printf("\r Read binary image           ");
	if (boot_type == DV_RV) {
		dtreq_ptr->command = PLREAD;
		dtreq_ptr->type = DTDISK;
		for(; block <= last_block_read; block++) {	/* read all blocks.. */
			if((i1++ % 10) == 0)
				printf("."); 	/* show me right now.. */
			
			dtreq_ptr->memaddr = memaddr;
			dtreq_ptr->block = block;
			if (stat=dtreq(dtreq_ptr)) {
				printf("\nDisk read failed, stat:  %4X\n",stat);
				return;
			}
			memaddr += BLK_SIZE;
		}
	}
	else {
		dtreq_ptr->command = TPREAD;
		dtreq_ptr->type = DTTAPE;
		for(; block <= last_block_read; block++) {	/* read all blocks.. */
			if((i1++ % 10) == 0) 
				printf("."); 	/* show me right now.. */
			
			if (last_block_read == block)
				lastblk = 1;
			dtreq_ptr->memaddr = memaddr;
			dtreq_ptr->block = block;
			if (stat=dtreq(dtreq_ptr)) {
				printf("\nTape read failed, stat:  %4X\n",stat);
				return;
			}
			curr_blk++;
			lastblk = 0;
			memaddr += BLK_SIZE;
		}
		block = 0;		/* rewind tape after tape read done */
		printf("\nRewinding Tape....");
		dtreq_ptr->command = REWIND;
		dtreq_ptr->type = DTTAPE;
		dtreq(dtreq_ptr);
	}

/*******	reset some flags, move code, and execute it  ********/
	diskflg = booting=0;	/* show not in bootup mode now. */

	if (boot_type == DV_9T)
		wait_cnt(20000);

#ifdef	SPM_PROM
	mov_code();	/* move the code(tape/disk), and jump to it */
#endif

#ifdef	SPM_CONFIG
	vbrmv(0);	/* point vbr to the PROM  */
	trap1();	/* trap prom to copy code */
#endif

	return;		/* should never get here...... */
}

void
boot_iopm(print_only) /* boot from iopm. */
uint print_only;
{
	struct		sa_dir 		*dir = (struct sa_dir *)IOPM_RD_ADDR; 
	uint  		entry;
	unsigned  int	block;
	unsigned  int	block_sav;
	unsigned  int 	blk_cnt = 1;		/* directory size */
	unsigned  int	stat;

/*	
	Currently the iopm boot looks a lot like the edt boot, in the 
	future it will be made to look like the floppy boot.
*/

/*******	set flags, and maps for using IOM and MM (ie CSS bus) *******/
	diskflg = booting = 1;
	if (sub_slot == NO_SUB_SLOT)
		cssmap(MAP02,css_slot,(char)0x0F); /* setup io slot*/
	else
		cssmap(MAP02,css_slot, sub_slot); /* setup io slot*/

/*******	setup block to be directory block     *********/
	switch (boot_type) {
		case DV_IOPM_DK:	/* 32 slice IOPM disk */
			if ((block = get_iopm_blk_num(LD_BOOT_SLICE)) == 
							(uint)0xFFFFFFFF)
				return;
			break;
		case DV_IOPM_RV:	/* 16 slice 'A1000' disk */
			block = DIR_BLK_NUM;
			break;
		default:		/* Tape device */
			block = 0;
			break;
	}
	block_sav = block;

/*******	read directory....and setup pointer   ********/
	printf ("\rRead directory block");
	do {

		stat = xfer_iopm_dev (block, blk_cnt, 1);

		if (stat) {
			show_iopm_error(stat);
			return;
		}
	
/*******	Check directory, and scan for filename  *******/
		if(dir->magic_num != 0x5341) {		
			printf("Invalid magic number\n"); 
			printf("Was the proper device type selected?\n");
			check_iopm_dev_type (boot_type);
			return;
		}
	
/*******	dirme only prints the directory........so here it is *******/
		if (print_only) {  /* display the directory */
			if (print_entries (dir))
				return;
		}
		else {
			entry = find_entry (dir, filename);
		}

		block = block_sav + dir->next_dir_block;

	} while (dir->next_dir_block && (print_only || (entry == NO_ENTRY_FOUND)));

	if (print_only || (entry == NO_ENTRY_FOUND)) {
		return;
	}

/*******	offset relative to directory block...so set it up right ******/
	block = block_sav + dir->entry[entry].start_block_num; /* start block */
	blk_cnt = ((dir->entry[entry].byte_count + BLK_SIZE - 1) / BLK_SIZE);

/*******	read the file to the MM.....headers and all  ********/
	printf ("\rRead binary image       ");
	stat = xfer_iopm_dev (block, blk_cnt, 1);
	
	if (stat) {
		show_iopm_error(stat);
		return;
	}

/*******	setup some flags and move code then execute it  *******/
	diskflg = booting=0;	/* show not in bootup mode now. */

#ifdef	SPM_PROM
	mov_iopm_code();	/* move the code(tape/disk), and jump to it */
#endif

#ifdef	SPM_CONFIG
	vbrmv(0);	/* point vbr to the PROM  */
	trap2();	/* trap prom to copy code */
#endif

	return;			/* should never get here */
}

#ifdef	SPM_PROM
void
boot_floppy (print_only) 	/* boot floppy */
uint print_only;
{
	struct		sa_dir 		*dir = (struct sa_dir *)BUFPOINT;
	uint		entry;
	int		num_sect;
	int		scn;
	int		blk;
	uchar		*rdbuff;
	unsigned 	block = 0;		/* directory starts at 0 */
	unsigned 	stat;
	struct fhdr 	*fhptr = (struct fhdr *)BUFPOINT;
	struct scnhdr 	*scn_ptr;
	unsigned 	final, *startp, save, save1;

/*******	initialize floppy device **********/

	if (fcmdinit ())
		return;

/*******	read the sa directory    **********/
	printf ("\rRead directory block");
	do {
		if (stat = rd_flp_dev(block, buff)) { /* Read block in buffer */
			printf("Floppy directory read failed, stat:  %x\n", stat);
			moff();
			return;
		}

/*******	check directory, find file ********/
		if(dir->magic_num != 0x5341) {		
			printf("Invalid magic number in directory\n"); /* tell user. */
			moff();
			return;
		}

/*******	dirme only wants to print the directory, do it return ******/
		if (print_only) {  /* Used by dirme to display the directory; returns 0*/
			if (print_entries (dir))
				return;
		}
		else {
			entry = find_entry (dir, filename);
		}

		block = dir->next_dir_block;

	} while (dir->next_dir_block && (print_only || (entry == NO_ENTRY_FOUND)));

	if (print_only || (entry == NO_ENTRY_FOUND)) {
		recal();
		moff();
		return;
	}

	block = dir->entry[entry].start_block_num; /* starting block */

/*******	read in first block of file -- headers  **********/
	printf ("\rRead binary image    ");
	if(stat = rd_flp_dev (block, buff)) { 	/* read a block */
		printf("Floppy 1st block read failed, stat:  %x\n", stat);
		moff();
		return;
	}

	scn_ptr = fhptr->sctnhdr;
	startp = (unsigned *)(scn_ptr->s_paddr + 0x4);
	num_sect = set_flp_ptr(block, buff);

	for (scn = 0; scn != num_sect; scn++) {

		rdbuff = (uchar *)BUFPOINT;
		block = flp_blk[scn].st_block;
		if(stat = rd_flp_dev (block, rdbuff)) {	/* read a block */
			printf("Floppy block read failed, stat:  %x\n", stat);
			moff();
			return;
		}

		block++;		

		if (!scn) /* assume only section 0 (text) overlaps vectors */
			save = *(unsigned *)FD_INT_VECT;    /* PROM vector */

		rdbuff = mov_flp (scn);

		if (!scn) {
			save1 = *(unsigned *)FD_INT_VECT;   /* Image vector */
			*(unsigned *)FD_INT_VECT = save;    /* replace vector */
		}
		printf ("+");

		for (blk = 0; blk != flp_blk[scn].blocks;) {
			if(stat = rd_flp_dev (block, rdbuff)) {	/* read block */
				printf("Floppy block read failed, stat:  %x\n",
					stat);
				moff();
				return;
			}

			if (!(++blk % 10))
				printf (".");

			rdbuff += BLK_SIZE;
			block++;
		}
	}

/*******	setup fd interrrupt, turn off motor, and execute  ********/
	recal();
	moff();			/* turn motor off, just for fun. */

	*(unsigned *)FD_INT_VECT = save1;
	final = *(unsigned *)startp;
	(*(PFI)final)(); 	/* Jump to it!!! */
}
#endif

#ifdef	SPM_PROM
int
set_flp_ptr(block, fhdr)
uint	block;
struct	fhdr *fhdr;
{

	struct scnhdr 	*scn_ptr;
	int	cnt;
	int	scn;
	int	size;

	scn_ptr = fhdr->sctnhdr;
	for (cnt = 0, scn = 0; cnt != fhdr->filhdr.f_nscns; cnt++, scn_ptr++) {

		if (!scn_ptr->s_scnptr)
			continue;

		flp_blk[scn].addr = scn_ptr->s_paddr;
		flp_blk[scn].st_block = block + 
					(scn_ptr->s_scnptr / BLK_SIZE);
		flp_blk[scn].st_off = (scn_ptr->s_scnptr % BLK_SIZE);
		size = (scn_ptr->s_size - BLK_SIZE + flp_blk[scn].st_off);
		flp_blk[scn].blocks = ((size + BLK_SIZE - 1) / BLK_SIZE);
		scn++;
	}
	return (scn);
}
#endif

#ifdef	SPM_PROM
uchar *
mov_flp (scn)
uint	scn;
{

	unsigned  *dest = (unsigned *)flp_blk[scn].addr;
	unsigned  *src  = (unsigned *)(BUFPOINT + flp_blk[scn].st_off);
	unsigned  *limit = (unsigned *)(BUFPOINT + BLK_SIZE);

	for (; src < limit; src++, dest++) 
		*dest = *src;
	return((uchar *)dest);
}
#endif


print_dir_hdr()
{
	printf ("\n FILENAME              VERSION   START_BLK     SIZE        SPM_EXEC\n\n");
	
	return;
}


void
print_entry (dir, element)
struct	sa_dir 	*dir;	/* stand alone dir pointer. */
uint	element;
{
	
	printf (" %16s       %2d.%2d     %6d       %6d       %s\n", 
	dir->entry[element].name,
	dir->entry[element].major_version, dir->entry[element].minor_version,
	dir->entry[element].start_block_num, dir->entry[element].byte_count,
	(dir->entry[element].spm_exec) ? " SPM_EXEC  " : "NO_SPM_EXEC");
}

uint
find_entry (dir, file) /* see if file is in dir */
struct	sa_dir   *dir;
char	*file;
{
	int entry;

	for(entry = 0; entry < MAX_DIR_ENTRY; entry++) { 
			if(!strcmp(file, dir->entry[entry].name)) /* same */
				break;
	}
	
	if(entry == MAX_DIR_ENTRY) { /* if failed */
		if (! dir->next_dir_block) /* no more directories */
			printf("Did not find (%s) code. \n", file);
		return(NO_ENTRY_FOUND);
	}

	if(!dir->entry[entry].spm_exec) { /* if not allowed to exec. */
		printf("\nFile (%s) is not executable by SPM.\n", file);
		return(NO_ENTRY_FOUND); 		/* exit */
	}
	
	return (entry);
}


int
print_entries (dir)
struct  sa_dir  *dir;
{
	uchar  entry;

	if (!lines_printed)
		print_dir_hdr();

	for(entry = 0; entry < MAX_DIR_ENTRY; entry++) {    /* all entries */

		if (dir->entry[entry].name[0] == '\0')
			break;

		print_entry (dir, entry);
		lines_printed++;
		if (lines_printed == LINES_PER_SCREEN) {
			printf ("Anykey to continue, ESC to quit...");
			if (getchar () == ESCAPE) {
				printf ("\r                                  \r");
				return(1);
			}	
			lines_printed = 0;
			printf ("\r                                  \r");
			print_dir_hdr();
		}
	}
	return(0);
}

int
read_edt_dir (dtreq_ptr)
struct		dtreq_call 	*dtreq_ptr; 
{

	int	i  = 0;
	int	stat = 0;

	if (boot_type != DV_RV) {
		seek_tape(dtreq_ptr, dtreq_ptr->block);

		dtreq_ptr->command = TPREAD;
		if(stat=dtreq(dtreq_ptr)) {
			printf("Tape directory read failed, stat:  %4X\n",stat);
			return(1);
		}
		curr_blk++;
	}
	else {
		dtreq_ptr->command = PLREAD;
		dtreq_ptr->type = DTDISK;
		while(i++ != DISK_RETRY)	{ /* retries 45 times.. */
	   		if(stat=dtreq(dtreq_ptr)) {
				wait_cnt(180);
				if(ifesc())
					return(1);  /* exit */
			}
			else
				break;
		}
		if(i == DISK_RETRY) {
			printf("Disk read failed, stat:  %4X\n", stat);
			return(1);
		}
	}
	return(0);
}

int
seek_tape (dtreq_ptr, blk_num)
struct		dtreq_call 	*dtreq_ptr; 
uint	blk_num;
{
	int	stat = 0;
	
	dtreq_ptr->type = DTTAPE;

	if (curr_blk == -1)
		rewind_tape (dtreq_ptr);

	if (curr_blk > blk_num)
		rewind_tape (dtreq_ptr);

	dtreq_ptr->command = TPREAD;
	for (; curr_blk != blk_num; curr_blk++) {
		if(stat=dtreq(dtreq_ptr)) {
			printf("\nTape seek failed, stat:  %4X\n",stat);
			dtreq_ptr->command = REWIND;
		  	dtreq(dtreq_ptr);
			return(1);
		}
	}
	return(0);
}

int
rewind_tape(dtreq_ptr)
struct		dtreq_call 	*dtreq_ptr; 
{
	int	i = 0;
	int	stat = 0;

	dtreq_ptr->command = REWIND;
	dtreq_ptr->type = DTTAPE;
	while(i++ != TAPE_RETRY)	{/* retries 15 times.. */
		if(stat=dtreq(dtreq_ptr)) {
			wait_cnt(180);
			if(ifesc())
				curr_blk = -1;
				return(1);	
		}
		else
			break;
	}

	if(i == TAPE_RETRY) {
		printf("Tape rewind failed, stat:  %4X\n", stat);
		curr_blk = -1;
		return(1);
	}
	curr_blk = 0;
	return(0);
}

